/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 * Allocate a file descriptor
 */

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include "fd.h"

/* start out with NDFILE */
#define NDFILE          20 
#define NDEXTENT        50

/* then extend by `extendby' - which is doubled every time */
static	int extendby = NDFILE;

#ifdef THREAD_SAFE
static pthread_mutexattr_t	mutex_attributes;
pthread_mutex_t			libc_fd_array_mutex;
#endif

static  int didinit  = 0;
void	fd_init(void);

/*
 * The 3 stdio descriptors have to be statically initialized so that
 * console printf works from the get go. This makes the first real
 * allocation a little messy, since we are going to toss this array
 * away once we get the intialization call. Life is messy.
 */
extern oskit_stream_t console_stream;

int	fd_arraylen = 3;
fd_t    initial_fd_array[3] = {
	{ (oskit_iunknown_t*) &console_stream },
	{ (oskit_iunknown_t*) &console_stream },
	{ (oskit_iunknown_t*) &console_stream },
};
fd_t	*fd_array = initial_fd_array;

int	
fd_alloc(oskit_iunknown_t *obj, int min_fd)
{
	int		i;
	oskit_size_t 	bytes;
	fd_t		*newarray;

	if (! didinit) {
#ifdef  THREAD_SAFE
		panic("fd_alloc: oskit_init_libc was not called!");
#else
		fd_init();
#endif
	}
	fd_array_lock();

try_again:
	/* search for a free fd */
	for (i = min_fd; i < fd_arraylen; i++) {
		if (fd_array[i].obj == 0) {
			/* found an unused slot */
#ifdef THREAD_SAFE
			pthread_mutex_init(fd_array[i].lock,
					   &mutex_attributes);
			pthread_cond_init(fd_array[i].cond,
					  &pthread_condattr_default);
			fd_array[i].writer = 0;
			fd_array[i].reader = 0;
			fd_array[i].writecount = 0;
			fd_array[i].readcount = 0;
#endif
			oskit_iunknown_addref(obj);
			fd_array[i].obj = obj;
			fd_array_unlock();
			return i;
		}
	}

	/*
	 * Create the fd_array if we don't have one yet, or realloc
	 * if we do.
	 */
	if (fd_arraylen == 0) {
		extendby = NDFILE;
		bytes = extendby * sizeof(fd_t);
		newarray = (fd_t *)malloc(bytes);
        }
	else {
		bytes = (fd_arraylen + extendby) * sizeof(fd_t);
		newarray = (fd_t *)realloc(fd_array, bytes);
	}
	if (newarray == NULL) {
		errno = ENOMEM;
		fd_array_unlock();
		return -1;
	}
	/* zero out new part of the array */
	memset(newarray + fd_arraylen, 0, extendby * sizeof(fd_t));
	
#ifdef  THREAD_SAFE
	/* allocate the mutex and condition variable structures. */
	{
		pthread_mutex_t	*pmutex;
		pthread_cond_t  *pcond;
		fd_t		*pfd = newarray + fd_arraylen;
		
		if ((pmutex = (pthread_mutex_t *)
		     malloc(sizeof(pthread_mutex_t) * extendby)) == NULL) {
			errno = ENOMEM;
			fd_array_unlock();
			return -1;
		}
		if ((pcond = (pthread_cond_t *)
		     malloc(sizeof(pthread_cond_t) * extendby)) == NULL) {
			free(pmutex);
			errno = ENOMEM;
			fd_array_unlock();
			return -1;
		}
		for (i = 0; i < extendby; i++) {
			pfd[i].lock = pmutex++;
			pfd[i].cond = pcond++;
		}
	}
#endif
	fd_array = newarray;
	fd_arraylen += extendby;
	extendby *= 2;
	goto try_again;
}

void
fd_init(void)
{
	if (didinit)
		return;
	
#ifdef THREAD_SAFE
	pthread_mutex_init(&libc_fd_array_mutex, &pthread_mutexattr_default);

	/*
	 * Because interrupt routines can call printf, the mutex needs
	 * to allow recursive lock attempts.
	 */
	pthread_mutexattr_init(&mutex_attributes);
	pthread_mutexattr_settype(&mutex_attributes, PTHREAD_MUTEX_RECURSIVE);
#endif
	/*
	 * Now create the actual fd_array by requesting descriptors for
	 * the 3 stdio stream. Zero out the initial version of it so
	 * that fd_alloc creates a new one from scratch. 
	 */
	fd_arraylen = 0;
	fd_array    = 0;
	didinit     = 1;

	fd_alloc((oskit_iunknown_t*)&console_stream, 0);
	fd_alloc((oskit_iunknown_t*)&console_stream, 1);
	fd_alloc((oskit_iunknown_t*)&console_stream, 2);
}
