/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * Switch to a real tty driver for the console. If we find a tty driver
 * to use, replace the console_stream used for stdin/stdout/stderr with
 * it. The application should not call this unless it has initialized
 * a console like device.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <oskit/io/ttystream.h>
#include <oskit/fs/file.h>	/* XXX OSKIT_O_RDWR */
#include <oskit/dev/dev.h>
#include <oskit/dev/tty.h>
#include <oskit/io/asyncio.h>
#include <oskit/io/posixio.h>
#include <oskit/com/wrapper.h>
#include "fd.h"

/*
 * Default settings for the console TTY.
 */
#define	TTYDEF_IFLAG \
	(OSKIT_BRKINT|OSKIT_ICRNL|OSKIT_IXON|OSKIT_IXANY)
#define TTYDEF_OFLAG \
	(OSKIT_OPOST|OSKIT_ONLCR)
#define TTYDEF_LFLAG \
	(OSKIT_ECHO|\
	 OSKIT_ICANON|OSKIT_ISIG|OSKIT_IEXTEN|OSKIT_ECHOE|OSKIT_ECHOK)
#define	TTYDEF_CFLAG \
	(OSKIT_CREAD|OSKIT_CS8|OSKIT_HUPCL)

void
oskit_console_init(void)
{
	oskit_ttydev_t		**ttydev;
	oskit_ttystream_t	*ttystrm;
	oskit_termios_t		termios;
	int			rc, ndev, i, enabled;

#ifdef  THREAD_SAFE
	osenv_process_lock();
#endif
	/*
	 * Find all TTY device nodes.
	 */
	ndev = osenv_device_lookup(&oskit_ttydev_iid, (void***)&ttydev);
	if (ndev <= 0)
		panic("no TTY devices found!");

	/*
	 * Open the first TTY device found. I am going to say thats
	 * the console device the application wanted to use.
	 */
	rc = oskit_ttydev_open(ttydev[0], OSKIT_O_RDWR, &ttystrm);
	if (rc)
		panic("unable to open tty0: %d", rc);

	/*
	 * Okay, release our references.
	 */
	for (i = 0; i < ndev; i++)
		oskit_ttydev_release(ttydev[i]);

	/*
	 * Set properties for a normal tty.
	 */
	rc = oskit_ttystream_getattr(ttystrm, &termios);
	if (rc)
		panic("unable to getattr: %d", rc);
	
	termios.iflag = TTYDEF_IFLAG;
	termios.oflag = TTYDEF_OFLAG;
	termios.cflag = TTYDEF_CFLAG;
	termios.lflag = TTYDEF_LFLAG;
	termios.cc[3] = 8; /* ^H */
	
	rc = oskit_ttystream_setattr(ttystrm, OSKIT_TCSANOW, &termios);
	if (rc)
		panic("unable to setattr: %d", rc);

#ifdef  THREAD_SAFE
	/*
	 * Need to wrap the stream in a thread-safe wrapper.
	 */
	{
		oskit_ttystream_t	*wrapped_ttystrm;
		
		rc = oskit_wrap_ttystream(ttystrm,
			  (void (*)(void *))osenv_process_lock, 
			  (void (*)(void *))osenv_process_unlock,
			  0, &wrapped_ttystrm);

		if (rc)
			panic("oskit_wrap_ttystream() failed: "
			      "errno 0x%x\n", rc);

		/* Don't need the original anymore, the wrapper has a ref. */
		oskit_ttystream_release(ttystrm);

		ttystrm = wrapped_ttystrm;
		osenv_process_unlock();
	}
#endif
	/*
	 * Okay, now replace the console_stream for the stdio descriptors.
	 */
	if ((enabled = osenv_intr_enabled()))
		osenv_intr_disable();
		
	fd_init();
	fd_free(0);
	fd_free(1);
	fd_free(2);
	fd_alloc((oskit_iunknown_t*)ttystrm, 0);
	fd_alloc((oskit_iunknown_t*)ttystrm, 1);
	fd_alloc((oskit_iunknown_t*)ttystrm, 2);

	if (enabled)
		osenv_intr_enable();

	/*
	 * Release our reference since fd_alloc took one for each.
	 */
	oskit_ttystream_release(ttystrm);
}
