/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 * This file implements some of the methods for "console" fd's.
 * All this is is a big hack so that 'write(1, ' actually shows up
 * on the console.
 */

#include <stdio.h>
#include <oskit/io/asyncio.h>
#include <oskit/io/posixio.h>
#include <oskit/x86/pc/base_console.h>
#include "fd.h"

static struct oskit_asyncio_ops asyncio_ops;
static struct oskit_stream_ops  stream_ops;
static struct oskit_posixio_ops  posixio_ops;

struct oskit_stream	    console_stream  = { &stream_ops  };
static struct oskit_asyncio console_asyncio = { &asyncio_ops };
static struct oskit_posixio console_posixio = { &posixio_ops };

static OSKIT_COMDECL
query(oskit_stream_t *si, const struct oskit_guid *iid, void **out_ihandle)
{
	if (memcmp(iid, &oskit_iunknown_iid, sizeof(*iid)) == 0 ||
	    memcmp(iid, &oskit_stream_iid, sizeof(*iid)) == 0) {
		*out_ihandle = &console_stream;
		return 0;
	}

	if (memcmp(iid, &oskit_posixio_iid, sizeof(*iid)) == 0) {
		*out_ihandle = &console_posixio;
		return 0;
	}

	if (memcmp(iid, &oskit_asyncio_iid, sizeof(*iid)) == 0) {
		*out_ihandle = &console_asyncio;
		return 0;
	}

	*out_ihandle = NULL;
	return OSKIT_E_NOINTERFACE;
}

static OSKIT_COMDECL_U
addref(oskit_stream_t *si)
{
	return 1;
}

static OSKIT_COMDECL_U
release(oskit_stream_t *si)
{
	return 1;
}

static OSKIT_COMDECL
read(oskit_stream_t *si, void *buf, oskit_u32_t len, oskit_u32_t *out_actual)
{
	int	i = 0, c;
	char	*str = (char *) buf;

	if (len == 0)
		return 0;

	/* implement simple line discipline */
	while (i < len) {
		c = console_getchar();
		if (c == EOF) {
			if (i == 0)
				return NULL;
			break;
		}
		if (c == '\r')
			c = '\n';
		else if (c == '\b') {
			if (i > 0) {
				console_putchar(c);
				console_putchar(' ');
				console_putchar(c);
				i--;
			}
			continue;
		}
		else if (c == '\025') {		/* ^U -- kill line */
			while (i) {
				console_putchar('\b');
				console_putchar(' ');
				console_putchar('\b');
				i--;
			}
			str[0] = '\0';
			continue;
		}
		console_putchar(c);
		str[i++] = c;
		if (c == '\n')
			break;
	}
	if (i < len)
		str[i] = '\0';

	*out_actual = i;
	return 0;
}

static OSKIT_COMDECL
write(oskit_stream_t *si, const void *buf, oskit_u32_t nb,
      oskit_u32_t *out_actual)
{
	extern int console_putbytes(const char *, int length);

	console_putbytes((const char *) buf, (int) nb);

	*out_actual = nb;
	return 0;
}

static OSKIT_COMDECL
seek(oskit_stream_t *si, oskit_s64_t ofs, oskit_seek_t whence,
		  oskit_u64_t *out_newpos)
{
	return OSKIT_ESPIPE;
}

static OSKIT_COMDECL
setsize(oskit_stream_t *si, oskit_u64_t new_size)
{
	return OSKIT_EINVAL;
}

static OSKIT_COMDECL
copyto(oskit_stream_t *s, oskit_stream_t *dst,
       oskit_u64_t size,
       oskit_u64_t *out_read,
       oskit_u64_t *out_written)
{
	return OSKIT_E_NOTIMPL;
}

static OSKIT_COMDECL
commit(oskit_stream_t *si, oskit_u32_t commit_flags)
{
	return OSKIT_E_NOTIMPL;
}

static OSKIT_COMDECL
revert(oskit_stream_t *si)
{
	return OSKIT_E_NOTIMPL;
}

static OSKIT_COMDECL
lockregion(oskit_stream_t *si, oskit_u64_t offset,
			 oskit_u64_t size, oskit_u32_t lock_type)
{
	return OSKIT_E_NOTIMPL;
}

static OSKIT_COMDECL
unlockregion(oskit_stream_t *si, oskit_u64_t offset,
			   oskit_u64_t size, oskit_u32_t lock_type)
{
	return OSKIT_E_NOTIMPL;
}

static OSKIT_COMDECL
stat(oskit_stream_t *si, oskit_stream_stat_t *out_stat,
		  oskit_u32_t stat_flags)
{
	return OSKIT_E_NOTIMPL;
}

static OSKIT_COMDECL
clone(oskit_stream_t *si, oskit_stream_t **out_stream)
{
	return OSKIT_E_NOTIMPL;
}

static struct oskit_stream_ops stream_ops = {
	query, addref, release,
	read, write, seek, setsize, copyto,
	commit, revert, lockregion, unlockregion, stat, clone
};

static OSKIT_COMDECL
posix_stat(oskit_posixio_t *pio, struct oskit_stat *st)
{
	memset(st, 0, sizeof *st);
	st->ino = 1;
        st->mode = OSKIT_S_IFCHR | OSKIT_S_IRWXG | OSKIT_S_IRWXU | OSKIT_S_IRWXO;
	return 0;
}

static OSKIT_COMDECL
setstat(oskit_posixio_t *pio, oskit_u32_t mask, const struct oskit_stat *stats)
{
	return OSKIT_E_NOTIMPL;
}

static OSKIT_COMDECL
pathconf(oskit_posixio_t *pio, oskit_s32_t option, oskit_s32_t *out_val)
{
	return OSKIT_E_NOTIMPL;
}

static struct oskit_posixio_ops posixio_ops = {
	query, addref, release,
	posix_stat, setstat, pathconf
};

/*
 * this should trigger applications who select on a console fd to
 * go into read->getchar or write->putchar immediately.
 */
static OSKIT_COMDECL
asyncio_poll(oskit_asyncio_t *f)
{
	return OSKIT_ASYNCIO_WRITABLE | OSKIT_ASYNCIO_READABLE;
}

/* add a listener */
static OSKIT_COMDECL
asyncio_add_listener(oskit_asyncio_t *f, struct oskit_listener *l,
	oskit_s32_t mask)
{
	return OSKIT_E_NOTIMPL;
}

/* remove a listener */
static OSKIT_COMDECL
asyncio_remove_listener(oskit_asyncio_t *f, struct oskit_listener *l0)
{
	return OSKIT_E_NOTIMPL;
}

/* how many bytes can be read */
static OSKIT_COMDECL
asyncio_readable(oskit_asyncio_t *f)
{
	return 0;
}

static struct oskit_asyncio_ops asyncio_ops = {
	query, addref, release,	/* will generate three leg. warnings */
	asyncio_poll,
	asyncio_add_listener,
	asyncio_remove_listener,
	asyncio_readable
};

/*
 * Stubbed. No longer needed since the initial 3 descriptors are statically
 * initialized.
 */
oskit_error_t
fd_set_console(void)
{
	return 0;
}
