/*
 * Copyright (c) 1996, 1998, 1999 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * External definitions for threads package.
 */
#ifndef	_PTHREAD_H_
#define _PTHREAD_H_

#include <oskit/types.h>
#include <oskit/time.h>
#include <oskit/queue.h>
#include <oskit/machine/spin_lock.h>

/*
 * A thread ID is really just a small integer.
 */
typedef int	pthread_t;

/*
 * Thread priorities.
 */
enum {
	PRIORITY_MIN	=	0,
	PRIORITY_NORMAL =	16,
	PRIORITY_MAX	=	31,
};

/*
 * Scheduler types
 */
enum {
	SCHED_FIFO      =	0,
	SCHED_RR        =	1,
	SCHED_DECAY     =	2,
};

typedef struct sched_param {
	int		priority;
} sched_param_t;

/*
 * Thread creation inheritance state.
 */
enum {
	PTHREAD_INHERIT_SCHED   = 1,
	PTHREAD_EXPLICIT_SCHED  = 2,
};
        
/*
 * Detach state.
 */
enum {
        PTHREAD_CREATE_JOINABLE = 1,
        PTHREAD_CREATE_DETACHED = 2,
};
        
/*
 * Thread attributes object.
 */
struct pthread_attr_t {
        int		detachstate;
	int		priority;
	int		stacksize;
	oskit_u32_t	stackaddr;
	int		guardsize;
	int		policy;
	int		inherit;
	/*
	 * CPU Inheritance attributes. The opaque data is used to pass
	 * info to the scheduler, like tickets or reservation. It will
	 * do for now.
	 */
	pthread_t	scheduler;
	oskit_u32_t	opaque;
};

/*
 * Mutex object.
 */
struct pthread_mutex_t {
	spin_lock_t	mlock;		/* Actual mutex lock */
	spin_lock_t	dlock;		/* Lock for this data structure */
	queue_head_t	waiters;	/* queue for mutex */
	void	       *holder;		/* Opaque to user */
	int		count;		/* For recursive mutexes */
	int		inherit;	/* Does priority inheritance */
	int		type;		/* Mutex type. See below */
};

/*
 * Mutex attributes object.
 */
struct pthread_mutexattr_t {
        int		priority_inherit;
	int		type;
};

/*
 * Mutex protocol attributes.
 */
enum {
	PTHREAD_PRIO_NONE	= 0,
	PTHREAD_PRIO_INHERIT,
};

/*
 * Mutex type attributes.
 */
enum {
	PTHREAD_MUTEX_NORMAL	= 0,
	PTHREAD_MUTEX_ERRORCHECK,
	PTHREAD_MUTEX_RECURSIVE,
	PTHREAD_MUTEX_DEFAULT,
};

/*
 * Condition variable object.
 */
struct pthread_cond_t {
	spin_lock_t	lock;		/* Lock for this data structure */
        queue_head_t	waiters;        /* queue for cond var */
};

/*
 * Condition variable attributes object.
 */
struct pthread_condattr_t {
        int attr;
};

typedef struct pthread_attr_t pthread_attr_t;
typedef struct pthread_mutex_t pthread_mutex_t;
typedef struct pthread_mutexattr_t pthread_mutexattr_t;
typedef struct pthread_cond_t pthread_cond_t;
typedef struct pthread_condattr_t pthread_condattr_t;

/*
 * Default attributes.
 */
extern pthread_attr_t		pthread_attr_default;
extern pthread_mutexattr_t	pthread_mutexattr_default;
extern pthread_condattr_t	pthread_condattr_default;

/*
 * Keys are just small ints (for now). Not the right way to do this.
 */
typedef int			pthread_key_t;
#define PTHREAD_KEYS_MAX	128

/*
 * Minimum stack size.
 */
#define PTHREAD_STACK_MIN	(5 * 0x1000)

/*
 * Cancel state stuff.
 * According to ANSI/IEEE Std 1003.1, these must be macros.
 */
#define	PTHREAD_CANCEL_ENABLE		0
#define	PTHREAD_CANCEL_DISABLE 		1
#define	PTHREAD_CANCEL_DEFERRED		2
#define	PTHREAD_CANCEL_ASYNCHRONOUS	3

#define PTHREAD_CANCELED	OSKIT_ECANCELED

/*
 * Prototypes.
 */
void		pthread_init(int preemptible);
void		pthread_init_withhooks(int preemptible,
			void *(*allocator)(oskit_size_t),
			void (*deallocator)(void*));
int		pthread_create(pthread_t *tid, const pthread_attr_t *pattr,
			void (*function)(void *),
			void *argument);
pthread_t	pthread_self(void);
void		sched_yield(void);
void		pthread_exit(void *status);
int		pthread_detach(pthread_t tid);
int		pthread_join(pthread_t tid, void **status);
int             pthread_mutex_init(pthread_mutex_t *m,
			const pthread_mutexattr_t *attr);
int		pthread_mutex_lock(pthread_mutex_t *m);
int		pthread_mutex_trylock(pthread_mutex_t *m);
int		pthread_mutex_unlock(pthread_mutex_t *m);
int		pthread_mutex_destroy(pthread_mutex_t *m);
int		pthread_cond_init(pthread_cond_t *c,
			const pthread_condattr_t *attr);
int		pthread_cond_wait(pthread_cond_t *c, pthread_mutex_t *m);
int		pthread_cond_timedwait(pthread_cond_t *c, pthread_mutex_t *m,
			oskit_timespec_t *abstime);
int		pthread_cond_destroy(pthread_cond_t *c);
int		pthread_cond_signal(pthread_cond_t *c);
int		pthread_cond_broadcast(pthread_cond_t *c);
int		pthread_cancel(pthread_t tid);
int		pthread_setprio(pthread_t tid, int prio);
int		pthread_suspend(void);
int		pthread_resume(pthread_t tid);
int		pthread_sleep(oskit_s64_t millis);
int		pthread_wakeup(pthread_t tid);
int		pthread_attr_init(pthread_attr_t *pattr);
int		pthread_attr_setprio(pthread_attr_t *pattr, int pri);
int		pthread_attr_setstacksize(pthread_attr_t *pattr,
			oskit_size_t threadStackSize);
int		pthread_attr_setstackaddr(pthread_attr_t *attr,
			oskit_u32_t stackaddr);
int		pthread_attr_setguardsize(pthread_attr_t *attr,
			oskit_size_t guardsize);
int		pthread_mutexattr_init(pthread_mutexattr_t *attr);
int		pthread_condattr_init(pthread_condattr_t *attr);
int		pthread_whichcpu(void);
int		pthread_mutexattr_setprotocol(pthread_mutexattr_t *attr,
			int inherit);
int		pthread_mutexattr_settype(pthread_mutexattr_t *attr, int type);
int		pthread_setschedparam(pthread_t tid, int policy,
			const struct sched_param *param);
int		pthread_key_create(pthread_key_t *key,
			void (*destructor)(void *));
int		pthread_key_delete(pthread_key_t key);
int		pthread_setspecific(pthread_key_t key, const void *value);
void	       *pthread_getspecific(pthread_key_t key);
int		pthread_attr_setdetachstate(pthread_attr_t *attr, int state);
int		pthread_attr_setschedpolicy(pthread_attr_t *attr, int policy);
int		pthread_attr_setschedparam(pthread_attr_t *attr,
			const struct sched_param *param);
int		pthread_attr_setopaque(pthread_attr_t *attr, oskit_u32_t op);
int		pthread_attr_setscheduler(pthread_attr_t *attr, pthread_t tid);
void		pthread_cleanup_pop(int execute);
void		pthread_cleanup_push(void (*routine)(void *), void *arg);
void		pthread_testcancel(void);
int		pthread_setcancelstate(int state, int *oldstate);
int		pthread_setcanceltype(int type, int *oldtype);
void		osenv_process_lock(void);
void		osenv_process_unlock(void);
oskit_u32_t	pthread_cputime(pthread_t tid);
oskit_u32_t	pthread_realtime(void);
oskit_u32_t	pthread_childtime(pthread_t tid);

struct oskit_socket_factory;
void		pthread_init_socketfactory(struct oskit_socket_factory *sf);

/*
 * This is a local hack to get the current thread state.
 */
typedef struct pthread_state {
	oskit_size_t		stacksize;
	oskit_u32_t	stackbase;
	oskit_u32_t	stackptr;
	oskit_u32_t	frameptr;
} pthread_state_t;
int		pthread_getstate(pthread_t tid, pthread_state_t *pstate);

/*
 * Timer. The resolution is 10ms.
 */
#define PTHREAD_HZ	100
#define PTHREAD_TICK	(1000/PTHREAD_HZ)
#endif /* _PTHREAD_H_ */
