/*
 * Copyright (c) 1998 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 * Declarations for -loskit_startup functions, a random collection
 * of functions to start things up in vaguely canonical ways.
 */

#ifndef _OSKIT_STARTUP_H_
#define _OSKIT_STARTUP_H_

#include <oskit/error.h>
#include <oskit/io/blkio.h>


/*
 * Start the system clock and plug it into libc.
 * This function should be called only once.
 */
void start_clock(void);

/*
 * Convenience function to initialize/probe device drivers,
 * using link-time dependencies to select sets to initialize.
 * This can safely be called more than once, but is not thread-safe.
 * You do not necessarily have to call this, since other start_*
 * functions do.
 */
void start_devices(void);

/*
 * Open the named disk device and partition on it (or the whole disk
 * if `partition' is null).
 * This calls start_devices if need be, and causes it to probe block devices.
 */
oskit_error_t start_disk(const char *disk, const char *partition,
			 int read_only, oskit_blkio_t **out_bio);


/*
 * Initialize the disk-based filesystem code, mount a filesystem from
 * the given block device, and plug it into libc.
 */
void start_fs_on_blkio(oskit_blkio_t *part);

/* version used with pthreads */
void start_fs_on_blkio_pthreads(oskit_blkio_t *part);

/*
 * This just calls start_disk and start_fs_on_blkio for you.
 */
void start_fs(const char *diskname, const char *partname);

/* version used with pthreads */
void start_fs_pthreads(const char *diskname, const char *partname);


/*
 * Start up a network interface using BOOTP, initialize the network stack,
 * and register its socketfactory.
 * This calls start_devices if need be, and causes it to probe net devices.
 * This should be called only once.
 */
void start_network(void);

/* version used with pthreads */
void start_network_pthreads(void);


/*
 * Start the SVM page-protection system, paging to the given disk partition.
 * If disk is `null', does no disk paging.  Otherwise, args are passed
 * to `start_disk' (see above).
 */
oskit_error_t start_svm(const char *disk, const char *partition);

/* version used with pthreads */
oskit_error_t start_svm_pthreads(const char *disk, const char *partition);


/*
 * Convenience function to start up clock, disk filesystem, swap, and network
 * in canonical ways and plug them into libc.
 *
 * Looks for environment/command-line variables called "root" and "swap"
 * for device/partition selection with user-friendly syntax, and variable
 * "read-only" to open the root fs read-only.  If there is no "root"
 * variable, makes the bmod fs the root fs.  If there is no "swap" variable,
 * turns on page-protection with SVM, but no disk paging.
 * Starts the network with bootp.
 *
 * This can safely be called more than once, but is not thread-safe.
 */
void start_world(void);


#endif /* _OSKIT_STARTUP_H_ */
