/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 * This header file defines common types and functions used by
 * the minimal C library's default POSIX file system functions
 * (open, rename, link, unlink, symlink, etc.).
 * This header file is NOT a standard POSIX or Unix header file;
 * instead its purpose is to expose the implementation of this facility
 * so that the client can fully control it and use it in arbitrary contexts.
 *
 * The main function of this module is to keep track of
 * a root directory and a current working directory.
 * Upon initialization with a oskit_dir_t posing as rootdir,
 * it will perform open, access etc. with pathnames relative to the root.
 */
#ifndef _OSKIT_C_FS_H_
#define _OSKIT_C_FS_H_

#include <oskit/c/errno.h>
#include <oskit/fs/netbsd.h>
#include <oskit/fs/filesystem.h>
#include <oskit/fs/file.h>
#include <oskit/fs/openfile.h>
#include <oskit/fs/dir.h>
#include <oskit/compiler.h>

/* References to the root directory and current directory */
oskit_dir_t		*fs_root;
oskit_dir_t		*fs_cwd;

/* List of mount points and the nodes mounted on them */
struct fs_mountpoint {
	oskit_file_t		*subtree;	/* dir/file to mount */
	oskit_iunknown_t	*subtree_iu;	/* iunknown intf of above */
	oskit_dir_t		*mountover;	/* dir to mount it on */
	oskit_iunknown_t	*mountover_iu;	/* iunknown intf of above */
	struct fs_mountpoint	*next;
};
struct fs_mountpoint	*fs_mountpoints;

oskit_mode_t		fs_cmask; /* Creation mask on permissions (umask).  */


OSKIT_BEGIN_DECLS

/*** Initialization functions ***/
/*
 * The client calls these functions as appropriate
 * to initialize and build its file system namespace.
 */

/*
 * Provide a root directory defining the file system namespace.
 * The current directory will initially also be this root directory.
 */
oskit_error_t fs_init(oskit_dir_t *root);

/*
 * This releases the root and current directory references,
 * leaving the file system module in its original uninitialized state.
 */
void fs_release(void);

/*
 * BSD-like mount/unmount functions which the client can use
 * to build its file system namespace out of multiple file systems.
 * The fs_lookup() function below takes care of crossing mount points
 * while traversing the file system namespace.
 */
oskit_error_t fs_mount(const char *path, oskit_file_t *subtree);
oskit_error_t fs_unmount(const char *path);


/*** Internal functions and macros ***/
/*
 * The client normally doesn't need to use these directly,
 * but it can if it wants to do other special things.
 */

/* just do lookup - needed for opendir */
oskit_error_t fs_lookup(const char *name, oskit_file_t **out_file);

/* look up the dir a file is in */
oskit_error_t fs_lookup_dir(const char **inout_name, oskit_dir_t **out_dir);

OSKIT_END_DECLS

#endif /* _OSKIT_C_FS_H_ */
