/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#include "fs_glue.h"

#include <sys/domain.h>
#include <oskit/c/stdarg.h>
#include <machine/intr.h>

int cpl = 0;
int securelevel = 1;
struct vnode *rootvp = 0;
const char *panicstr = 0;
int imask[7] = {1, 2, 4, 8, 16, 32, 64};
struct proclist allproc;
struct proc *curproc = 0;
struct timeval time;
struct timeval mono_time; 
struct domain *domains = 0;

int copyinstr(void *udaddr, void *kaddr, size_t len, size_t *done)
{
    return copystr(udaddr,kaddr,len,done);
}


int copyoutstr(void *kaddr, void *udaddr, size_t len, size_t *done)
{
    return copystr(udaddr,kaddr,len,done);
}


int copyin(void *udaddr, void *kaddr, size_t len)
{
    bcopy(udaddr,kaddr,len);
    return 0;
}


int copyout(void *kaddr, void *udaddr, size_t len)
{
    bcopy(kaddr,udaddr,len);
    return 0;        
}


void delay(int n)
{
    struct proc *p;
    
    p = curproc;
    fs_delay(n);
    curproc = p;
    return;            
}


void printf(const char *fmt, ...)
{
    va_list ap;
    struct proc *p;

    va_start(ap, fmt);
    p = curproc;    
    fs_vprintf(fmt,ap);
    curproc = p;
    va_end(ap);
}


void uprintf(const char *fmt, ...)
{
    va_list ap;
    struct proc *p;

    va_start(ap, fmt);
    p = curproc;    
    fs_vprintf(fmt,ap);
    curproc = p;
    va_end(ap);
}


void log(int level, const char *fmt, ...)
{
    va_list ap;
    struct proc *p;

    va_start(ap, fmt);
    p = curproc;
    fs_vprintf(fmt,ap);
    curproc = p;
    va_end(ap);
}


int sprintf(char *buf, const char *fmt, ...)
{
    va_list ap;
    struct proc *p;

    va_start(ap, fmt);
    p = curproc;    
    fs_vsprintf(buf,fmt,ap);
    curproc = p;
    va_end(ap);
    return 0;
}


void panic(const char *fmt, ...)
{
    va_list ap;
    struct proc *p;

    va_start(ap, fmt);
    p = curproc;    
    fs_vprintf(fmt,ap);
    curproc = p;
    va_end(ap);
    fs_panic();
}


int gettime(struct timeval *tvp)
{
    struct oskit_timespec ts;
    oskit_error_t ferror;
    struct proc *p;
    
    p = curproc;
    ferror = fs_gettime(&ts);
    curproc = p;
    if (ferror)
	    return EFAULT; /* XXX */
	    
    TIMESPEC_TO_TIMEVAL(tvp,&ts);
    return 0;
}


void wakeup(void *chan)
{
    SSTATE_DECL;
    
    SSTATE_SAVE;
    fs_wakeup(chan);
    SSTATE_RESTORE;
}


int tsleep(void *chan, int pri, char *wmesg, int timo)
{
    oskit_error_t ferror;
    SSTATE_DECL;

    SSTATE_SAVE;
    ferror = fs_tsleep(chan,pri,wmesg,timo);
    SSTATE_RESTORE;
    
    if (ferror)
	    return EFAULT; /* XXX */
    return 0;
}


void *malloc(unsigned long size, int type, int flags)
{
    struct proc *p;
    void *ptr;
    
    p = curproc;
    ptr = fs_malloc(size);
    curproc = p;
    if (!ptr && (flags & M_WAITOK))
	    panic("out of memory");
    return ptr; 
}

    
void free (void *addr, int type)
{
    struct proc *p;
    p = curproc;
    fs_free(addr);
    curproc = p;
}


void *realloc(void *curaddr, unsigned long newsize)
{
    struct proc *p;
    void *ptr;
    
    p = curproc;
    ptr = fs_realloc(curaddr,newsize);
    curproc = p;
    return ptr;
}


void inittodr(time_t base)
{
    time.tv_sec = base;
    time.tv_usec = 0;
}

void microtime(struct timeval *tvp)
{
    if (gettime(tvp))
	    panic("gettimeofday");

    while (tvp->tv_usec > 1000000)
    {
	tvp->tv_sec++;
	tvp->tv_usec -= 1000000;
    }
}


void tablefull(const char *tab)
{
    printf("%s:  table is full\n", tab);
}


void sleep(void *chan, int pri)
{
    tsleep(chan, pri, "sleep", 0);
}


int copystr(void *f, void *t, size_t maxlen, size_t *lencopied)
{
    char *from = (char *) f;
    char *to = (char *) t;
    size_t len;


    len = 0;
    while (len < maxlen)
    {
	len++;
	if ((*to++ = *from++) == '\0')
	{
	    if (lencopied)
		    *lencopied = len;
	    return 0;
	}
    }

    if (lencopied)
	*lencopied = len;
    return ENAMETOOLONG;
}


int soo_stat(struct socket *so, struct stat *sb)
{
    bzero(sb,sizeof(sb));
    sb->st_mode = S_IFSOCK;
    return 0;
}




