/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#include "getproc.h"

static pid_t nextpid = 1;
extern struct vnode *rootvnode;

/* XXX mem leaks */

oskit_error_t getproc(struct proc **procp)
{
    oskit_principal_t *prin;
    oskit_identity_t id;
    oskit_error_t ferror;
    struct proc *proc;
    struct pcred *pcred;
    struct ucred *cred;
    struct filedesc0 *fd;
    struct plimit *limit;
    int index;

    
    proc = malloc(sizeof(struct proc),M_TEMP,M_WAITOK); 
    if (!proc)
	    return OSKIT_ENOMEM;
    bzero(proc,sizeof(struct proc));

    pcred = malloc(sizeof(struct pcred),M_TEMP,M_WAITOK);
    if (!pcred)
	    return OSKIT_ENOMEM;
    bzero(pcred,sizeof(struct pcred));
    pcred->p_refcnt = 1;
    proc->p_cred = pcred;

    ferror = oskit_get_call_context(&oskit_principal_iid, (void *)&prin);
    if (ferror)
	    return ferror;

    ferror = oskit_principal_getid(prin, &id);
    if (ferror)
	    return ferror;

    if (id.ngroups > NGROUPS)
	    return OSKIT_EINVAL;

    cred = crget();
    if (!cred)
	    return OSKIT_ENOMEM;

    cred->cr_uid = id.uid;
    cred->cr_gid = id.gid;
    cred->cr_ngroups = id.ngroups;
    for (index = 0; index < id.ngroups; index++)
	    cred->cr_groups[index] = id.groups[index];

    pcred->p_ruid = pcred->p_svuid = id.uid;
    pcred->p_rgid = pcred->p_svgid = id.gid;
    
    FREE(id.groups, M_TEMP);
    oskit_principal_release(prin);

    proc->p_ucred = cred;

    fd = malloc(sizeof(struct filedesc0),M_TEMP,M_WAITOK);
    if (!fd)
	    return OSKIT_ENOMEM;
    bzero(fd,sizeof(struct filedesc0));
    fd->fd_fd.fd_refcnt = 1;
    fd->fd_fd.fd_cmask = 0;	/* umask is applied by client.  */
    fd->fd_fd.fd_ofileflags = fd->fd_dfileflags;
    fd->fd_fd.fd_nfiles = NDFILE;
    proc->p_fd = &fd->fd_fd;

    fd->fd_fd.fd_rdir = rootvnode = 0;

    limit = malloc(sizeof(struct plimit),M_TEMP,M_WAITOK);
    if (!limit)
	    return OSKIT_ENOMEM;
    limit->p_refcnt = 1;
    for (index = 0; index < RLIM_NLIMITS; index++)
	    limit->pl_rlimit[index].rlim_cur = 
		    limit->pl_rlimit[index].rlim_max = RLIM_INFINITY;
    proc->p_limit = limit;

    proc->p_stats = malloc(sizeof(struct pstats),M_TEMP,M_WAITOK);
    if (!proc->p_stats)
	    return OSKIT_ENOMEM;
    bzero(proc->p_stats, sizeof(struct pstats));

    if (nextpid == 0)
	    nextpid = 1;
    proc->p_pid = nextpid++;    

    *procp = proc;

    curproc = proc;

    return 0;
}


void prfree(struct proc *proc)
{
    free(proc->p_limit,M_TEMP);
    free(proc->p_fd,M_TEMP);
    crfree(proc->p_ucred);
    free(proc->p_cred,M_TEMP);
    free(proc->p_stats,M_TEMP);
    free(proc,M_TEMP);
    curproc = 0;
}
