/*
 * Copyright (c) 1996, 1997, 1998 The University of Utah and the Flux Group.
 * 
 * This file is part of the OSKit Linux Glue Libraries, which are free
 * software, also known as "open source;" you can redistribute them and/or
 * modify them under the terms of the GNU General Public License (GPL),
 * version 2, as published by the Free Software Foundation (FSF).
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */
/*
 * Definitions of things normally provided by Linux.
 */

#include <oskit/dev/dev.h>
#include <oskit/dev/net.h>
#include <oskit/io/bufio.h>
#include <oskit/io/netio.h>
#ifdef HPFQ
#include <oskit/hpfq.h>
#endif

#include <oskit/c/assert.h>

#include "net.h"
#include "net_debug.h"

#include <linux/skbuff.h>		/* struct sk_buff */
#include <linux/malloc.h>		/* kmalloc/free */
#include <linux/netdevice.h>		/* struct device */
#include <linux/interrupt.h>		/* NET_BH */
#include <linux/autoconf.h>		/* CONFIG_foo */
#include <linux/string.h>		/* memcmp */

#include "irq.h"

static struct sk_buff_head backlog;	/* the receive queue */
static int backlog_size = 0;
const int backlog_max = 300;		/* wow, but that is what Linux has */


/*
 * Forward declarations.
 */
static OSKIT_COMDECL skbuff_io_query(oskit_bufio_t *io, const oskit_iid_t *iid,
                               void **out_ihandle);
static OSKIT_COMDECL_U skbuff_io_addref(oskit_bufio_t *io);
static OSKIT_COMDECL_U skbuff_io_release(oskit_bufio_t *io);

static OSKIT_COMDECL skbuff_io_getsize(oskit_bufio_t *io, oskit_off_t *out_size);
static OSKIT_COMDECL skbuff_io_setsize(oskit_bufio_t *io, oskit_off_t new_size);

static OSKIT_COMDECL skbuff_io_read(oskit_bufio_t *io, void *dest, 
				   oskit_off_t offset, oskit_size_t count,
				   oskit_size_t *out_actual);
static OSKIT_COMDECL skbuff_io_write(oskit_bufio_t *io, const void *dest, 
				    oskit_off_t offset, oskit_size_t count,
				    oskit_size_t *out_actual);

static OSKIT_COMDECL skbuff_io_map(oskit_bufio_t *io, void **out_addr,
				  oskit_off_t offset, oskit_size_t count);
static OSKIT_COMDECL skbuff_io_unmap(oskit_bufio_t *io, void *addr,
				    oskit_off_t offset, oskit_size_t count);
static OSKIT_COMDECL skbuff_io_wire(oskit_bufio_t *io, oskit_addr_t *out_phys_addr,
				   oskit_off_t offset, oskit_size_t count);
static OSKIT_COMDECL skbuff_io_unwire(oskit_bufio_t *io, oskit_addr_t phys_addr,
				     oskit_off_t offset, oskit_size_t count);

static OSKIT_COMDECL skbuff_io_copy(oskit_bufio_t *io, 
				   oskit_off_t offset,
				   oskit_size_t count,
				   oskit_bufio_t **out_io);


static struct oskit_bufio_ops linux_skbuff_io_ops = {
        skbuff_io_query, 
	skbuff_io_addref, 
	skbuff_io_release,
        (void *) 0,
	skbuff_io_read, 
	skbuff_io_write, 
	skbuff_io_getsize,
	skbuff_io_setsize,
        skbuff_io_map, 
	skbuff_io_unmap, 
	skbuff_io_wire, 
	skbuff_io_unwire,
	skbuff_io_copy
};




/*
 *      Add data to an sk_buff
 */

/*
 * Add data to an sk_buff.
 *
 * Linux	net/core/skbuff.c
 */
unsigned char *
skb_put(struct sk_buff *skb, int len)
{
        unsigned char *tmp = skb->tail;

        IS_SKB(skb);
        skb->tail += len;
        skb->len += len;
        IS_SKB(skb);

        if (skb->tail > skb->end)
                panic("skput:over: %p:%d", __builtin_return_address(0),len);

        return tmp;
}



/*
 * preallocate some space at the beginning of the sk_buff buffer. 
 *
 * Linux	net/core/skbuff.c
 */
void 
skb_reserve(struct sk_buff *skb, int len)
{
        IS_SKB(skb);
        skb->data += len;
        skb->tail += len;

        if (skb->tail > skb->end)
                panic("sk_res: over");
        if (skb->data < skb->head)
                panic("sk_res: under");
        IS_SKB(skb);
}


/*
 * Initialize the Linux network glue module.
 * This roughly corresponds to Linux's net_dev_init(),
 * except it doesn't walk the static device list
 * because we don't have one!
 */
oskit_error_t
oskit_linux_net_init(void)
{
	static int initialized;

	if (initialized)
		return 0;
	initialized = 1;

	skb_queue_head_init(&backlog);
	init_bh(NET_BH, net_bh);

	return 0;
}


/*
 * Close the device DEV.
 *
 * Compare:
 *	Linux	net/core/dev.c
 *	Mach4	linux_net.c
 *
 * The Linux one calls the dev->stop routine and purges any queued packets.
 * That is what we do.
 * The Mach one does NOTHING!  We used to do that but it was a bad idea.
 */
int
dev_close(struct device *dev)
{
	int ct = 0;

	/*
	 *	Call the device specific close. This cannot fail.
	 *	Only if device is UP
	 */
	 
	if ((dev->flags & IFF_UP) && dev->stop)
		dev->stop(dev);

	/*
	 *	Device is now down.
	 */
	 
	dev->flags&=~(IFF_UP|IFF_RUNNING);

	/*
	 *	Purge any queued packets when we down the link 
	 */
	while(ct < DEV_NUMBUFFS)
	{
		struct sk_buff *skb;
		while((skb = skb_dequeue(&dev->buffs[ct])) != NULL)
			kfree_skb(skb, FREE_WRITE);
		ct++;
	}
	return 0;
}


/* Memory allocation. */


#define ALIGNBYTES              7
#define ALIGN(p)                (((u_long)(p) + ALIGNBYTES) & ~ALIGNBYTES)

/*
 * Allocate an sk_buff with SIZE bytes of data space and fill in a few
 * fields.
 *
 * This is adapted from the Mach one so we set skb->len to size
 * instead of zero like the Linux version does.
 * It is not clear to me why this is but I think it has to do with Linux
 * using the skb_put, etc macros to set len.
 * Also, the Mach one called kmalloc with GFP_KERNEL instead of ATOMIC for
 * some unknown reason.
 *
 * Compare
 *	Linux	net/core/skbuff.c
 *	Mach4	linux_net.c
 */
struct sk_buff *
alloc_skb(unsigned int size, int priority)
{
	struct sk_buff *skb;
	int len = size;

	assert(priority == GFP_ATOMIC);
	
	size += ALIGN(sizeof(struct sk_buff));

	skb = kmalloc(size, priority); 

	if (skb) {
		skb->dev = NULL;
		skb->len = 0;
		skb->truesize = size;
	
		skb->prev = skb->next = NULL;
		skb->list = NULL;

		skb->data = (len ? 
			     (char *)(skb) + ALIGN(sizeof(struct sk_buff)) : 
			     NULL);

		skb->head = skb->tail = skb->data;
		skb->end = skb->data + len;

		skb->fdev_count = 1;
		skb->data_io = NULL;

		skb->ioi.ops = &linux_skbuff_io_ops;
	}
	return skb;
}

/*
 * Wrapper for alloc_skb.
 *
 * The Linux version allocates 16 bytes too much for some unknown
 * reason.  The Mach version doesn't but reverses the sense of
 * alloc_skb and dev_alloc_skb.
 *
 * Compare
 *	Linux	net/core/skbuff.c
 *	Mach4	linux_net.c
 */
struct sk_buff *
dev_alloc_skb(unsigned int size)
{
        struct sk_buff *skb;

	/* Allocate and extra 16 bytes at the beginning for an
	 * ethernet header.
	 */
        skb = alloc_skb(size + 16, GFP_ATOMIC);
        if (skb)
                skb_reserve(skb, 16);

        return skb;
}

/*
 * Free the sk_buff SKB.
 *
 * The Linux version does some lock and refcount checking but,
 * like the Mach one, we don't.  And like before, I don't know why.
 * 
 * Compare
 *	Linux	net/core/skbuff.c
 *	Mach4	linux_net.c
 */
void
kfree_skb(struct sk_buff *skb, int rw)
{

	if (skb == NULL) {
                printk(KERN_CRIT "kfree_skb: skb = NULL (from %p)\n",
                        __builtin_return_address(0));
                return;
	}

	/*
	 * If we've got a data_io pointer, then we need to unmap the 
	 * buffer and release our handler to the bufio object.
	 */
	if (skb->data_io) {
		oskit_bufio_unmap(skb->data_io, skb->data, 0, skb->len);
		oskit_bufio_release(skb->data_io);
	}

	/* 
	 * We don't have to deallocate our buffer explicitly, since
	 * the sk_buff and data get allocated as one huge contiguous
	 * buffer, our implementation of linux_kfree will deallocate
	 * the whole thing.
	 */
	kfree(skb);
}

/*
 * Wrapper for kfree_skb.
 *
 * Linux does some lock checking but Mach and us do not.
 *
 * Compare
 *	Linux	net/core/skbuff.c
 *	Mach4	linux_net.c
 */
void
dev_kfree_skb(struct sk_buff *skb, int mode)
{
	kfree_skb(skb, mode);
}

/* Software interrupts. */

/*
 * This is similar but not equivalent to Linux's dev_transmit,
 * therefore it is not named dev_transmit.
 */
static void
my_dev_transmit(void)
{
	int len;
	struct sk_buff *skb;
	struct device *dev;

	/* Start transmission on interfaces.  */
	for (dev = dev_base; dev != NULL; dev = dev->next) {
		/* Linux just checks for non-zero dev->flags. */
		if (!(dev->flags & IFF_UP) || dev->tbusy)
			continue;

#ifdef HPFQ
		if (oskit_pfq_root) {
			oskit_pfq_reset_path(oskit_pfq_root);
			continue;
		}
#endif

		DPRINTF(D_VSEND, "%s: trying to send\n", dev->name);
		for (;;) {
			skb = skb_dequeue(&dev->buffs[0]);
			if (skb == NULL) {
				DPRINTF(D_VSEND, "%s: nothing to send\n",
					dev->name);
				break;
			}
			len = skb->len;

			if ((*dev->hard_start_xmit)(skb, dev) != 0) {
				DPRINTF(D_VSEND, "%s: send failed, requeue\n",
					dev->name);
				skb_queue_head(&dev->buffs[0], skb);
				mark_bh(NET_BH);
				break;
			}
			else
				DPRINTF(D_SEND, "%s: sent pkt w/len %d\n",
					dev->name, len);
		}
	}
}

/*
 * The software interrupt handler, or bottom-half handler.
 * Its job is to start transmission on all interfaces and to
 * try to empty the receive queue.
 *
 * We are called with interrupts on (XXX make an assert()ion.)
 *
 * The Linux one makes sure it is called atomically but that is only
 * needed if this is NOT called thru the bottom-half mechanism.
 *
 *	Linux	net/core/dev.c
 *	Mach4	linux_net.c
 */
void
net_bh(void)
{
	struct sk_buff *skb;
	
	my_dev_transmit();

	/*
	 * Be careful not to break out of this loop strangely
	 * since we need the cli before dequeueing.
	 * No, we can't just use the non-underscore version of skb_dequeue
	 * since we need to serialize access to backlog_size too.
	 */
	linux_cli();
	while ((skb = __skb_dequeue(&backlog)) != NULL) {
		struct device *dev = skb->dev;
		oskit_error_t rc;

		backlog_size--;
		linux_sti();
		DPRINTF(D_RECV, "passing packet of len %ld to OS\n", skb->len);

		rc = oskit_netio_push(dev->my_alias->recv_ioi, &skb->ioi, skb->len);
		if (rc != 0) {
			DPRINTF(D_VRECV, "oskit_netio_push failed\n");
		}			
		oskit_bufio_release(&skb->ioi);
		linux_cli();
	}
	linux_sti();

	my_dev_transmit();		/* supposedly a ood idea */
}


/* Hardware interrupts. */

/*
 * Accept packet SKB received on an interface and queue it.
 *
 * Compare
 *	Linux	net/core/dev.c
 *	Mach4	linux_net.c
 *
 * The Mach one copies it into a Mach ipc_kmsg_t, frees it, and calls
 * Mach's net_packet (which does queuing.)
 * The Linux one adds it to a backlog queue (which is emptied in the
 * bottom half handler), and then mark_bh(NET_BH)
 */
void
netif_rx(struct sk_buff *skb)
{
	static int dropping = 0;

	/*
	 *	Check that we aren't overdoing things.
	 */
	DPRINTF(D_RECV, "%s gave me pkt of len %ld, thanks! can I keep it?\n",
		skb->dev->name, skb->len);
	DPRINTF(D_VRECV, "backlog size %d\n", backlog_size);
	if (!backlog_size)
  		dropping = 0;
	else if (backlog_size > backlog_max)
		dropping = 1;

	if (dropping) 
	{
		kfree_skb(skb, FREE_READ);
		return;
	}

	/*
	 *	Add it to the "backlog" queue. 
	 */
#if CONFIG_SKB_CHECK
	IS_SKB(skb);
#endif	

	skb_queue_tail(&backlog, skb);
	backlog_size++;
  
	/*
	 *	If any packet arrived, mark it for processing after the
	 *	hardware interrupt returns.
	 */

#ifdef CONFIG_NET_RUNONIRQ	/* Dont enable yet, needs some driver mods */
	net_bh();
#else
	mark_bh(NET_BH);
#endif
	return;
}



/*
 * skbuff COM interface definitions.
 */

#define GET_SKB_FROM_BUF_IO(io, skb) { \
	/* our bufio ops struct is at the end of the skbuff. */ \
	(skb) = (struct sk_buff *)(((char *)(io)) - \
		(sizeof(struct sk_buff) - sizeof(struct oskit_bufio))); \
	if ((skb) == NULL) \
                panic("%s:%d: null bufio_t", __FILE__, __LINE__); \
        if ((skb)->fdev_count == 0) \
                panic("%s:%d: bad count", __FILE__, __LINE__); \
} 


static OSKIT_COMDECL 
skbuff_io_query(oskit_bufio_t *io, const oskit_iid_t *iid,
			    void **out_ihandle) 
{
	struct sk_buff *skb;

	
	GET_SKB_FROM_BUF_IO(io, skb);

        if (memcmp(iid, &oskit_iunknown_iid, sizeof(*iid)) == 0 ||
            memcmp(iid, &oskit_bufio_iid, sizeof(*iid)) == 0) {
                *out_ihandle = &skb->ioi;
                ++skb->fdev_count;

                return 0;
        }

        *out_ihandle = NULL;
        return OSKIT_E_NOINTERFACE;
}


static OSKIT_COMDECL_U 
skbuff_io_addref(oskit_bufio_t *io)
{
	struct sk_buff *skb;

	GET_SKB_FROM_BUF_IO(io, skb);

        return ++skb->fdev_count;
}

static OSKIT_COMDECL_U 
skbuff_io_release(oskit_bufio_t *io) 
{
	struct sk_buff *skb;
	unsigned newcount;

	GET_SKB_FROM_BUF_IO(io, skb);

	if ((newcount = --skb->fdev_count) == 0) {
                /* ref count = 0, need to free! */

		/* 
		 * We ignore the parameter to kfree_skb, since it only
		 * has to do with freeing the struct sock in an sk_buff,
		 * which we don't support anyway. =)
		 */
		kfree_skb(skb, 0);
        }

        return newcount;
}


static OSKIT_COMDECL skbuff_io_getsize(oskit_bufio_t *io, oskit_off_t *out_size)
{
	struct sk_buff *skb;

	GET_SKB_FROM_BUF_IO(io, skb);

	*out_size = skb->len;
	return 0;
}

static OSKIT_COMDECL skbuff_io_setsize(oskit_bufio_t *io, oskit_off_t new_size)
{
	return OSKIT_E_NOTIMPL;
}


static OSKIT_COMDECL 
skbuff_io_read(oskit_bufio_t *io, void *dest,
	       oskit_off_t offset, oskit_size_t count, oskit_size_t *out_actual) 
{
	struct sk_buff *skb;

	GET_SKB_FROM_BUF_IO(io, skb);

	if (offset + count > skb->len)
		count = skb->len - offset;

        memcpy(dest, skb->data + offset, count);
	
	*out_actual = count;

        return (0);
}


static OSKIT_COMDECL 
skbuff_io_write(oskit_bufio_t *io, const void *src,
		oskit_off_t offset, oskit_size_t count,
		oskit_size_t *out_actual)
{
	struct sk_buff *skb;

	GET_SKB_FROM_BUF_IO(io, skb);

	if (offset + count > skb->len)
		count = skb->len - offset;

        memcpy(skb->data + offset, src, count);

	*out_actual = count;

        return 0;
}


static OSKIT_COMDECL 
skbuff_io_map(oskit_bufio_t *io, void **out_addr,
	      oskit_off_t offset, oskit_size_t count)
{
	struct sk_buff *skb;

	GET_SKB_FROM_BUF_IO(io, skb);

	if (offset + count > skb->len)
		return OSKIT_EINVAL;

 	*out_addr = skb->data + offset;

        return 0;
}

/* 
 * XXX These should probably do checking on their inputs so we could 
 * catch bugs in the code that calls them.
 */

static OSKIT_COMDECL 
skbuff_io_unmap(oskit_bufio_t *io, void *addr,
		oskit_off_t offset, oskit_size_t count) 
{
	return 0;
}

static OSKIT_COMDECL 
skbuff_io_wire(oskit_bufio_t *io, oskit_addr_t *out_phys_addr,
	       oskit_off_t offset, oskit_size_t count)
{
	return OSKIT_E_NOTIMPL; /*XXX*/
}

static OSKIT_COMDECL 
skbuff_io_unwire(oskit_bufio_t *io, oskit_addr_t phys_addr,
	         oskit_off_t offset, oskit_size_t count)
{
	return OSKIT_E_NOTIMPL; /*XXX*/
}

static OSKIT_COMDECL skbuff_io_copy(oskit_bufio_t *io, 
				   oskit_off_t offset,
				   oskit_size_t count,
				   oskit_bufio_t **out_io)
{
    return OSKIT_E_NOTIMPL;
}

