/*
 * Copyright (c) 1994-1995, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/* This is a special "feature" (read: kludge)
   intended for use only for kernel debugging.
   It enables an extremely simple console output mechanism
   that sends text straight to CGA/EGA/VGA video memory.
   It has the nice property of being functional right from the start,
   so it can be used to debug things that happen very early
   before any devices are initialized.  */

#include <oskit/c/string.h>
#include <oskit/base_critical.h>
#include <oskit/x86/base_vm.h>
#include <oskit/x86/pio.h>
#include <oskit/x86/pc/direct_cons.h>


unsigned char *vidbase;
unsigned short video_port_reg;
#define video_port_val (video_port_reg + 1)

#define MONO_BUF ((unsigned char *)phystokv(0xb0000))
#define MONO_BASE 0x3b4
#define CGA_BUF ((unsigned char *)phystokv(0xb8000))
#define CGA_BASE 0x3d4


static void
fillw(unsigned short pat, const void *base, oskit_size_t cnt) {
	volatile unsigned short *p;

	p = (unsigned short *)base;

	while (cnt--)
		*p++ = pat;
}

static void
set_cursor(unsigned int pos) 
{
	pos += 80 * 24;  

	/* set cursor position high byte */
	outb_p(video_port_reg, 0xe);
	outb_p(video_port_val, (unsigned char)((pos >> 8)) & 0xff);
	
	/* set cursor position low byte */
        outb_p(video_port_reg, 0xf);
        outb_p(video_port_val, (unsigned char)(pos & 0xff));
}

void
direct_cons_putchar(unsigned char c)
{
	static int ofs = -1;

	base_critical_enter();

	if (ofs < 0)
	{
		/* Called for the first time - initialize.  */ 
		volatile unsigned short *p = (unsigned short *)CGA_BUF, val;

		val = *p;
		*p = (unsigned short) 0xa55a;
		if (*p != 0xa55a) {
			vidbase = MONO_BUF;
			video_port_reg = MONO_BASE;
		} else {
			*p = val;
			vidbase = CGA_BUF;
			video_port_reg = CGA_BASE;
		}

		ofs = 0;
		direct_cons_putchar('\n');
	}

	switch (c)
	{
	case '\n':
		memcpy(vidbase, vidbase + 80*2, 80*24*2);
		fillw(0x0f00, vidbase + 80*24*2, 80);

		/* fall through... */
	case '\r':
		ofs = 0;
		break;
	case '\b':
		if (ofs > 0) ofs--;
		break;
	case '\t':
		do {
			direct_cons_putchar(' ');
		} while ((ofs & 7) != 0);
		break;

	default:
		/* Wrap if we reach the end of a line.  */
		if (ofs >= 80) {
			direct_cons_putchar('\n');
		}

		/* Stuff the character into the video buffer. */
		{
			volatile unsigned char *p = vidbase + 
						    (80*24 + ofs) * 2;

			p[0] = c;
			p[1] = 0x0f;

			ofs++;
		}

		break;
	}

	set_cursor(ofs);

	base_critical_leave();
}

