/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 * Implements oskit_file_t, oskit_dir_t, and oskit_openfile_t for the bmod
 * filesystem.
 *
 * We only implement the parts that make sense.
 */
#include <oskit/fs/dir.h>
#include <oskit/fs/openfile.h>
#include <oskit/fs/filesystem.h>
#include <oskit/io/absio.h>
#include <oskit/io/bufio.h>
#include <oskit/x86/multiboot.h>
#include <oskit/x86/base_paging.h>
#include <oskit/x86/pc/base_multiboot.h>
#include <oskit/x86/pc/phys_lmm.h>
#include <oskit/fs/bmodfs.h>
#include <string.h>
#include <sys/stat.h>
#include <assert.h>
#include <errno.h>
#include <stdarg.h>
#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <fcntl.h>

#ifndef VERBOSITY
#define VERBOSITY 	0
#endif

#if VERBOSITY > 20
#define DMARK printf(__FILE__ ":%d: " __FUNCTION__ "\n", __LINE__)
#else
#define DMARK ((void)0)		/* no-op */
#endif

/*
 * struct to describe a bmod file or directory;
 * the ops vector differentiates.
 */
struct bmod {
	oskit_file_t filei;
	oskit_u32_t count;

	char *name;		/* name of this file */
	oskit_size_t namelen;	/* bytes in name, not including terminator */
        struct bmod *next, **prevp; /* siblings in directory */

	union {
		struct {
			oskit_bufio_t bufioi;
			oskit_size_t size;	/* file size */
			void *data;		/* file data */
			oskit_size_t allocsize;	/* bytes smalloc'd */
			oskit_bool_t can_sfree;	/* data is smalloc'd */
			oskit_bool_t inhibit_resize; /* disallow resizing */
		} file;
		struct {
			oskit_dir_t *parent; /* .. directory, null for self */
			struct bmod *contents; /* files in this directory */
		} dir;
	} data;
};

static struct oskit_file_ops bmod_file_ops, bmod_symlink_ops;
static struct oskit_dir_ops bmod_dir_ops;
static struct oskit_bufio_ops bmod_bufio_ops;


static OSKIT_COMDECL
bmod_dir_query(oskit_dir_t *b0, const oskit_iid_t *iid, void **out_ihandle)
{
	struct bmod *b = (struct bmod *)b0;
	assert(b->count);

        if (memcmp(iid, &oskit_iunknown_iid, sizeof(*iid)) == 0 ||
            memcmp(iid, &oskit_posixio_iid, sizeof(*iid)) == 0 ||
            memcmp(iid, &oskit_file_iid, sizeof(*iid)) == 0 ||
            memcmp(iid, &oskit_dir_iid, sizeof(*iid)) == 0) {
                *out_ihandle = b;
                ++b->count;
                return 0;
        }

        *out_ihandle = NULL;
        return OSKIT_E_NOINTERFACE;
}

static OSKIT_COMDECL
bmod_file_query(oskit_file_t *b0, const oskit_iid_t *iid, void **out_ihandle)
{
        struct bmod *b = (struct bmod *)b0;
	assert(b->count);

        if (memcmp(iid, &oskit_iunknown_iid, sizeof(*iid)) == 0 ||
            memcmp(iid, &oskit_posixio_iid, sizeof(*iid)) == 0 ||
            memcmp(iid, &oskit_file_iid, sizeof(*iid)) == 0) {
                *out_ihandle = b;
                ++b->count;
                return 0;
        }

	if (memcmp(iid, &oskit_bufio_iid, sizeof(*iid)) == 0 ||
	    memcmp(iid, &oskit_absio_iid, sizeof(*iid)) == 0) {
		*out_ihandle = &b->data.file.bufioi;
		++b->count;
		return 0;
	}

        *out_ihandle = NULL;
        return OSKIT_E_NOINTERFACE;
}

static OSKIT_COMDECL_U
bmod_file_addref(oskit_file_t *b0)
{
        struct bmod *f = (struct bmod *)b0;
	assert(f->count);
	return ++f->count;
}


static OSKIT_COMDECL_U
bmod_file_release(oskit_file_t *b0)
{
        struct bmod *b = (struct bmod *)b0;
	assert(b->count);

	if (--b->count)
		return b->count;

	sfree(b->name, b->namelen + 1);
	if ((void *)b->filei.ops == &bmod_dir_ops) {
		if (b->data.dir.parent)
			oskit_dir_release(b->data.dir.parent);

	} else {
		if (b->data.file.can_sfree)
			sfree(b->data.file.data, b->data.file.allocsize);
	}
	sfree(b, sizeof *b);
	return 0;
}


static struct bmod *
bmod_file_allocate(const char *name, void *ops)
{
	struct bmod *new = smalloc(sizeof *new);
	if (new) {
		new->namelen = strlen(name);
		new->name = smalloc(new->namelen + 1);
		if (!new->name) {
			sfree(new, sizeof *new);
			return NULL;
		}
		memcpy(new->name, name, new->namelen + 1);

		new->count = 1;
		new->filei.ops = ops;
		if (ops == &bmod_dir_ops) {
			new->data.dir.parent = NULL;
			new->data.dir.contents = NULL;
		} else {
			new->data.file.bufioi.ops = &bmod_bufio_ops;
			new->data.file.size = 0;
			new->data.file.allocsize = 0;
			new->data.file.can_sfree = 0;
			new->data.file.inhibit_resize = 0;
		}
	}
	return new;
}



/*** Operations inherited from oskit_posixio_t ***/

static OSKIT_COMDECL
bmodfile_stat(oskit_file_t *f, struct oskit_stat *st)
{
	struct bmod *b = (void *)f;

	memset(st, 0, sizeof *st);
	st->ino = (oskit_ino_t) b;
	st->size = b->data.file.size;
        st->mode = OSKIT_S_IFREG | OSKIT_S_IRWXG | OSKIT_S_IRWXU | OSKIT_S_IRWXO;
	return 0;
}

static OSKIT_COMDECL
bmoddir_stat(oskit_dir_t *f, struct oskit_stat *st)
{
	struct bmod *b = (void *)f;

	memset(st, 0, sizeof *st);
	st->ino = (oskit_ino_t) b;
        st->mode = OSKIT_S_IFDIR | OSKIT_S_IRWXG | OSKIT_S_IRWXU | OSKIT_S_IRWXO;
	return 0;
}

static oskit_error_t
bmod_resize(struct bmod *b, oskit_size_t size, oskit_bool_t override_inhibit)
{
	if (size < 0)
		return OSKIT_EINVAL;
	else if (size == b->data.file.size)
		return 0;
	else if (!override_inhibit && b->data.file.inhibit_resize)
		return OSKIT_EPERM;
	else if (size < b->data.file.allocsize) {
		if (size > b->data.file.size)
				/*
				 * Zero-fill the new space.
				 */
			memset((char *)b->data.file.data +
			       b->data.file.size, 0,
			       size - b->data.file.size);
		else if (size == 0 && b->data.file.can_sfree) {
			sfree(b->data.file.data,
			      b->data.file.allocsize);
			b->data.file.allocsize = 0;
			b->data.file.can_sfree = 0;
		}
		b->data.file.size = size;
	} else {
		/*
		 * Must grow the file.
		 */
		char *new;
		oskit_size_t newsize = round_page(size);
		new = smemalign(PAGE_SIZE, newsize);
		if (!new)
			return OSKIT_ENOSPC;
		else {
			memcpy(new, b->data.file.data,
			       b->data.file.size);
			memset(new + b->data.file.size, 0,
			       size - b->data.file.size);
			if (b->data.file.can_sfree)
				sfree(b->data.file.data,
				      b->data.file.allocsize);
			b->data.file.allocsize = newsize;
			b->data.file.can_sfree = 1;
			b->data.file.size = size;
			b->data.file.data = new;
		}
	}
	return 0;
}

static OSKIT_COMDECL
bmodfile_setstat(oskit_file_t *f, oskit_u32_t mask, const struct oskit_stat *stats)
{
	struct bmod *b = (void *)f;

	if (mask & ~OSKIT_STAT_SIZE)
		return OSKIT_ENOTSUP;
	if (mask & OSKIT_STAT_SIZE) {
		oskit_error_t rc;
		oskit_bmod_lock();
		rc = bmod_resize(b, stats->size, 0);
		oskit_bmod_unlock();
		return rc;
	}
	return 0;
}

static OSKIT_COMDECL
no_setstat(oskit_file_t *f, oskit_u32_t mask, const struct oskit_stat *stats)
{
	return OSKIT_E_NOTIMPL;
}

static OSKIT_COMDECL
no_pathconf(oskit_file_t *f, oskit_s32_t option, oskit_s32_t *out_val)
{
	return OSKIT_E_NOTIMPL;
}

static OSKIT_COMDECL
bmodfile_sync(oskit_file_t *f, oskit_bool_t wait)
{
	return 0;		/* Data is sunk as far as it goes.  */
}

static OSKIT_COMDECL
bmodfile_datasync(oskit_file_t *f, oskit_bool_t wait)
{
	return 0;		/* Data is sunk as far as it goes.  */
}

static OSKIT_COMDECL
bmodfile_access(oskit_file_t *f, oskit_amode_t mask)
{
	/* Sure, access me big boy.  */
	return 0;
}

static OSKIT_COMDECL
bmodfile_readlink(oskit_file_t *f, char *buf, oskit_u32_t len,
		  oskit_u32_t *out_actual)
{
	return OSKIT_E_NOTIMPL;
}

static OSKIT_COMDECL
bmodfile_open(oskit_file_t *d, oskit_oflags_t flags,
	     struct oskit_openfile **out_opendir)
{
	/*
	 * We allow opening, but we don't do per-open state ourselves.
	 */
	*out_opendir = NULL;
	return 0;
}

static OSKIT_COMDECL
bmoddir_open(oskit_dir_t *d, oskit_oflags_t flags,
	     struct oskit_openfile **out_opendir)
{
	if ((flags & OSKIT_O_ACCMODE) != OSKIT_O_RDONLY)
		return OSKIT_EISDIR;

	/*
	 * We allow opening, but we don't do per-open state ourselves.
	 */
	*out_opendir = NULL;
	return 0;
}


static oskit_file_t *
bmod_lookup(struct bmod *dir, const char *name)
{
	struct bmod *b;
	oskit_size_t len = strlen(name);

	if (len == 0 || (len == 1 && name[0] == '.'))
		return &dir->filei;
	if (len == 2 && name[0] == '.' && name[1] == '.')
		return (oskit_file_t *)dir->data.dir.parent ?: &dir->filei;

	for (b = dir->data.dir.contents; b; b = b->next)
		if (b->namelen == len && memcmp(name, b->name, len) == 0)
			return &b->filei;

	return NULL;
}

static OSKIT_COMDECL
bmoddir_lookup(oskit_dir_t *d, const char *name, oskit_file_t **out_file)
{
	struct bmod *dir = (void *)d;
	oskit_file_t *file;

#if VERBOSITY > 1
	printf(__FUNCTION__": asked to look up `%s'\n", name);
#endif

	file = bmod_lookup(dir, name);
	if (file) {
		oskit_file_addref(file);
		*out_file = file;
		return 0;
	}

	return OSKIT_ENOENT;
}

/*
 * Link B into DIR's contents chain.
 */
static inline void
dir_contents_link(struct bmod *dir, struct bmod *b)
{
	b->next = dir->data.dir.contents;
	b->prevp = &dir->data.dir.contents;
	if (dir->data.dir.contents)
		dir->data.dir.contents->prevp = &b->next;
	dir->data.dir.contents = b;
}

static OSKIT_COMDECL
bmoddir_create(oskit_dir_t *d, const char *name,
	       oskit_bool_t excl, oskit_mode_t mode,
	       oskit_file_t **out_file)
{
	struct bmod *dir = (void *)d;
	oskit_file_t *file;
	oskit_error_t rc;

#if VERBOSITY > 1
	printf(__FUNCTION__": asked to create `%s'\n", name);
#endif

	oskit_bmod_lock();

	file = bmod_lookup(dir, name);
	if (!file) {
		struct bmod *b = bmod_file_allocate(name, &bmod_file_ops);
		if (!b)
			rc = OSKIT_ENOSPC;
		else {
			dir_contents_link(dir, b);

			/*
			 * The directory holds one ref for the file object,
			 * and we add another for the pointer going out.
			 */
			++b->count;
			*out_file = &b->filei;
			rc = 0;
		}
	} else if (excl)
		rc = OSKIT_EEXIST;
	else {
		oskit_file_addref(file);
		*out_file = file;
		rc = 0;
	}

	oskit_bmod_unlock();
	return rc;
}

static OSKIT_COMDECL
bmoddir_link(oskit_dir_t *d, char *name, oskit_file_t *file)
{
	return OSKIT_E_NOTIMPL;
}

/*
 * Remove B from its containing directory's contents chain.
 */
static inline void
dir_contents_unlink(struct bmod *b)
{
	if (b->next)
		b->next->prevp = b->prevp;
	*b->prevp = b->next;
}

static OSKIT_COMDECL
bmoddir_unlink(oskit_dir_t *d, const char *name)
{
	struct bmod *dir = (void *)d;
	struct bmod *b;
	oskit_error_t rc;

	oskit_bmod_lock();

	b = (struct bmod *)bmod_lookup(dir, name);
	if (b && b->filei.ops != (void *)&bmod_dir_ops) {
		dir_contents_unlink(b);
	}

	oskit_bmod_unlock();

	if (!b)
		return OSKIT_ENOENT;

	if (b->filei.ops == (void *)&bmod_dir_ops)
		rc = OSKIT_EISDIR;
	else
		rc = 0;

	bmod_file_release((oskit_file_t *)b);
	return rc;
}

static OSKIT_COMDECL
bmoddir_rename(oskit_dir_t *old_dir, char *old_name,
	       oskit_dir_t *new_dir, char *new_name)
{
	return OSKIT_E_NOTIMPL;
}

static OSKIT_COMDECL
bmoddir_mkdir(oskit_dir_t *d, const char *name, oskit_mode_t mode)
{
	struct bmod *dir = (void *)d;
	oskit_file_t *file;
	oskit_error_t rc;

#if VERBOSITY > 1
	printf(__FUNCTION__": asked to mkdir `%s'\n", name);
#endif

	oskit_bmod_lock();

	file = bmod_lookup(dir, name);
	if (!file) {
		struct bmod *b = bmod_file_allocate(name, &bmod_dir_ops);
		if (!b)
			rc = OSKIT_ENOSPC;
		else {
			dir_contents_link(dir, b);
			rc = 0;
		}
	} else
		rc = OSKIT_EEXIST;

	oskit_bmod_unlock();
	return rc;
}


static OSKIT_COMDECL
bmoddir_rmdir(oskit_dir_t *d, const char *name)
{
	struct bmod *dir = (void *)d;
	struct bmod *b;
	oskit_error_t rc;

#if VERBOSITY > 1
	printf(__FUNCTION__": asked to rmdir `%s'\n", name);
#endif

	oskit_bmod_lock();

	b = (struct bmod *)bmod_lookup(dir, name);

	if (!b)
		rc = OSKIT_ENOENT;
	else if ((void *)b->filei.ops != &bmod_dir_ops)
		rc = OSKIT_ENOTDIR;
	else if (b->data.dir.contents)
		rc = OSKIT_ENOTEMPTY;
	else {
		dir_contents_unlink(b);
		rc = 0;
	}

	oskit_bmod_unlock();

	if (rc == 0)
		bmod_file_release((oskit_file_t *)b);
	return rc;
}


/* oskit/fs/dir.h says:
 *
 * Read one or more entries in this directory.
 * The offset of the entry to be read is specified in 'inout_ofs';
 * the caller must initially supply zero as the offset,
 * and during each successful call the offset increases
 * by an unspecified amount dependent on the file system.
 * The file system will return at least 'nentries'
 * if there are at least this many remaining in the directory;
 * however, the file system may return more entries.
 *
 * The caller must free the dirents array and the names
 * using the task allocator when they are no longer needed.
 */
/*
 * Since nentries doesn't mean much, we'll just return everything
 * we have.
 */
static OSKIT_COMDECL
bmoddir_getdirentries(oskit_dir_t *d, oskit_u32_t *inout_ofs,
		      oskit_u32_t nentries,
		      struct oskit_dirents *out_dirents)
{
	struct bmod *dir = (void *)d;
	struct bmod *b;
	struct oskit_dirent *dirents;
	int count = 0;

	/*
	 * if inout_ofs is non-zero, we assume the caller wants more entries
	 * but we've given all we have during the first call
	 */
	if (*inout_ofs != 0) {
		out_dirents->count = 0;
		out_dirents->dirents = 0;	/* in case they free it */
		return 0;
	}

	/* count files */
	for (b = dir->data.dir.contents; b; b = b->next)
		count++;


	dirents = malloc(sizeof dirents[0] * count);
	if (!dirents) {
		out_dirents->dirents = NULL;
		out_dirents->count = 0;
		return OSKIT_ENOMEM;
	}
	out_dirents->dirents = dirents;
	out_dirents->count = count;

	for (b = dir->data.dir.contents; b; dirents++, b = b->next) {
		oskit_size_t len = strlen(b->name)+1;
#if VERBOSITY > 4
		printf(__FUNCTION__": content bmod `%s'\n", b->name);
#endif
		/* copy filename in */
		dirents->name = malloc(len);
		if (dirents->name)
			memcpy(dirents->name, b->name, len);
#if VERBOSITY > 4
		printf(__FUNCTION__": copy out `%s'\n", dirents->name);
#endif

		dirents->ino = (oskit_ino_t) b;
	}
	*inout_ofs = 1;
	return 0;
}

static OSKIT_COMDECL
bmoddir_mknod(oskit_dir_t *d, char *name,
		oskit_mode_t mode, oskit_dev_t dev)
{
	return OSKIT_E_NOTIMPL;
}

static OSKIT_COMDECL
bmoddir_symlink(oskit_dir_t *d, char *link_name, char *dest_name)
{
	struct bmod *dir = (void *)d;
	oskit_file_t *file;
	oskit_error_t rc;

#if VERBOSITY > 1
	printf(__FUNCTION__": asked to create symlink `%s' -> `%s'\n",
	       link_name, dest_name);
#endif

	oskit_bmod_lock();

	file = bmod_lookup(dir, link_name);
	if (file)
		rc = OSKIT_EEXIST;
	else {
		struct bmod *b = bmod_file_allocate(link_name,
						    &bmod_symlink_ops);
		if (!b)
			rc = OSKIT_ENOSPC;
		else {
			size_t len = strlen(dest_name);
			rc = bmod_resize(b, len, 0);
			if (rc)
				bmod_file_release((oskit_file_t *)b);
			else {
				memcpy(b->data.file.data, dest_name, len);
				dir_contents_link(dir, b);
			}
		}
	}

	oskit_bmod_unlock();
	return rc;
}

/*
 * Create a new "virtual" directory object
 * referring to the same underlying directory as this one,
 * but whose logical parent directory
 * (the directory named by the '..' entry in this directory)
 * is the specified directory node handle.
 *
 * If 'parent' is NULL, lookups of the '..' entry in the
 * virtual directory object will return a reference to
 * the virtual directory object itself; as with physical
 * root directories, virtual root directories will have
 * self-referential '..' entries.
 */
static OSKIT_COMDECL
bmoddir_reparent(oskit_dir_t *d, oskit_dir_t *parent, oskit_dir_t **out_dir)
{
	return OSKIT_E_NOTIMPL;
}


static OSKIT_COMDECL
bmodfile_getfs(oskit_file_t *f, struct oskit_filesystem **out_fs)
{
	return OSKIT_E_NOTIMPL;	/* XXX implement  */
}


static struct oskit_file_ops bmod_file_ops = {
	bmod_file_query,
	bmod_file_addref,
	bmod_file_release,
	bmodfile_stat,
	bmodfile_setstat,
	no_pathconf,
	bmodfile_sync,
	bmodfile_datasync,
	bmodfile_access,
	bmodfile_readlink,
	bmodfile_open,
	bmodfile_getfs
};

static struct oskit_dir_ops bmod_dir_ops = {
	bmod_dir_query,
	bmod_file_addref,
	bmod_file_release,
	bmoddir_stat,
	no_setstat,
	no_pathconf,
	bmodfile_sync,
	bmodfile_datasync,
	bmodfile_access,
	bmodfile_readlink,
	bmoddir_open,
	bmodfile_getfs,
	bmoddir_lookup,
	bmoddir_create,
	bmoddir_link,
	bmoddir_unlink,
	bmoddir_rename,
	bmoddir_mkdir,
	bmoddir_rmdir,
	bmoddir_getdirentries,
	bmoddir_mknod,
	bmoddir_symlink,
	bmoddir_reparent
};

/*
 * Symbolic link nodes.
 */

static OSKIT_COMDECL
bmodsym_readlink(oskit_file_t *f, void *buf, oskit_u32_t len,
		 oskit_u32_t *out_actual)
{
	struct bmod *b = (void *)f;

	oskit_bmod_lock();
	if (len > b->data.file.size)
		len = b->data.file.size;
	memcpy(buf, b->data.file.data, len);
	oskit_bmod_unlock();

	*out_actual = len;
	return 0;
}

static OSKIT_COMDECL
bmodsym_stat(oskit_file_t *f, struct oskit_stat *st)
{
	struct bmod *b = (void *)f;

	memset(st, 0, sizeof *st);
	st->ino = (oskit_ino_t) b;
	st->size = b->data.file.size;
        st->mode = OSKIT_S_IFLNK | OSKIT_S_IRWXG | OSKIT_S_IRWXU | OSKIT_S_IRWXO;
	return 0;
}

static struct oskit_file_ops bmod_symlink_ops = {
	bmod_file_query,
	bmod_file_addref,
	bmod_file_release,
	bmodsym_stat,
	no_setstat,
	no_pathconf,
	bmodfile_sync,
	bmodfile_datasync,
	bmodfile_access,
	bmodsym_readlink,
	bmodfile_open,
	bmodfile_getfs
};


/*** File System interface operations ***/


struct bmod_fsys {
	oskit_filesystem_t fsysi;
	oskit_u32_t count;

	struct bmod *rootdir;
};

/*** Operations inherited from oskit_unknown interface ***/


static oskit_error_t
bmodfilesystem_query(oskit_filesystem_t *b0,
		     const oskit_iid_t *iid, void **out_ihandle)
{
	struct bmod_fsys *b = (void *)b0;
	assert(b->count);

        if (memcmp(iid, &oskit_iunknown_iid, sizeof(*iid)) == 0 ||
            memcmp(iid, &oskit_filesystem_iid, sizeof(*iid)) == 0) {
                *out_ihandle = b;
                ++b->count;
                return 0;
        }

        *out_ihandle = NULL;
        return OSKIT_E_NOINTERFACE;
}

static OSKIT_COMDECL_U
bmodfilesystem_addref(oskit_filesystem_t *b0)
{
        struct bmod_fsys *f = (void *)b0;
	assert(f->count);
	return ++f->count;
}

static oskit_u32_t
bmodfilesystem_release(oskit_filesystem_t *b0)
{
	struct bmod_fsys *b = (void *)b0;
	assert(b->count);

	if (--b->count)
		return b->count;

	bmod_file_release((oskit_file_t *)b->rootdir);
	sfree(b, sizeof *b);
	return 0;
}


/*
 * Return general information about the file system
 * on which this node resides.
 */
static OSKIT_COMDECL
bmodfilesystem_statfs(oskit_filesystem_t *f, oskit_statfs_t *out_stats)
{
	/* XXX implement this */
        out_stats->bsize = 0;       /* file system block size */
        out_stats->frsize = 0;      /* fundamental file system block size */
        out_stats->blocks = 0;      /* total blocks in fs in units of frsize */
        out_stats->bfree = 0;       /* free blocks in fs */
        out_stats->bavail = 0;      /* free blocks avail to non-superuser */
        out_stats->files = 0;       /* total file nodes in file system */
        out_stats->ffree = 0;       /* free file nodes in fs */
        out_stats->favail = 0;      /* free file nodes avail to non-superuser */
        out_stats->fsid = 0;        /* file system id */
        out_stats->flag = 0;        /* mount flags */
        out_stats->namemax = 255;     /* max bytes in a file name */
	return 0;
}

/*
 * Write all data back to permanent storage.
 * If wait is true, this call should not return
 * until all pending data have been completely written;
 * if wait is false, begin the I/O but do not wait
 * to verify that it completes successfully.
 */
static OSKIT_COMDECL
bmodfilesystem_sync(oskit_filesystem_t *f, oskit_bool_t wait)
{
	/* oh yes, sync us */
	return 0;
}

/*
 * Return a reference to the root directory of this file system.
 */
static OSKIT_COMDECL
bmodfilesystem_getroot(oskit_filesystem_t *f, oskit_dir_t **out_dir)
{
	struct bmod_fsys *b = (void *)f;

	oskit_bmod_lock();
	b->rootdir->count++;
	*out_dir = (oskit_dir_t *)b->rootdir;
	oskit_bmod_unlock();

	return 0;
}

/*
 * Update the flags of this filesystem (eg rdonly -> rdwr).
 * May return OSKIT_E_NOTIMPL if this is not
 * a typical disk-based file system.
 */
static OSKIT_COMDECL
bmodfilesystem_remount(oskit_filesystem_t *f, oskit_u32_t flags)
{
	/* no, don't remount me for now */
	return OSKIT_E_NOTIMPL;
}

/*
 * Forcibly unmount this filesystem.
 *
 * A normal unmount occurs when all references
 * to the filesystem are released.  This
 * operation forces the filesystem to be
 * unmounted, even if references still exist,
 * and is consequently unsafe.
 *
 * Subsequent attempts to use references to the
 * filesystem or to use references to files within
 * the filesystem may yield undefined results.
 */
static OSKIT_COMDECL
bmodfilesystem_unmount(oskit_filesystem_t *f)
{
	/* sure, unmount me */
	return 0;
}

static OSKIT_COMDECL
bmodfilesystem_lookupi(oskit_filesystem_t *f, oskit_ino_t ino,
		       oskit_file_t **out_file)
{
	return OSKIT_E_NOTIMPL;
}

struct oskit_filesystem_ops bmodfilesystem_ops = {
	bmodfilesystem_query,
	bmodfilesystem_addref,
	bmodfilesystem_release,
	bmodfilesystem_statfs,
	bmodfilesystem_sync,
	bmodfilesystem_getroot,
	bmodfilesystem_remount,
	bmodfilesystem_unmount,
	bmodfilesystem_lookupi,
};

/*
 * bufio operations on bmod files
 */

static inline struct bmod *
bufio2bmod(oskit_bufio_t *io)
{
	return (struct bmod *)((char *)io - offsetof(struct bmod,
						       data.file.bufioi));
}

/* express an bufio as a pointer to a oskit_file_t */
#define bufio2bmodf(x) (&bufio2bmod(x)->filei)

static OSKIT_COMDECL
afile_query(oskit_bufio_t *io,
	    const struct oskit_guid *iid, void **out_ihandle)
{
	if (!io || io->ops != &bmod_bufio_ops)
		return OSKIT_E_INVALIDARG;

	return bmod_file_query(bufio2bmodf(io), iid, out_ihandle);
}

static OSKIT_COMDECL_U
afile_addref(oskit_bufio_t *io)
{
	if (!io || io->ops != &bmod_bufio_ops)
		return OSKIT_E_INVALIDARG;

	return bmod_file_addref(bufio2bmodf(io));
}

static OSKIT_COMDECL_U
afile_release(oskit_bufio_t *io)
{
	if (!io || io->ops != &bmod_bufio_ops)
		return OSKIT_E_INVALIDARG;

	return bmod_file_release(bufio2bmodf(io));
}


static OSKIT_COMDECL
afile_read(oskit_bufio_t *io, void *buf,
	       oskit_off_t offset, oskit_size_t amount,
	       oskit_size_t *out_actual)
{
	oskit_error_t rc;
	struct bmod *file;

	DMARK;
	if (!io || io->ops != &bmod_bufio_ops)
		return OSKIT_E_INVALIDARG;

	file = bufio2bmod(io);
	DMARK;

	oskit_bmod_lock();
	DMARK;

	if (offset > file->data.file.size)
		rc = OSKIT_EINVAL;
	else {
		DMARK;
		if (file->data.file.size - offset < amount)
			amount = file->data.file.size - offset;
		DMARK;
#if VERBOSITY > 20
		printf(__FUNCTION__": copying %d from %p+%x to %p\n",
		       amount, (char *)file->data.file.data, (int)offset,
		       buf);
#endif
		memcpy(buf, (char *)file->data.file.data + offset, amount);
		DMARK;
		*out_actual = amount;
		rc = 0;
	}

	oskit_bmod_unlock();

	return rc;
}


static OSKIT_COMDECL
afile_write(oskit_bufio_t *io, const void *buf,
		oskit_off_t offset, oskit_size_t amount,
		oskit_size_t *out_actual)
{
	oskit_error_t rc;
	struct bmod *file;

	if (!io || io->ops != &bmod_bufio_ops)
		return OSKIT_E_INVALIDARG;

	file = bufio2bmod(io);

	if (offset < 0)
		return OSKIT_EINVAL;

	oskit_bmod_lock();

	if (offset > file->data.file.size)
		rc = bmod_resize(file, offset + amount, 0);
	else
		rc = 0;

	if (!rc) {
		memcpy((char *)file->data.file.data + offset, buf, amount);
		*out_actual = amount;
	}

	oskit_bmod_unlock();

	return rc;
}


static OSKIT_COMDECL
afile_get_size(oskit_bufio_t *io, oskit_off_t *out_size)
{
	struct bmod *file;

	if (!io || io->ops != &bmod_bufio_ops)
		return OSKIT_E_INVALIDARG;

	file = bufio2bmod(io);

	oskit_bmod_lock();
	*out_size = file->data.file.size;
	oskit_bmod_unlock();

	return 0;
}


static OSKIT_COMDECL
afile_set_size(oskit_bufio_t *io, oskit_off_t new_size)
{
	oskit_error_t rc;
	struct bmod *file;

	if (!io || io->ops != &bmod_bufio_ops || new_size < 0)
		return OSKIT_E_INVALIDARG;

	file = bufio2bmod(io);

	oskit_bmod_lock();
	rc = bmod_resize(file, new_size, 0);
	oskit_bmod_unlock();

	return rc;
}

static OSKIT_COMDECL
afile_map(oskit_bufio_t *io, void **out_addr,
       oskit_off_t offset, oskit_size_t count)
{
	struct bmod *file;

	if (!io || io->ops != &bmod_bufio_ops || offset < 0)
		return OSKIT_E_INVALIDARG;

	file = bufio2bmod(io);
	if (offset + count > file->data.file.size)
		return OSKIT_E_INVALIDARG;

	*out_addr = file->data.file.data + offset;
	return 0;
}

static OSKIT_COMDECL
afile_unmap(oskit_bufio_t *io, void *addr, 
	oskit_off_t offset, oskit_size_t count)
{
	return 0;		/* consider yourself unmapped */
}

static OSKIT_COMDECL
afile_wire(oskit_bufio_t *io, oskit_addr_t *out_phys_addr, 
	oskit_off_t offset, oskit_size_t count)
{
	return 0;		/* already wired to the floor */
}

static OSKIT_COMDECL
afile_unwire(oskit_bufio_t *io, oskit_addr_t phys_addr, 
		oskit_off_t offset, oskit_size_t count)
{
	return 0;
}

static OSKIT_COMDECL
afile_copy(oskit_bufio_t *io, oskit_off_t offset, 
	oskit_size_t count, oskit_bufio_t **out_io)
{
	return OSKIT_E_NOTIMPL;
}


static struct oskit_bufio_ops bmod_bufio_ops = {
    afile_query,
    afile_addref,
    afile_release,
    (void *)0,			/* slot reserved for getblocksize */
    afile_read,
    afile_write,
    afile_get_size,
    afile_set_size,
    afile_map,
    afile_unmap,
    afile_wire,
    afile_unwire,
    afile_copy
};



/*
 * initialize bmod filesystem
 */
oskit_dir_t *
oskit_bmod_init()
{
        int i;
        struct bmod *b, *dir;
        struct multiboot_module *m;

	dir = bmod_file_allocate("boot module directory", &bmod_dir_ops);
	assert(dir);
#if VERBOSITY > 20
	printf(__FUNCTION__ ": created bmod root dir %p\n", dir);
#endif

        /*
         * Copy the boot modules.
         */
        if (boot_info.flags & MULTIBOOT_MODS) {
                m = (struct multiboot_module *)phystokv(boot_info.mods_addr);
                for (i = 0; i < boot_info.mods_count; i++, m++) {
			char *name = (char *)m->string, *p;
			oskit_addr_t start, end;
			oskit_dir_t *addto;
			oskit_error_t rc;

			/*
			 * Skip any leading slashes in the name.
			 */
			while (*name == '/')
				++name;

			/*
			 * Create any intervening directories in the pathname.
			 */
			addto = (oskit_dir_t *)dir;
			while (1) {
				oskit_dir_t *sub;

				p = strchr(name, '/');
				if (!p)
					break;
				*p = '\0';

				rc = oskit_dir_lookup(addto, name,
						     (oskit_file_t **)&sub);
				if (rc == 0) {
					if (!p || sub->ops != &bmod_dir_ops) {
						panic("bmodfs: name conflict");
					}
					addto = sub;
					oskit_dir_release(sub);
				} else if (p) {
					rc = oskit_dir_mkdir(addto, name, 0777);
					assert(rc == 0);
					rc = oskit_dir_lookup(addto, name,
							     (oskit_file_t**)
							     &sub);
					assert(rc == 0);
					addto = sub;
					oskit_dir_release(sub);
				}

				name = p + 1;
			}

			/*
			 * Create the bmod file itself, in the
			 * final directory created above.
			 */
			rc = oskit_dir_create(addto, name, 1,
					     0666, (oskit_file_t **)&b);
			assert(rc == 0);
			--b->count; /* release param ref, leave ref in dir */

			/*
			 * Now put the data into the bmod.
			 */
                        start = phystokv(m->mod_start);
                        end = phystokv(m->mod_end);
#if VERBOSITY > 15
			printf(__FUNCTION__": bmod %s at %x..%x [%x%x]\n",
			       (char *)m->string, start, end,
			       ((char*)start)[0],((char*)start)[1]);
#endif

                        if ((start & (PAGE_SIZE - 1)) ||
			    (end & (PAGE_SIZE - 1))) {
				oskit_error_t rc;

				DMARK;
				rc = bmod_resize(b, round_page(end - start),
						 0);
				assert(rc == 0);

				b->data.file.size = end - start;
				memcpy(b->data.file.data,
				       (void *)start, end - start);
#if VERBOSITY > 20
				printf(__FUNCTION__": copied %d of %d bytes: %x%x\n",
				       b->data.file.size,
				       b->data.file.allocsize,
				       ((char*)b->data.file.data)[0],
				       ((char*)b->data.file.data)[1]);
#endif

#if 0 /* XXX until MultiBoot multiple-identical-modules issue is resolved */
                                if (trunc_page(m->mod_end)
                                    - round_page(m->mod_start) > 0)
                                        (phys_lmm_add
                                         (round_page(m->mod_start),
                                          (trunc_page(m->mod_end)
                                           - round_page(m->mod_start))));
#endif
                        } else {
				b->data.file.data = (void *)start;
				b->data.file.size = b->data.file.allocsize =
					end - start;
                                b->data.file.can_sfree = 0;
				DMARK;
			}
                }
        }

#if VERBOSITY > 10
	for (b = dir->data.dir.contents; b; b = b->next) {
		if (b->filei.ops == (void *)&bmod_dir_ops)
			printf(__FUNCTION__": initial bmod subdir `%s'\n",
			       b->name);
		else {
			char *data = b->data.file.data;
			assert(b->filei.ops == &bmod_file_ops);
			printf (__FUNCTION__
				": initial bmod `%s', %d bytes [%x%x]\n",
				b->name, b->data.file.size,
				data[0], data[1]);
		}
	}
#endif
	oskit_bmod_lock_init();
	
	return (oskit_dir_t *)dir;
}

oskit_error_t
oskit_bmod_file_set_contents(oskit_file_t *file,
			     void *data, oskit_off_t size,
			     oskit_off_t allocsize,
			     oskit_bool_t can_sfree,
			     oskit_bool_t inhibit_resize)
{
	struct bmod *b = (void *)file;
	oskit_error_t rc;

	if (file->ops != &bmod_file_ops || size > allocsize)
		return OSKIT_EINVAL;

	rc = bmod_resize(b, 0, 1);
	if (rc)
		return rc;

	b->data.file.data = data;
	b->data.file.size = size;
	b->data.file.allocsize = allocsize;
	b->data.file.can_sfree = can_sfree;
	b->data.file.inhibit_resize = inhibit_resize;

	return 0;
}
