/*
 * Copyright (c) 1996-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 * Default console implementation code for kernels on the x86.
 */

#include <oskit/x86/pc/base_console.h>
#include <oskit/x86/pc/com_cons.h>
#include <oskit/x86/pc/direct_cons.h>
#include <oskit/x86/pc/reset.h>
#include <oskit/x86/base_gdt.h>
#include <oskit/x86/debug_reg.h>
#include <oskit/x86/proc_reg.h>
#include <oskit/x86/base_paging.h>
#include <oskit/x86/base_vm.h>
#include <oskit/x86/base_cpu.h>
#include <oskit/c/stdio.h>
#include <oskit/c/unistd.h>
#include <oskit/c/stdlib.h>
#include <oskit/c/termios.h>
#include <oskit/tty.h>
#include <oskit/gdb.h>
#include <oskit/gdb_serial.h>
#include <oskit/base_critical.h>
#include <oskit/c/string.h>
#include <oskit/config.h>
#include <oskit/x86/pio.h>
#include <oskit/x86/proc_reg.h>
#include <oskit/x86/base_idt.h>
#include <oskit/x86/pc/pic.h>
#include <oskit/x86/pc/base_irq.h>

int cons_com_port = 1;			/* first serial (or screen) console */
int gdb_com_port = 1;			/* second serial for gdb */

/* This is set if RB_SERIAL is passed by the FreeBSD bootblocks */
int serial_console = 0;			/* set to 1 to use serial comsole, */
					/*   0 to use keyboard */
/* This is set by giving "-d" to the FreeBSD bootblocks */
int enable_gdb = 0;			/* set to 1 for kernel debugging */

/* This is set by giving "-p" to the FreeBSD bootblocks */
/* XXX it's hardwired for now (!).  The bootmanager won't
 * pass in flags it's not aware of, like -p */
int enable_gprof = 1;                   /* set to 1 for kernel profiling */

/* Kill switch using second tipline. */
int enable_killswitch = 0;

/*
 * This overrides the _exit() function in libc.
 * If the serial console (or remote GDB) is being used, it waits
 * until all the data has cleared out of the FIFOs; if the VGA 
 * display is being used (normal console), then it waits for a keypress.
 * When it is done, it calls pc_reset() to reboot the computer.
 */
static void
our_exit(int rc)
{
	extern oskit_addr_t return_address;

	printf("_exit(%d) called; rebooting...\n", rc);
	if (return_address)
		printf("Have return_address 0x%08x\n\n", return_address);

	if (enable_gdb) {
		/* Detach from the remote GDB. */
		gdb_serial_exit(rc);

#ifdef HAVE_DEBUG_REGS
		/* Turn off the debug registers. */
		set_dr7(get_dr7() & ~(DR7_G0 | DR7_G1 | DR7_G2 | DR7_G3));
#endif

		/* Wait for GDB to finish receiving exit message. */
		com_cons_flush(gdb_com_port);
	}

	/* wait for `_exit called` message */
	if (serial_console)
		com_cons_flush(cons_com_port);
	else {
		/* This is so that the user has a chance to SEE the output */
		printf("Press a key to reboot");
		getchar();
	}

	if (return_address) {
		/*
		 * The cleanup needs to be done here instead of in the
		 * returned-to code because the return address may not
		 * be accessible with our current paging and segment
		 * state.
		 * The order is important here: paging must be disabled
		 * after we reload the gdt.
		 */
		cli();
		clts();
		phys_mem_va = 0;
		linear_base_va = 0;
		base_gdt_init();
		/* Reload all since we changed linear_base_va. */
		base_cpu_load();
		paging_disable();
		((void (*)(void))return_address)();
	}
	else
		pc_reset();
}


/*
 * This function defines our kernel "console device";
 * calls to printf() will eventually call putchar().
 * Our implementation simply writes characters to the local PC display,
 * or the serial line, depending on the info from the bootblocks.
 */
int
console_putchar(int c)
{
	if (serial_console) {
		if (enable_gdb && (cons_com_port == gdb_com_port)) {
			gdb_serial_putchar(c);
		} else {
			com_cons_putchar(cons_com_port, c);
		}
	} else {
		direct_cons_putchar(c);
	}
	return (c);
}

/*
 * Here we provide the means to read a character.
 */
int
console_getchar(void)
{
	if (serial_console) {
		if (enable_gdb && (cons_com_port == gdb_com_port)) {
			return gdb_serial_getchar();
		} else {
			return com_cons_getchar(cons_com_port);
		}
	} else {
		return direct_cons_getchar();
	}
}

/*
 * While it is not necessary to override puts, this allows us to handle
 * gdb printfs much more efficiently by packaging them together.
 * If we haven't enabled the debugger, we just spit them all out.
 */
int
console_puts(const char *s)
{
	if (serial_console && enable_gdb && (cons_com_port == gdb_com_port))
		gdb_serial_puts(s);
	else {
		base_critical_enter();
		while (*s)
			console_putchar(*s++);
		console_putchar('\n');
		base_critical_leave();
	}
	return 0;
}

/*
 * This is more efficient for console output, and allows similar treatment
 * in usermode where character based output is really bad.
 */
int
console_putbytes(const char *s, int len)
{
	base_critical_enter();
	while (len) {
		console_putchar(*s++);
		len--;
	}
	base_critical_leave();
	return 0;
}

/*
 * This function parses the multiboot command line and
 * initializes the serial lines.
 */
void
base_console_init(int argc, char **argv)
{
	int i;
	char *p;

	/*
	 * XXX: "-f" is a Utah-specific hack to allow FreeBSD bootblocks 
	 * to tell the kernel to run at 115200 instead of the default 9600.
	 * Note: if -f is passed w/o -h, will use the keyboard.
	 * This is done so that "-f" can be hardcoded, and just
	 * change -h to select serial/keyboard.
	 */

	/* Initialize our configuration options from environment variables */
	if ((p = getenv("CONS_COM")) != NULL) {
		cons_com_port = atoi(p);
		serial_console = 1;
	}
	if ((p = getenv("GDB_COM")) != NULL) {
		gdb_com_port = atoi(p);
		enable_gdb = 1;
	}
	if ((p = getenv("BAUD")) != NULL) {
		base_cooked_termios.c_ispeed = atoi(p);
		base_cooked_termios.c_ospeed = atoi(p);
		base_raw_termios.c_ispeed = atoi(p);
		base_raw_termios.c_ospeed = atoi(p);
	}

	/* Deal with any boot flags that we care about */
	for (i = 0; i < argc; i++) {
		if (strcmp(argv[i], "-f") == 0) {
			base_cooked_termios.c_ispeed = B115200;
			base_cooked_termios.c_ospeed = B115200;
			base_raw_termios.c_ispeed = B115200;	/* gdb */
			base_raw_termios.c_ospeed = B115200;
		} else if (strcmp(argv[i], "-h") == 0) {
			/* RB_SERIAL console flag from the BSD boot blocks */
			serial_console = 1;
		} else if (strcmp(argv[i], "-d") == 0) {
			/* enable gdb/gdb */
			enable_gdb = 1;
		} else if (strcmp(argv[i], "-p") == 0) {
			/* enable gprof */
			enable_gprof = 1;
		} else if (strcmp(argv[i], "-k") == 0) {
			/* enable killswitch */
			enable_killswitch = 1;
		}
	}

	/* Initialize the serial console, if we are using it */
	if (serial_console)
		com_cons_init(cons_com_port, &base_cooked_termios);

	/* set libc's _exit function pointer */
	oskit_libc_exit = our_exit;

	/* Initialize GDB if it is enabled */
	if (enable_gdb) {
		extern int main(int argc, char **argv);

		/*
		 * Initialize the GDB stub to use the specified COM port.
		 */
		gdb_pc_com_init(gdb_com_port, &base_raw_termios);

#ifdef HAVE_DEBUG_REGS
		/*
		 * Set up the debug registers
		 * to catch null pointer references,
		 * and to take a breakpoint at the entrypoint to main().
		 */
		set_b0(NULL, DR7_LEN_1, DR7_RW_INST);
		set_b1(NULL, DR7_LEN_4, DR7_RW_DATA);
		set_b2((unsigned)main, DR7_LEN_1, DR7_RW_INST);
#else
		/*
		 * Take an immediate breakpoint trap here
		 * so that the debugger can take control from the start.
		 */
		asm("int $3");	/* XXX */
#endif

		/*
		 * The Intel Pentium manual recommends
		 * executing an LGDT instruction
		 * after modifying breakpoint registers,
		 * and experience shows that this is necessary.
		 */
		base_gdt_load();
	}

	/*
	 * This gross hack, allows the second tipline (gdb line) to be
	 * used as a kill switch to force the kernel into a panic. Very
	 * useful when the kernel is looping someplace and interrupts are
	 * still enabled. Simply cat some characters into the special
	 * device file (/dev/tip/foo-gdb).
	 */
	if (enable_killswitch && !enable_gdb &&
	    gdb_com_port != cons_com_port) {
		int com_irq = gdb_com_port & 1 ? 4 : 3;
		extern void com_kill_intr();

		/*
		 * Initialize the serial port with base_raw_termios.
		 */
		com_cons_init(gdb_com_port, &base_raw_termios);

		/*
		 * Drain it.
		 */
		while (com_cons_trygetchar(gdb_com_port) != -1)
			;

		/* Hook the COM port's hardware interrupt. */
		fill_irq_gate(com_irq,
			      (unsigned)com_kill_intr, KERNEL_CS, ACC_PL_K);

		/* Enable the COM port interrupt.  */
		com_cons_enable_receive_interrupt(gdb_com_port);
		pic_enable_irq(com_irq);
	}
}
