/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/*
 * This is a test program for the libc socket implementation based on
 * the TCP/IP stack.
 *
 * It first does two fingers and then waits for incoming date requests.
 */
#if OSKIT
#include <oskit/dev/dev.h>
#include <oskit/dev/clock.h>
#include <oskit/dev/linux.h>
#include <oskit/dev/freebsd.h>
#include <oskit/dev/ethernet.h>
#include <oskit/net/freebsd.h>
#include <oskit/fs/bmodfs.h>
#include <oskit/c/fs.h>
#include "bootp.h"

char GATEWAY[20];
char NETMASK[20];

#define IFNAME		"de0"

/* MST: Salt Lake City, UT */
long secondswest = 6 * 60 * 60;
#define LOCAL_TO_GMT(t) (t)->tv_sec += secondswest

#endif /* OSKIT */

char IPADDR[256];

#include <sys/types.h>
#include <sys/time.h>
#include <time.h>
#include <errno.h>
#include <sys/socket.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

/*
 * this is a simple finger client, programmed to the libbsdnet socket interface
 */
void
fingerclient(char *name, char *host)
{
	int	so;
	int 	namelen, retval, msg_len;
	struct  sockaddr_in addr;
	char    message[256];

	if ((so = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
		perror("socket");
		return;
	}

	/* fill in the sin_addr structure */
	memset(&addr, 0, namelen = sizeof(addr));
	addr.sin_addr.s_addr = inet_addr(host);
	addr.sin_family = AF_INET;
	addr.sin_port = htons(79);

	if ((-1 == connect(so, (struct sockaddr *)&addr, sizeof(addr)))) 
	{
		perror("connect");
		return;
	}

	sprintf(message, "%s\r\n", name);
	retval = write(so, message, msg_len = strlen(message));
	if (retval < 0) {
		perror("write");
		return;
	} else {
		printf("write %d out of %d bytes\n", retval, msg_len);
	}

	do {
		retval = read(so, message, sizeof(message));
		if (retval >= 0) {
			int i;
			for (i = 0; i < retval; i++)
				putchar(message[i]); 
		} else {
			perror("read");
			return;
		}
	} while (retval > 0);

	if ((-1 == close(so))) 
		perror("close");
}

volatile int stop = 0;

/* ARGSUSED */
void timeout(int signal)
{
	stop = 1;
}

/*
 * this is a simple server responding to requests on a given port (daytime)
 */
void
dateserver(short port, int seconds)
{
	int 	so, newso;
	int 	namelen;
	struct  sockaddr_in addr;
	struct  itimerval expire;
	struct  timeval time;

	if ((so = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
		perror("socket");
		return;
	}

	/* fill in the sin_addr structure */
	memset(&addr, 0, namelen = sizeof(addr));
	addr.sin_addr.s_addr = htonl(INADDR_ANY);
	addr.sin_family = AF_INET;
	addr.sin_port = htons(port);

	if ((-1 == bind(so, (struct sockaddr *)&addr, sizeof(addr)))) {
		perror("bind");
		return;
	}

	if ((-1 == listen(so, 5))) {
		perror("listen");
		return;
	}

	signal(SIGALRM, timeout);
	memset(&expire, 0, sizeof expire);
	expire.it_value.tv_sec = seconds;
	setitimer(ITIMER_REAL, &expire, NULL);

	for (;!stop;) {
		char * time_message;
		int  msg_len, written;

		if ((newso = accept(so, (struct sockaddr *)&addr, 
				&namelen)) == -1) 
		{
			perror("accept");
			return;
		}

		gettimeofday(&time, 0);
		time_message = asctime(localtime(&time.tv_sec));	
		msg_len = strlen(time_message);

		written = write(newso, time_message, msg_len);
		if (written < 0)
		{
			perror("write");
			return;
		} else {
			printf("write %d out of %d bytes\n", written, msg_len);
		}

		if ((-1 == close(newso))) {
			perror("close");
			return;
		}
	}
	printf("bailing out of loop...\n");
	close(so);
}

/*
 * the initialization follows the ping_reply example...
 */
int
main(int argc, char **argv)
{
	short port = 13;

#if OSKIT
	int err, ndev;
	oskit_etherdev_t	**etherdev;
	oskit_socket_factory_t	*fsc;
	oskit_clock_t	*clock;
	oskit_timespec_t time;
	struct oskit_freebsd_net_ether_if *eif;

	oskit_init_libc();
	fs_init(oskit_bmod_init());

        printf("Initializing devices...\n");
	oskit_dev_init();
	oskit_linux_init_net();

	printf("Probing devices...\n");
	oskit_dev_probe();
	oskit_dump_devices();

        /*
         * Find all the Ethernet device nodes.
         */
        ndev = osenv_device_lookup(&oskit_etherdev_iid, (void***)&etherdev);
        if (ndev <= 0)
                panic("no Ethernet adaptors found!");

	/* get ip addr via bootp or console */
	get_ipinfo(etherdev[0], IPADDR, GATEWAY, NETMASK, &secondswest);

	err = oskit_freebsd_net_init(&fsc);	/* initialize network code */
	assert(!err);

	oskit_register(&oskit_socket_factory_iid, fsc);

	/*
	 * for now, we just go for the first device...
	 */
    	err = oskit_freebsd_net_open_ether_if(etherdev[0], &eif);
	assert(!err);

	err = oskit_freebsd_net_ifconfig(eif, IFNAME, IPADDR, NETMASK);
	assert(!err);
	if ((err = oskit_freebsd_net_add_default_route(GATEWAY))) {
		printf("couldn't add default route (%d)\n", err);
	}
	clock = oskit_clock_init();
	oskit_rtc_get(&time);
	LOCAL_TO_GMT(&time);
	oskit_clock_settime(clock, &time);

	set_system_clock(clock);

#endif /* OSKIT */

	/*
	 * if you want to test pingreply, choose the first piece of code
	 */
	fingerclient("oskit", "155.99.212.1"); /* fast.cs.utah.edu */
	fingerclient("ftp", "155.99.212.1"); 	/* fast */
	printf("responding to date requests for 1 minute\n");

#if !OSKIT
	gethostname(IPADDR, sizeof IPADDR);
	port = 3000;
#endif /* !OSKIT */

	printf("type 'telnet %s %d' to test\n", IPADDR, port);
	dateserver(port, 60);

#if OSKIT
	/* close etherdev and release net_io devices */
	oskit_freebsd_net_close_ether_if(eif);
#endif	/* OSKIT */

	exit(0);
}
