/*
 * Copyright (c) 1996, 1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 * Tests the NetBSD filesystem code via the POSIX interfaces.
 *
 * This does the same things as ../netbsd_fs_com.c.
 *
 * Compare also unsupported/start_fs.c
 *
 * You must have a valid filesystem with a dev/sd0g node in it for the
 * test to run.  (You can redefine PARTITION_NAME).
 */
#define DISK_NAME	"sd0"
#define PARTITION_NAME	"g"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <fcntl.h>
#include <stdlib.h>

#include <oskit/dev/dev.h>
#include <oskit/dev/linux.h>
#include <oskit/principal.h>
#include <oskit/boolean.h>
#include <oskit/diskpart/diskpart.h>
#include <oskit/c/fs.h>

oskit_principal_t *cur_principal; /* identity of current client process */

oskit_error_t oskit_get_call_context(const struct oskit_guid *iid, void **out_if)
{
    if (memcmp(iid, &oskit_iunknown_iid, sizeof(*iid)) == 0 ||    
	memcmp(iid, &oskit_principal_iid, sizeof(*iid)) == 0) {
	*out_if = cur_principal;
	oskit_principal_addref(cur_principal);
	return 0;
    }
    
    *out_if = 0;
    return OSKIT_E_NOINTERFACE;
}
	

static unsigned int buf[1024];

/* Partition array, filled in by the diskpart library. */
#define MAX_PARTS 30
static diskpart_t part_array[MAX_PARTS];

int main(int argc, char **argv)
{
    oskit_blkio_t *disk;
    oskit_blkio_t *part;
    oskit_filesystem_t *fs;
    oskit_dir_t *root;
    oskit_identity_t id;
    struct stat sb;
    oskit_u32_t actual;
    oskit_error_t rc;
    char *diskname = DISK_NAME;
    char *partname = PARTITION_NAME;
    int numparts;
    int 	fd, i;

#define CHECKRC(x)        { if ((rc = (x)) != 0) { printf("%s\n", #x); \
	printf(__FILE__":%d in "__FUNCTION__"\n", __LINE__); exit(rc); } }

#define CHECK(x)        { if (-1 == (x)) { perror(#x); \
	printf(__FILE__":%d in "__FUNCTION__"\n", __LINE__); exit(-1); } }

    oskit_init_libc();

    /* initialize device drivers */
    printf(">>>Initializing device drivers\n");
    oskit_dev_init();
    oskit_linux_init_devs();
    oskit_dump_drivers();
    printf(">>>Probing devices");
    oskit_dev_probe();
    oskit_dump_devices();

    /* initialize the file system code */
    printf(">>>Initializing the file system\n");
    CHECKRC(fs_netbsd_init())

    /* establish client identity */
    printf(">>>Establishing client identity\n");
    id.uid = 0;
    id.gid = 0;
    id.ngroups = 0;
    id.groups = 0;
    CHECKRC(oskit_principal_create(&id, &cur_principal))

    printf(">>>Opening the disk %s\n", diskname);
    CHECKRC(oskit_linux_block_open(diskname,
				   OSKIT_DEV_OPEN_READ|OSKIT_DEV_OPEN_WRITE,
				   &disk));

    printf(">>>Reading partition table and looking for partition %s\n",
	   partname);
    numparts = diskpart_blkio_get_partition(disk, part_array, MAX_PARTS);
    if (numparts == 0)
    {
	printf("No partitions found\n");
	exit(1);
    }
    if (diskpart_blkio_lookup_bsd_string(part_array, partname,
					 disk, &part) == 0)
    {
	printf("Couldn't find partition %s\n", partname);
	exit(1);
    }

    /* mount a filesystem */    
    printf(">>>Mounting %s\n", PARTITION_NAME);
    CHECKRC(fs_netbsd_mount(part, OSKIT_FS_RDONLY, &fs));

    /* get a handle to the filesystem root dir */
    CHECKRC(oskit_filesystem_getroot(fs, &root))

    /* remount the file system read-write */
    CHECKRC(oskit_filesystem_remount(fs, 0))

    /* now, initialize the posix layer and use it */
    CHECKRC(fs_init(root))

    /* fs_init grabbed a reference to root so we can release ours */
    oskit_dir_release(root);

    /* create a directory */
    printf(">>>Creating /a\n");
    CHECK(mkdir("/a", 0755))

    printf(">>>Changing into /a\n");
    CHECK(chdir("/a"))

    /* create a subdirectory */
    printf(">>>Creating /a/b\n");
    CHECK(mkdir("b", 0700))

    printf(">>>Changing into /a/b\n");
    CHECK(chdir("b"))

    /* create a file in the subdirectory */
    printf(">>>Creating foo\n");
    CHECK(fd = open("foo", O_CREAT | O_RDWR, 0600)) 

    /* fill the buffer with a nonrepeating integer sequence */
    for (i = 0; i < sizeof(buf)/sizeof(unsigned int); i++)
	    buf[i] = i; 

    /* write the sequence */
    printf(">>>Writing to foo\n");
    CHECK(actual = write(fd, buf, sizeof buf))

    if (actual != sizeof(buf))
    {
	printf("only wrote %d bytes of %d total\n",
	       actual, sizeof(buf));
	exit(1);
    }

    /* reopen the file */
    printf(">>>Reopening foo\n");
    CHECK(close(fd))

    CHECK(fd = open("foo", O_RDWR)) 
    memset(buf, 0, sizeof(buf));

    /* read the file contents */
    printf(">>>Reading foo\n");
    CHECK(actual = read(fd, buf, sizeof buf))

    if (actual != sizeof(buf))
    {
	printf("only read %d bytes of %d total\n", actual, sizeof(buf));
	exit(1);
    }

    CHECK(close(fd))
    
    /* check the file contents */
    printf(">>>Checking foo's contents\n");
    for (i = 0; i < sizeof(buf)/sizeof(unsigned int); i++)
    {
	if (buf[i] != i)
	{
	    printf("word %d of file was corrupted\n", i);
	    exit(1);
	}
    }

    /* lots of other tests */
    printf(">>>Linking bar to foo\n");
    CHECK(link("foo", "bar"))

    printf(">>>Renaming bar to rab\n");
    CHECK(rename("bar", "rab"))

    printf(">>>Stat'ing foo\n");
    memset(&sb, 0, sizeof(sb));
    CHECK(stat("foo", &sb))

    printf(">>>Checking foo's mode and link count\n");
    if (!S_ISREG(sb.st_mode) || ((sb.st_mode & 0777) != 0600))
    {
	printf("foo has the wrong mode (0%o)\n", sb.st_mode);
	exit(1);
    }

    if ((sb.st_nlink != 2))
    {
	printf("foo does not have 2 links?, nlink=%d\n", sb.st_nlink);
	exit(1);
    }

    printf(">>>Stat'ing .\n");
    memset(&sb, 0, sizeof(sb));
    CHECK(stat(".", &sb))

    printf(">>>Checking .'s mode and link count\n");
    if (!S_ISDIR(sb.st_mode) || ((sb.st_mode & 0777) != 0700))
    {
	printf("a/b has the wrong mode (0%o)\n", sb.st_mode);
	exit(1);
    }

    if ((sb.st_nlink != 2))
    {
	printf("a/b does not have 2 links?, nlink=%d\n", sb.st_nlink);
	exit(1);
    }

    printf(">>>Stat'ing ..\n");
    memset(&sb, 0, sizeof(sb));
    CHECK(stat("..", &sb))

    printf(">>>Checking ..'s mode and link count\n");
    if (!S_ISDIR(sb.st_mode) || ((sb.st_mode & 0777) != 0755))
    {
	printf("a has the wrong mode (0%o)\n", sb.st_mode);
	exit(1);
    }

    if ((sb.st_nlink != 3))
    {
	printf("a does not have 3 links?, nlink=%d\n", sb.st_nlink);
	exit(1);
    }

    printf(">>>Changing foo's mode\n");
    CHECK(chmod("foo", 0666))

    printf(">>>Changing foo's user and group\n");
    CHECK(chown("foo", 5458, 101))

    printf(">>>Stat'ing foo\n");
    memset(&sb, 0, sizeof(sb));
    CHECK(stat("foo", &sb))

    printf(">>>Checking foo's mode and ownership\n");
    if (!S_ISREG(sb.st_mode) || ((sb.st_mode & 0777) != 0666))
    {
	printf("chmod(foo) didn't work properly\n");
	exit(1);
    }

    if (sb.st_uid != 5458)
    {
	printf("chown(foo) didn't work properly\n");
	exit(1);
    }

    if (sb.st_gid != 101)
    {
	printf("chown(foo) didn't work properly\n");
	exit(1);
    }

    printf(">>>Unlinking foo\n");
    CHECK(unlink("foo"))

    printf(">>>Unlinking rab\n");
    CHECK(unlink("rab"))

    printf(">>>Changing cwd to ..\n");
    CHECK(chdir(".."))

    printf(">>>Removing b\n");
    CHECK(rmdir("b"))

    printf(">>>Changing cwd to ..\n");
    CHECK(chdir(".."))

    printf(">>>Removing a\n");
    CHECK(rmdir("a"))

    printf(">>>Syncing and unmounting\n");
    /* this releases the reference to the root and current dir */
    fs_release();

    /* sync and unmount */
    oskit_filesystem_release(fs); 

    printf(">>>Exiting\n");
    exit(0);
}

