/*
 * Copyright (c) 1997-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 * This is the Simple Packet Filter interface.
 * It filters the input from a netio object and passes it to another.
 */

#include <oskit/io/netio.h>
#include <oskit/c/string.h>
#include <oskit/c/stdlib.h>
#include <oskit/c/malloc.h>
#include <oskit/c/assert.h>

#if 0
/*
 * Example use:
 */
	/*
	 * Preconditions:
	 *
	 * filtered_netio is an initialized netio object that will
	 * receive the filtered packets.
	 *
	 * Variables:
	 *
	 * fnetio_out is used to transmit packets
	 * fnetio_in is the netio that receives raw incoming packets
	 *
	 * device_push_netio *could* be used to bypass the outgoing PF
	 */

	/* incoming fanout */
	fnetio_in = oskit_netio_fanout_create();

	/* outgoing fanout */
	fnetio_out = oskit_netio_fanout_create();

	/*
	 * Open the ethernet device, with incoming_netio as         
	 * the receiver; get the device send netio device_push_netio
	 * it initializes device_push_netio
	 */
	err = oskit_etherdev_open(etherdev, FLAGS, fnetio_in,
	        &device_push_netio);
	assert(!err);

	/* add device netio to outgoing fanout */
	oskit_netio_fanout_add_listener(fnetio_out, device_push_netio);

	/* 
	 * fanout keeps a ref, so we can release ours, if the fanout is
	 * destroyed later down the road, it'll release the ref to the
	 * device
	 */
	oskit_netio_release(device_push_netio);

	/* 
	 * Create a SPF that receives packets from push_netio and
	 * forwards them to filtered_netio if they match the params
	 * it initializes push_netio
	 */
	oskit_netio_spf_create(..., filtered_netio, &push_netio);
   
	/* make sure pf receives incoming packets */
	oskit_netio_fanout_add_listener(fnetio_in, push_netio);     

	/*
	 * We want to get filtered packets that are being transmitted 
	 * as well as received:
	 */
	oskit_netio_fanout_add_listener(fnetio_out, push_netio);   
	 
	/*
	 * Okay, now filtered_netio is receiving all packets that
	 * match the filter, for both incoming and outgoing packets.
	 * device_push_netio is used to transmit packets
	 */

/*
 * End example
 */
#endif


struct spfdata {
	oskit_netio_t ioi;		/* COM interface */
	unsigned count;			/* reference count */
	oskit_netio_t *netio;
	char *filter;
	int len;
	int offset;
};

/*
 * Query a net I/O object for its interfaces.
 * This is extremely easy because we only export one interface
 * (plus its base type, IUnknown).
 */
static OSKIT_COMDECL
net_query(oskit_netio_t *io, const oskit_iid_t *iid, void **out_ihandle)
{
	struct spfdata *po = (struct spfdata *)io;

	assert (po != NULL);
	assert (po->count != 0);

	if (memcmp(iid, &oskit_iunknown_iid, sizeof(*iid)) == 0 ||
	    memcmp(iid, &oskit_netio_iid, sizeof(*iid)) == 0) {
		*out_ihandle = &po->ioi;
		++po->count;
		return 0;
	}

	*out_ihandle = NULL;
	return OSKIT_E_NOINTERFACE;
}


/*
 * Clone a reference to a netio object
 */
static OSKIT_COMDECL_U
net_addref(oskit_netio_t *io)
{
	struct spfdata *po = (struct spfdata *)io;

	assert (po != NULL);
	assert (po->count != 0);

	return ++po->count;
}


/*
 * Close ("release") a netio object.
 */
static OSKIT_COMDECL_U
net_release(oskit_netio_t *io)
{
	struct spfdata *po = (struct spfdata *)io;
	unsigned newcount;

	assert (po != NULL);
	assert (po->count != 0);

	newcount = po->count - 1;
	if (newcount == 0) {
		oskit_netio_release(po->netio);
		free(po->filter);
		free(po);
		return 0;
	}

	return po->count = newcount;
}

/*
 * This is the function that does the filtering.
 */
static OSKIT_COMDECL
net_push(oskit_netio_t *ioi, oskit_bufio_t *b, oskit_size_t pkt_size)
{
	struct spfdata *data = (struct spfdata *)ioi;
	char *pkt;
	int err;

	assert (data != NULL);
	assert (data->count != 0);

	/* if packet isn't big enough... */
	if (data->len + data->offset > pkt_size) {
		return 0;
	}

	/* check to see if the bufio matches the filter */
	err = oskit_bufio_map(b, (void **)&pkt, data->offset, data->len);
	assert(!err);

	if (!memcmp(data->filter, pkt, data->len)) {
		/* matches, call the netio's push routine */
		oskit_netio_push(data->netio, b, pkt_size);
	}
	/* if doesn't match, just return */

	err = oskit_bufio_unmap(b, (void *)pkt, data->offset, data->len);
	assert(!err);

	return 0;
}

/*
 * spf vtable
 */
static 
struct oskit_netio_ops spfdata_ops = {
	net_query,
	net_addref, 
	net_release,
	net_push
};

/*
 * This function creates a SPF netio.
 */
int
oskit_netio_spf_create(char *filter, int len, int offset,
		oskit_netio_t *out_netio, oskit_netio_t **push_netio)
{
	struct spfdata *data;

	data = malloc(sizeof(*data));
	if (!data)
		return OSKIT_ENOMEM;

	data->netio = out_netio;
	oskit_netio_addref(data->netio);

	data->filter = malloc(len);
	if (!data->filter) {
		free(data);
		return OSKIT_ENOMEM;
	}

	memcpy(data->filter, filter, data->len = len);
	data->offset = offset;

	data->ioi.ops = &spfdata_ops;
	data->count = 1;
	*push_netio = &data->ioi;
	return 0;
}
