/*
 * Copyright (c) 1997,1999 The University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * Contributed by the Computer Security Research division,
 * INFOSEC Research and Technology Office, NSA.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

#include <oskit/c/assert.h>
#include <oskit/c/stddef.h>
#include <oskit/c/malloc.h>
#include <oskit/c/stdio.h>
#include "sfs.h"

#define INOTAB_SIZE 23

static unsigned int inohash(hashtab_key_t key)
{
    unsigned int h;
    
    h = (unsigned int) key;
    return h % INOTAB_SIZE;
}

static int inocmp(hashtab_key_t key1, hashtab_key_t key2)
{
    return !(key1 == key2);
}

static struct oskit_filesystem_ops fs_ops;
static struct oskit_comsid_ops sid_ops;

oskit_error_t sfilesystem_create(oskit_dir_t *root, 
				security_id_t sid,
				oskit_filepsid_t *psid,
				oskit_avc_t *avc,
				struct sfilesystem **out_sfs)
{
    struct sfilesystem *sfs;
    oskit_error_t rc;

    sfs = malloc(sizeof(struct sfilesystem));
    if (!sfs)
	    return OSKIT_ENOMEM;
    
    sfs->fsi.ops = &fs_ops;
    sfs->sidi.ops = &sid_ops;
    sfs->count = 1;
    sfs->sid = sid;
    sfs->root = root;
    oskit_dir_addref(root);

    sfs->ino2sf = hashtab_create(inohash, inocmp, INOTAB_SIZE);
    if (!sfs->ino2sf)
    {
	oskit_dir_release(root);
	free(sfs);
	return OSKIT_ENOMEM;	
    }

    sfs->fs = NULL;
    rc = oskit_dir_getfs(root, &sfs->fs);
    if (rc)
    {
	/*
	 * Nonfatal, but not desirable.
	 */
	printf("sfs_create():  warning!  no underlying fs object\n");
    }

    sfs->psid = psid;
    oskit_filepsid_addref(psid);

    sfs->avc = avc;
    oskit_avc_addref(avc);

    *out_sfs = sfs;
    
    return 0;
}



/*
 * oskit_filesystem methods
 */

static OSKIT_COMDECL filesystem_query(oskit_filesystem_t *f,
				     const struct oskit_guid *iid,
				     void **out_ihandle)
{
    struct sfilesystem *sfs;

    sfs = (struct sfilesystem*)f;
    assert(sfs);
    assert(sfs->count);
    
    if (memcmp(iid, &oskit_iunknown_iid, sizeof(*iid)) == 0 ||
	memcmp(iid, &oskit_filesystem_iid, sizeof(*iid)) == 0) {
	*out_ihandle = &sfs->fsi;
	++sfs->count;
	return 0;
    }

    if (memcmp(iid, &oskit_comsid_iid, sizeof(*iid)) == 0) 
    {
	*out_ihandle = &sfs->sidi;
	++sfs->count;
	return 0;
    }

    *out_ihandle = NULL;
    return OSKIT_E_NOINTERFACE;    
}


static OSKIT_COMDECL_U filesystem_addref(oskit_filesystem_t *f)
{
    struct sfilesystem *sfs;

    sfs = (struct sfilesystem*)f;
    assert(sfs);
    assert(sfs->count);

    return ++sfs->count;
}


static OSKIT_COMDECL_U filesystem_release(oskit_filesystem_t *f)
{
    struct sfilesystem *sfs;
    unsigned newcount;
    
    sfs = (struct sfilesystem*)f; 
    assert(sfs);
    assert(sfs->count);

    newcount = --sfs->count;
    if (newcount == 0)
    {
	oskit_dir_release(sfs->root);
	hashtab_destroy(sfs->ino2sf);
	oskit_filepsid_release(sfs->psid);
	if (sfs->fs)
		oskit_filesystem_release(sfs->fs);
	free(sfs);
    }

    return newcount;
}


static OSKIT_COMDECL filesystem_statfs(oskit_filesystem_t *f,
				      oskit_statfs_t *out_stats)
{
    struct sfilesystem *sfs = (struct sfilesystem *) f; 
    security_id_t csid;
    oskit_error_t rc;

    
    if (!sfs || !sfs->count || !sfs->fs)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECK(sfs->avc,SUBJECT, csid, FILESYSTEM, sfs->sid, GETATTR,&rc);
    if (rc)
	    return rc;

    return oskit_filesystem_statfs(sfs->fs, out_stats);
}


static OSKIT_COMDECL filesystem_sync(oskit_filesystem_t *f,
				    oskit_bool_t wait)
{
    struct sfilesystem *sfs = (struct sfilesystem *) f; 
    security_id_t csid;
    oskit_error_t rc;

    
    if (!sfs || !sfs->count || !sfs->fs)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECK(sfs->avc,SUBJECT, csid, FILESYSTEM, sfs->sid, SYNC,&rc);
    if (rc)
	    return rc;

    return oskit_filesystem_sync(sfs->fs, wait);
}


static OSKIT_COMDECL filesystem_getroot(oskit_filesystem_t *f,
				       struct oskit_dir **out_dir)
{
    struct sfilesystem *sfs = (struct sfilesystem *) f; 
    struct sfiledir *sdir;
    security_id_t csid;
    oskit_error_t rc;

    
    if (!sfs || !sfs->count || !sfs->fs)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECK(sfs->avc,SUBJECT, csid, FILESYSTEM, sfs->sid, GETROOT,&rc);
    if (rc)
	    return rc;
    
    rc = sfiledir_create(sfs, (oskit_file_t*)sfs->root, &sdir);
    if (rc)
	    return rc;

    *out_dir = (oskit_dir_t*)&sdir->filei;
    return 0;
}


static OSKIT_COMDECL filesystem_remount(oskit_filesystem_t *f,
				       oskit_u32_t flags)
{
    struct sfilesystem *sfs = (struct sfilesystem *) f; 
    security_id_t csid;
    oskit_error_t rc;

    
    if (!sfs || !sfs->count || !sfs->fs)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECK(sfs->avc,SUBJECT, csid, FILESYSTEM, sfs->sid, REMOUNT,&rc);
    if (rc)
	    return rc;

    return oskit_filesystem_remount(sfs->fs, flags);
}


static OSKIT_COMDECL filesystem_unmount(oskit_filesystem_t *f)
{
    struct sfilesystem *sfs = (struct sfilesystem *) f; 
    security_id_t csid;
    oskit_error_t rc;

    
    if (!sfs || !sfs->count || !sfs->fs)
	    return OSKIT_E_INVALIDARG;

    CSID(&csid);
    COMCHECK(sfs->avc,SUBJECT, csid, FILESYSTEM, sfs->sid, UNMOUNT,&rc);
    if (rc)
	    return rc;

    return oskit_filesystem_unmount(sfs->fs);
}


static OSKIT_COMDECL filesystem_lookupi(oskit_filesystem_t *f,
					oskit_ino_t ino,
					oskit_file_t **out_file)
{
	/* XXX should do something like the above. */
	return OSKIT_E_NOTIMPL;
}


static struct oskit_filesystem_ops fs_ops = {
    filesystem_query,
    filesystem_addref,
    filesystem_release,
    filesystem_statfs,
    filesystem_sync,
    filesystem_getroot,
    filesystem_remount,
    filesystem_unmount,
    filesystem_lookupi,
};



/*
 * oskit_comsid methods
 */

static OSKIT_COMDECL sid_query(oskit_comsid_t *s,
			      const struct oskit_guid *iid,
			      void **out_ihandle)
{
    struct sfilesystem *sfs;

    sfs = (struct sfilesystem*) ((char *) s - 
				 offsetof(struct sfilesystem, sidi));
    assert(sfs);
    assert(sfs->count);

    return oskit_filesystem_query(&sfs->fsi, iid, out_ihandle);
}


static OSKIT_COMDECL_U sid_addref(oskit_comsid_t *s)
{
    struct sfilesystem *sfs;

    sfs = (struct sfilesystem*) ((char *) s - 
				 offsetof(struct sfilesystem, sidi));
    assert(sfs);
    assert(sfs->count);

    return oskit_filesystem_addref(&sfs->fsi);
}


static OSKIT_COMDECL_U sid_release(oskit_comsid_t *s)
{
    struct sfilesystem *sfs;

    sfs = (struct sfilesystem*) ((char *) s - 
				 offsetof(struct sfilesystem, sidi));
    assert(sfs);
    assert(sfs->count);

    return oskit_filesystem_release(&sfs->fsi);
}


static OSKIT_COMDECL sid_get(oskit_comsid_t *s,
			    security_class_t *outclass,
			    security_id_t *outsid)
{
    struct sfilesystem *sfs;

    sfs = (struct sfilesystem*) ((char *) s - 
				 offsetof(struct sfilesystem, sidi));
    assert(sfs);
    assert(sfs->count);

    *outclass = SECCLASS_FILESYSTEM;
    *outsid = sfs->sid;
    return 0;
}


static OSKIT_COMDECL sid_set(oskit_comsid_t *s,
			    security_id_t newsid)
{
    struct sfilesystem *sfs;

    sfs = (struct sfilesystem*) ((char *) s - 
				 offsetof(struct sfilesystem, sidi));
    assert(sfs);
    assert(sfs->count);

    return OSKIT_E_NOTIMPL;
}


static struct oskit_comsid_ops sid_ops = {
    sid_query,
    sid_addref,
    sid_release,
    sid_get,
    sid_set
};

