/*
 * Copyright (c) 1996-1998 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */
/*
 * Generic COM interface registration module.
 * Basically maintains a list of COM interface pointers
 * for a particular interface GUID (IID).
 * Given an IID, you can find all the registered COM interfaces with that IID.
 * Interfaces will be returned in the order in which they were registered.
 * It's harmless to register an interface multiple times;
 * only a single entry in the table will be retained.
 * Currently just does everything with simple lists;
 * if we end up having lots of objects, we may need smarter algorithms.
 */

#include <string.h>
#include <malloc.h>

#include <oskit/com.h>
#include <oskit/debug.h>

/* One of these nodes represents each registered COM interface (object) */
struct objnode {
	struct objnode *next;
	oskit_iunknown_t *intf;
};

/* We keep one iidnode for each unique IID we see */
struct iidnode {
	struct iidnode *next;
	oskit_guid_t iid;
	struct objnode *objs;
	int objcount;
};

static struct iidnode *iids;


oskit_error_t
oskit_register(const struct oskit_guid *iid, void *interface)
{
	oskit_iunknown_t *iu = (oskit_iunknown_t*)interface;
	struct iidnode *in;
	struct objnode *on, **onp;

	/* Find or create the appropriate iidnode */
	for (in = iids; ; in = in->next) {
		if (in == NULL) {
			in = malloc(sizeof(*in));
			if (in == NULL)
				return OSKIT_E_OUTOFMEMORY;
			in->iid = *iid;
			in->objs = NULL;
			in->objcount = 0;
			in->next = iids;
			iids = in;
			break;
		}
		if (memcmp(&in->iid, iid, sizeof(*iid)) == 0)
			break;
	}

	/* Make sure this interface isn't already registered */
	for (onp = &in->objs; *onp; onp = &(*onp)->next) {
		if ((*onp)->intf == interface)
			return 0;
	}

	/* Create a new objnode for this interface */
	on = malloc(sizeof(*on));
	if (on == NULL)
		return OSKIT_E_OUTOFMEMORY;
	on->next = NULL;
	on->intf = iu;	oskit_iunknown_addref(iu);
	*onp = on;
	in->objcount++;

	return 0;
}

oskit_error_t
oskit_unregister(const struct oskit_guid *iid, void *interface)
{
	struct iidnode *in;
	struct objnode *on, **onp;

	/* Find the appropriate iidnode */
	for (in = iids; ; in = in->next) {
		if (in == NULL)
			return OSKIT_E_INVALIDARG;
		if (memcmp(&in->iid, iid, sizeof(*iid)) == 0)
			break;
	}

	/* Find and remove the objnode */
	for (onp = &in->objs; ; onp = &on->next) {
		on = *onp;
		if (on == NULL)
			return OSKIT_E_INVALIDARG;
		if (on->intf == interface)
			break;
	}
	*onp = on->next;
	oskit_iunknown_release(on->intf);
	free(on);
	in->objcount--;

	return 0;
}

oskit_error_t
oskit_lookup(const oskit_guid_t *iid,
	    void ***out_interface_array)
{
	struct iidnode *in;
	struct objnode *on;
	void **arr;
	int i;

	/* Find the appropriate iidnode */
	for (in = iids; ; in = in->next) {
		if (in == NULL) {
			*out_interface_array = NULL;
			return 0;
		}
		if (memcmp(&in->iid, iid, sizeof(*iid)) == 0)
			break;
	}
	if (in->objcount == 0) {
		*out_interface_array = NULL;
		return 0;
	}

	/* Allocate an array to hold the interface pointers to return */
	arr = malloc(sizeof(*arr)*in->objcount);
	if (arr == NULL)
		return OSKIT_E_OUTOFMEMORY;

	/* Fill it in */
	for (i = 0, on = in->objs; i < in->objcount; i++, on = on->next) {
		assert(on != NULL);
		arr[i] = on->intf;
		oskit_iunknown_addref(on->intf);
	}
	assert(on == NULL);

	*out_interface_array = arr;
	return in->objcount;
}

oskit_error_t
oskit_lookup_first(const oskit_guid_t *iid, void **out_intf)
{
	struct iidnode *in;
	oskit_iunknown_t *intf;

	/* Find the appropriate iidnode */
	for (in = iids; ; in = in->next) {
		if (in == NULL) {
			*out_intf = NULL;
			return 0;
		}
		if (memcmp(&in->iid, iid, sizeof(*iid)) == 0)
			break;
	}
	if (in->objcount == 0) {
		*out_intf = NULL;
		return 0;
	}

	*out_intf = intf = in->objs->intf;
	oskit_iunknown_addref(intf);
	return 0;
}

