	Page	58,132
	Title	SUPPORT.ASM	Apple Emulator Support Routines
;******************************************************************************
;
;   Name:	SUPPORT.ASM	Apple Emulator Support Routines
;
;   Group:	Emulator
;
;   Revision:	1.00
;
;   Date:	January 30, 1988
;
;   Author:	Randy W. Spurlock
;
;******************************************************************************
;
;  Module Functional Description:
;
;		This module contains all the code for the support routines
;	needed by the Apple ][ emulator program.
;
;******************************************************************************
;
;  Changes:
;
;    DATE     REVISION				DESCRIPTION
;  --------   --------	-------------------------------------------------------
;   1/30/88	1.00	Original
;
;******************************************************************************
	Page
;
;  Public Declarations
;
	Public	Get_Parameter		; Get parameter routine
	Public	Upper_Case		; Convert to upper case routine
	Public	Match_Parameter 	; Match parameter routine
	Public	Move_Block		; Move memory block routine
	Public	ASCII_Binary		; ASCII to binary conversion routine
	Page
;
;  LOCAL Equates
;
HEX		Equ	16		; Base 16 - Hexadecimal
DECIMAL 	Equ	10		; Base 10 - Decimal
OCTAL		Equ	8		; Base 8  - Octal
BINARY		Equ	2		; Base 2  - Binary
TAB_END 	Equ	00h		; End of table value
VALID_DIGIT	Equ	0Ah		; Valid decimal digit check value
NEXT_DELIMITER	Equ	02h		; Offset to next delimiter character
UPPER_MIN	Equ	61h		; Upper case minimum range value
UPPER_MAX	Equ	7Ah		; Upper case maximum range value
DECIMAL_ADJUST	Equ	30h		; ASCII decimal to binary adjust value
HEX_ADJUST	Equ	07h		; ASCII hex to binary adjust value
UPPER_CONVERT	Equ	0DFh		; Uppercase conversion mask value
PARM_FOUND	Equ	0FFh		; Parameter found flag value
MAX_LENGTH	Equ	0FFFFh		; Maximum move block length value
;
;  Define any include files needed
;
	Include 	Macros.inc	; Include the macro definitions
	Include 	Equates.inc	; Include the equate definitions
	.286c				; Include 80286 instructions
	Page
;
;  Define the emulator code segment
;
Emulate Segment Word Public 'EMULATE'   ; Emulator code segment
	Assume	cs:Emulate, ds:Nothing, es:Nothing
	Subttl	Get_Parameter		Get Parameter Procedure
	Page	+
;******************************************************************************
;
;  Routine Functional Description
;
;	Get_Parameter(Terminator, Delimiter, Length, Offset, Buffer)
;
;		Save the required registers
;		Calculate current buffer offset
;		Initialize counters
;		If there is room in the output buffer (Size > 1)
;			While white space in the input buffer
;				Get next character from the input buffer
;				Check for a whitespace character (Space or Tab)
;			EndWhile
;			Set the parameter found flag
;			While character is not the buffer terminator
;				If character is NOT a delimiter
;					If character is NOT whitespace
;						If room left in the buffer
;							Store this character
;						Endif
;					Endif
;				Else
;					Exit the while loop (Parameter end)
;				Endif
;			EndWhile
;			Store buffer terminator
;			Set flags to correct state (Zero if NULL parameter)
;		Else
;			Store buffer terminator character in output buffer
;			Set zero and carry flags indicating no parameter
;		Endif
;		Restore the required registers
;		Return control to the caller
;
;	Registers on Entry:
;
;		AL    - Parameter terminator byte
;		AH    - Buffer terminator byte
;		ES:BX - Pointer to delimiter table
;		CL    - Length of output buffer (>= 2)
;		DX    - Starting offset in input buffer
;		DS:SI - Pointer to input buffer
;		ES:DI - Pointer to output buffer
;
;	Registers on Exit:
;
;		AL    - Parameter length (0 = NULL parameter)
;		AH    - Delimiter type found
;		DX    - Offset in input buffer updated for next scan
;		FL    - CY set if no parameter found (NOT set for NULL)
;			ZR set if NULL parameter
;
;******************************************************************************
Get_Parameter	Proc	Near		; Get a parameter procedure
	Save	bx,cx,si,di,bp		; Save the required registers
	add	si,dx			; Setup the input buffer pointer
	push	ax			; Save the terminator bytes
	mov	bp,bx			; Setup the delimiter pointer
	xor	bh,bh			; Reset the parameter found flag
	xor	bl,bl			; Zero the output counter
	xor	ch,ch			; Default the delimiter type (None)
	dec	cl			; Correct count for terminator byte
	jz	Finish			; Finished if no buffer space left
Remove:
	lodsb				; Get next byte from the input buffer
	inc	dx			; Increment the buffer offset
	cmp	al,SPACE		; Check for a blank character
	je	Remove			; Keep scanning if blank character
	cmp	al,HT			; Check for a horizontal tab character
	je	Remove			; Keep scanning if tab character
	dec	si			; Correct the buffer
	dec	dx			;		     pointer and offset
Scan:
	lodsb				; Get next byte from the input buffer
	cmp	al,ah			; Check for a terminator character
	je	Finish			; Jump if terminator (Buffer empty)
	inc	dx			; Increment the buffer offset
	push	bp			; Save the delimiter table pointer
	mov	bh,PARM_FOUND		; Set the parameter found flag
Del_Check:
	cmp	al,Byte Ptr es:[bp]	; Check against current delimiter
	je	Del_Match		; Jump if this is a delimiter
	add	bp,NEXT_DELIMITER	; Increment delimiter table pointer
	cmp	Byte Ptr es:[bp],NULL	; Check for end of delimiter table
	jne	Del_Check		; Jump if more delimiters to check
	pop	bp			; Restore delimiter table pointer
	cmp	al,SPACE		; Check for a blank character
	je	Scan			; Jump if a blank (Never store)
	cmp	al,HT			; Check for a horizontal tab character
	je	Scan			; Jump if a tab (Never store)
	or	cl,cl			; Check for more room in output buffer
	jz	Scan			; Jump if no more room (Skip store)
	stosb				; Store the output buffer character
	inc	bl			; Increment the output counter
	dec	cl			; Decrement the buffer count
	jmp	Short Scan		; Keep scanning the input buffer
Del_Match:
	mov	ch,Byte Ptr es:[bp+1]	; Get the delimiter type value
	pop	bp			; Restore the delimiter table pointer
Finish:
	mov	cl,bl			; Save the actual parameter length
	pop	ax			; Restore the parameter terminator byte
	stosb				; Store the terminator in buffer
	or	bh,bh			; Check for valid parameter found
	jz	Bad			; Jump if NO parameter was found
Good:
	mov	ax,cx			; Get terminator/length values
	or	al,al			; Set flags to the correct state
	jmp	Short Get_Return	; Go return control to the caller
Bad:
	mov	ax,cx			; Get terminator/length values
	stc				; Set carry indicating no parameter
Get_Return:
	Restore bx,cx,si,di,bp		; Restore the required registers
	ret				; Return to the caller
Get_Parameter	Endp			; End of Get_Parameter procedure
	Subttl	Upper_Case		Uppercase Conversion Procedure
	Page	+
;******************************************************************************
;
;  Routine Functional Description
;
;	Upper_Case(Length, Offset, Buffer)
;
;		Save the required registers
;		While there are characters to convert
;			Get the next character to convert
;			If character is lowercase
;				Convert character to uppercase
;			Endif
;			Store the converted character
;		Endwhile characters to convert
;		Restore the required registers
;		Return control to the caller
;
;	Registers on Entry:
;
;		AL    - Parameter length (0 = NULL parameter)
;		ES:DI - Pointer to buffer
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Upper_Case	Proc	Near		; Convert to upper case procedure
	Save	ax,cx,di		; Save the required registers
	xor	ah,ah			; Convert parameter length to full word
	mov	cx,ax			; Setup to convert to upper case
	jcxz	Upper_Exit		; Jump if no characters to convert
Upper_Loop:
	mov	al,Byte Ptr es:[di]	; Get the next parameter character
	cmp	al,UPPER_MIN		; Compare against upper minimum value
	jb	Upper_Next		; Jump if below the minimum range
	cmp	al,UPPER_MAX		; Compare against upper maximum value
	ja	Upper_Next		; Jump if above the maximum range
	and	al,UPPER_CONVERT	; Convert character to upper case
	mov	Byte Ptr es:[di],al	; Update the parameter character value
Upper_Next:
	inc	di			; Increment to the next character value
	loop	Upper_Loop		; Loop till all characters processed
Upper_Exit:
	Restore ax,cx,di		; Restore the required registers
	ret				; Return to the caller
Upper_Case	Endp			; End of Upper_Case procedure
	Subttl	Match_Parameter 	Match Paramater Procedure
	Page	+
;******************************************************************************
;
;  Routine Functional Description
;
;	Match_Parameter(Size, Length, Table, String)
;
;		Save the required registers
;		Setup the match length value
;		While there are entries in the match table
;			Get the next match entry minimum match length
;			If enough characters for a match test
;				Check for a string match
;				If a string match was found
;					Exit the while loop with match found
;				Endif
;			Endif
;			Increment to the next match entry
;		Endwhile
;		If no matching entry was found
;			Set carry flag indicating no match
;		Else
;			Clear carry flag indicating match found
;		Endif
;		Restore the required registers
;		Return control to the caller
;
;	Registers on Entry:
;
;		AL    - Parameter length
;		AH    - Match table entry size
;		DS:SI - Pointer to match table
;		ES:DI - Pointer to parameter string
;
;	Registers on Exit:
;
;		AL    - Matching entry number (Zero based)
;		AH    - Actual match length used
;		FL    - CY set if no match found
;
;******************************************************************************
Match_Parameter Proc	Near		; Match parameter procedure
	Save	bx,cx,si,di		; Save the required registers
	cmp	al,ah			; Check parameter length against size
	jb	Match_Setup		; Jump if parameter length ok
	mov	al,ah			; Use entry size for
	dec	al			;		     parameter length
Match_Setup:
	mov	cl,al			; Setup the match
	xor	ch,ch			;		  length value
Start_Check:
	xor	bx,bx			; Zero the match entry counter
	xor	al,al			; Convert table entry
	xchg	al,ah			;		      size to a word
Match_Check:
	Save	ax,cx,si,di		; Save the required registers
	lodsb				; Get the minimum match length required
	cmp	cl,al			; Check against the match length
	jb	Skip_Check		; Jump if not enough characters
	repe	cmpsb			; Check for a string match
Skip_Check:
	Restore ax,cx,si,di		; Restore the required registers
	je	Found			; Jump if a match was found
	inc	bx			; Increment the match counter
	add	si,ax			; Increment to the next match entry
	cmp	Byte Ptr ds:[si],NULL	; Check for end of the match table
	stc				; Set carry flag in case table end
	je	Match_Return		; Jump if no more match entries
	jmp	Short Match_Check	; Go check the next match entry
Found:
	clc				; Clear the carry flag indicating match
	mov	ah,cl			; Get the actual match length used
	mov	al,bl			; Get the matching entry number
Match_Return:
	Restore bx,cx,si,di		; Restore the required registers
	ret				; Return to the caller
Match_Parameter Endp			; End of the Match procedure
	Subttl	Move_Block		Move Memory Block Routine
	Page	+
;******************************************************************************
;
;	Move_Block(Source, Destination, Count)
;
;		Save the required registers
;		If variable move amount requested
;			Scan source to determine move amount
;		Else
;			Use requested amount for move
;		Endif
;		Perform the move operation
;		Restore the required registers
;		Return control to the caller
;
;	Registers on Entry:
;
;		AL    - Terminator byte if variable length
;		AH    - Byte count (Zero means variable)
;		DS:SI - Pointer to source memory block
;		ES:DI - Pointer to destination memory block
;
;	Registers on Exit:
;
;		AX    - Actual byte count moved
;
;******************************************************************************
Move_Block	Proc	Near		; Move memory block procedure
	Save	cx,si,di		; Save the required registers
	or	ah,ah			; Check for a variable move amount
	jz	Variable		; Jump if variable move amount
	xchg	al,ah			; Move byte count value into position
	xor	ah,ah			; Convert byte count to a full word
	jmp	Short Move_Memory	; Go move the memory block
Variable:
	Save	si,di,es		; Save the required registers
	mov	cx,ds			; Setup access
	mov	es,cx			;	       to the
	xchg	si,di			;		      source block
	mov	cx,MAX_LENGTH		; Setup to search maximum length
	repne	scasb			; Search for the terminator character
	Restore si,di,es		; Restore the required registers
	mov	ax,MAX_LENGTH		; Compute the actual
	sub	ax,cx			;		     block length value
Move_Memory:
	mov	cx,ax			; Get the number of bytes to move
	rep	movsb			; Perform the actual block move
	Restore cx,si,di		; Restore the required registers
	ret				; Return to the caller
Move_Block	Endp			; End of the Move_Block procedure
	Subttl	ASCII_Binary		ASCII to Binary Conversion Routine
	Page	+
;******************************************************************************
;
;	ASCII_Binary(Base, String)
;
;		Save the required registers
;		Initialize the binary value to zero
;		While there are more characters in the string (ASCIIZ)
;			Get next character from the string
;			Convert the character to binary
;			If the character is not a valid digit
;				Set carry flag indicating string is non-numeric
;			Endif
;			Multiply current value by ten and add in new digit
;		Endwhile
;		Restore the required registers
;		Return control to the caller
;
;	Registers on Entry:
;
;		AX    - Number base
;		ES:DI - Pointer to ASCIIZ string
;
;	Registers on Exit:
;
;		AX    - Binary value of ASCIIZ string
;		FL    - CY set if string non-numeric
;
;******************************************************************************
ASCII_Binary	Proc	Near		; ASCII to binary conversion procedure
	Save	bx,cx,dx,si,ds		; Save the required registers
	mov	bx,es			; Setup access
	mov	ds,bx			;	       to the
	mov	si,di			;		      numeric string
	xor	bx,bx			; Initialize the binary number value
	mov	cx,ax			; Save the number base value
	xor	ah,ah			; Setup to get character bytes
Binary_Loop:
	lodsb				; Get the next ASCII character
	or	al,al			; Check for end of ASCIIZ string
	je	Binary_Exit		; Go return value to the caller
	call	Convert_Digit		; Call routine to convert to binary
	jc	Binary_Error		; Jump if not a valid digit value
	xchg	ax,bx			; Multiply
	mul	cx			;	   current value
	xchg	ax,bx			;			 by base
	add	bx,ax			; Add in new digit to current value
	jmp	Short Binary_Loop	; Loop till entire string is done
Binary_Error:
	stc				; Set carry indicating conversion error
Binary_Exit:
	mov	ax,bx			; Setup the binary value in AX
	Restore bx,cx,dx,si,ds		; Restore the required registers
	ret				; Return to the caller
ASCII_Binary	Endp			; End of the ASCII_Binary procedure
	Subttl	Convert_Digit	Convert to Binary Conversion Routine
	Page	+
;******************************************************************************
;
;  Routine Functional Description
;
;	Convert_Digit(Character, Base)
;
;		If character is a decimal digit (0 to 9)
;			Convert ASCII digit to binary (Subtract 30h)
;		Else character is not a decimal digit
;			Convert character to uppercase
;			Convert ASCII digit to binary (Subtract 37h)
;		Endif
;		If digit value is valid for given base
;			Clear carry indicating valid digit
;		Else
;			Set carry carry indicating invalid digit
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Digit to check (ASCII)
;		CL    - Number system base
;
;	Registers on Exit:
;
;		AL    - Binary value of digit (If it is a digit)
;		CY    - Set if character is not a digit in current base
;
;******************************************************************************
Convert_Digit	Proc	Near		; Convert character procedure
	sub	al,DECIMAL_ADJUST	; Adjust character for ASCII decimal
	cmp	al,DECIMAL		; Check for a valid decimal character
	cmc				; Set carry flag to correct state
	jnc	Convert_Check		; Jump if a valid decimal digit
	and	al,UPPER_CONVERT	; Convert anything else to uppercase
	sub	al,HEX_ADJUST		; Adjust character for ASCII hexadecimal
Convert_Check:
	cmp	al,cl			; Check for valid digit in this base
	cmc				; Set carry flag to correct state
Convert_Exit:
	ret				; Return to caller with value and flag
Convert_Digit	Endp			; End of the Convert_Digit procedure
;******************************************************************************
;
;	Define the end of the Emulator Code Segment
;
;******************************************************************************
Emulate Ends
	End				; End of the Support module
