/*	@(#)obj_int.h 1.2 86/10/07 SMI	*/

/*
 * Copyright (c) 1986 by Sun Microsystems, Inc.
 */

/***
 *
 * Object Parameters
 *
 ***/
#ifndef	_OBJ_INT
#include "obj.h"

typedef int	INT;		/* integer equivalent to object ID */
typedef union
  {
   OBJID	o;		/* object ID */
   INT		i;		/* integer */
  } ENTRY;
typedef	uint	SIZE;		/* block size */
typedef	uint	OFFSET;		/* object entry offset */

/***
 *
 * Object Type Information
 * Each object has a type which tells how the object is represented
 * internally in a general way. An object can contain integers only,
 * object ids only, or a combination of both. Variable size lists
 * have an integer as their first entry which gives the byte size
 * of the data area (which may be smaller than the total memory
 * allocated for the list).
 *
 ***/
#define	T_OOBJ		1	/* fixed size list of object ids */
#define	T_IOBJ		2	/* fixed size list of integers */
#define	T_VOBJ		3	/* fixed size list of objects / integers */
#define	T_LST		4	/* variable length lists */
#define	T_OLST		4	/* variable size list of object ids */
#define	T_ILST		5	/* variable size list of integers */
#define	T_VLST		6	/* variable size list of objects / integers */

#define	L_HDR		sizeof(ENTRY)		/* size of list header */
#define	L_RESIZE	(32*sizeof(ENTRY))	/* # entries to extend list */

/***
 *
 * OBJENT - Object Table Entry
 * This is just the location of the data part of the allocated block in memory.
 * Negative offsets reference the object header,
 * positive offsets reference data.
 *
 * OBJHDR - Object Header
 * Each allocated object in the system has a header as follows.
 * The object type determines whether object id numbers, integers or
 * both are stored in the object. The class tells how the data is
 * interpreted by the system (what it means).
 *
 ***/
#define	T_UNDEF	0
typedef struct
  {
   SIZE		o_size;		/* size (# bytes) of object in memory */
   uchar	o_type;		/* type of object */
   uchar	o_class;	/* class of object */
   uchar	o_use;		/* object use count */
   uchar	o_flag;		/* object flags */
  } OBJHDR;

typedef char	*OBJENT;

/***
 *
 * OBJNUM(o)
 * Returns a 0 based object offset given an OBJID
 *
 ***/
#define	OBJNUM(o) (o)

/***
 *
 * OBJENT ObjEnt(objid)
 * Returns the contents of an object entry. The object ID is really an
 * offset into the object use table. Offsets into the object entry
 * table must be scaled by this factor.
 *
 ***/
#define ObjEnt(o)       (o)

/***
 *
 * IsNull(objid)
 * Returns YES if the object ID is NULL, otherwise NO
 *
 ***/
#define	IsNull(o)	((o)==0)

/***
 *
 * IsObj(value)
 * Returns YES if the given value represents an object header
 * (currently that means if the low order bit is clear)
 * This test is meaningful only for data structures
 * which can contain object ids or integers
 *
 ***/
#define	IsObj(v)	~(((INT) (v))&1)

/***
 *
 * IsInt(value)
 * Returns YES if the given value represents an integer
 * (currently that means if the low order bit is set)
 * This test is meaningful only for data structures
 * which can contain object ids or integers
 *
 ***/
#define	IsInt(v)	(((INT) (v))&1)

/***
 *
 * MakeObj(value)
 * Returns a representation of an object id suitable for storing
 * in a mixed data structure (object ids and integers).
 * Since object IDs are offsets into the object table they
 * must be shifted left to insure they are even.
 *
 ***/
#define	MakeObj(v)	((OBJID) (v))

/***
 *
 * MakeInt(value)
 * Returns a representation of an integer suitable for storing
 * in a mixed data structure (integers and object ids).
 * Currently it shifts the number left by 1 and sets the low
 * order bit. Note that 1 bit of precision for integer representations
 * is lost if we do this. Integers are always odd.
 *
 ***/
#define	MakeInt(v)	(((INT) (v)<<1)|1)

/***
 *
 * NormObj(objid)
 * Returns a normal object id, given one that comes from a mixed
 * structure (object ids and integers). (Reverse of MakeObj).
 *
 ***/
#define	NormObj(o)	((OBJID) (o))

/***
 *
 * NormInt(i)
 * Returns a normal integer, given one that comes from a T_LIST
 * structure (integers and object ids mixed). (Reverse of MakeInt).
 *
 ***/
#define	NormInt(i)	(((INT) (i))>>1)

/***
 *
 * (type *) LstAddr(objid, type)
 * Returns the memory address of a list as a pointer to the given type
 *
 ***/
#define	LstAddr(o,t)	GetEntry(o,L_HDR,t)

/***
 *
 * OBJHDR *GetObjHdr(objid)
 * Returns the address of the object header referenced
 * by the given object id.
 *
 ***/
#define	GetObjHdr(o)	((OBJHDR *) (ObjEnt(o) - sizeof(OBJHDR)))

/***
 *
 * SIZE ObjSize(objid)
 * Returns the size of the given object
 *
 ***/
#define	ObjSize(o)	(GetObjHdr(o)->o_size)

/***
 *
 * SIZE LstSize(objid)
 * Returns the size of the given list
 *
 ***/
#define	LstSize(o)	(GetEntry(o, 0, ENTRY)->i)

/***
 *
 * OFFSET LstOfs(objid)
 * Returns the offset of the end of the given list
 *
 ***/
#define	LstOfs(o)	(LstSize(o)+L_HDR)

/***
 *
 * OBJID NewLst(type, size)
 * Makes a new list object
 *
 ***/
#define	NewLst(t,s)	NewObj(t,(s)+L_HDR)

/***
 *
 * uchar ObjType(objid)
 * Returns the type of the given object
 *
 ***/
#define	ObjType(o)	(GetObjHdr(o)->o_type)

/***
 *
 * uchar ObjClass(objid)
 * Returns the class of the given object
 *
 ***/
#define	ObjClass(o)	(GetObjHdr(o)->o_class)

/***
 *
 * uchar ObjUse(objid)
 * Returns the use count of the given object
 *
 ***/
#define	ObjUse(o)	(GetObjHdr(o)->o_use)

/***
 *
 * uchar ObjFlag(objid)
 * Returns the flags of the given object
 *
 ***/
#define	ObjFlag(o)	(GetObjHdr(o)->o_flag)

/***
 *
 * t *GetEntry(objid, n, t)
 * Given an object id, returns a pointer to the Nth byte in the object
 *
 ***/
#define	GetEntry(o,n,t)	((t *) (ObjEnt(o)+(n)))

/***
 *
 * VOID ObjLock(objid)
 * Lock object
 *
 ***/
#define	ObjLock(o)	(ObjFlag(o)|=O_LOCK)

/***
 *
 * VOID ObjUnlock(objid)
 * Unlock object
 *
 ***/
#define	ObjUnlock(o)	(ObjFlag(o)&=~O_LOCK)

/***
 *
 * VOID FreeObj(objid)
 * Decrement reference count and garbage collect object if necessary
 *
 * VOID KillObj(objid)
 * Garbage collect the object - do not check use count
 *
 ***/
#define	KillObj(o) free(GetObjHdr(o))
#define	FreeObj(o) (IsNull(o) ? NULL :\
		 (--ObjUse(NormObj(o)) <= 1) ? KillObj(o) : NULL)

/***
 *
 * int GetIntVal(objid, offset)
 * Returns an integer value from an object with integer only
 * entries (type T_IOBJ).
 *
 ***/
#define	GetIntVal(i,o) (*GetEntry(i,o,int))

/***
 *
 * short GetWrdVal(objid, offset)
 * Returns a word value from an object with integer only
 * entries (type T_IOBJ).
 *
 ***/
#define	GetWrdVal(i,o) (*GetEntry(i,o,short))

/***
 *
 * char GetChrVal(objid, offset)
 * Returns a character value from an object with integer only
 * entries (type T_IOBJ).
 *
 ***/
#define	GetChrVal(i,o) (*GetEntry(i,o,char))

/***
 *
 * int GetIntLst(objid, offset)
 * Returns an integer value from an integer list (type T_ILST).
 *
 ***/
#define	GetIntLst(i,o) (*GetEntry(i,L_HDR+(o),int))

/***
 *
 * short GetWrdLst(objid, offset)
 * Returns a word value from an integer list (type T_ILST).
 *
 ***/
#define	GetWrdLst(i,o) (*GetEntry(i,L_HDR+(o),short))

/***
 *
 * char GetChrLst(objid, offset)
 * Returns a character value from an integer list (type T_ILST).
 *
 ***/
#define	GetChrLst(i,o) (*GetEntry(i,L_HDR+(o),char))

/***
 *
 * OBJID GetObjVal(objid, offset)
 * Returns an object ID from an object containing only object
 * references (type T_OOBJ).
 *
 ***/
#define	GetObjVal(i,o) NormObj(*GetEntry(i,o,OBJID))

/***
 *
 * OBJID GetObjLst(objid, offset)
 * Returns an object ID from an object only list (type T_OLST).
 *
 ***/
#define	GetObjLst(i,o) NormObj(*GetEntry(i,L_HDR+(o),OBJID))

/***
 *
 * ENTRY GetVal(objid, offset)
 * Returns a value (integer or object) from a object containing
 * mixed items (type T_VOBJ).
 *
 ***/
#define	GetVal(i,o) (*GetEntry(i,o,ENTRY))

/***
 *
 * ENTRY GetLst(objid, offset)
 * Returns a value (integer or object) from a mixed list (type T_VLST).
 *
 ***/
#define	GetLst(i,o) (*GetEntry(i,(o)+L_HDR,ENTRY))

/***
 *
 * int PutIntVal(objid, offset, value)
 * Stores an integer value in an object with integer only
 * entries (type T_IOBJ).
 *
 ***/
#define	PutIntVal(i,o,v) (*GetEntry(i,o,int) = v)

/***
 *
 * short PutWrdVal(objid, offset, value)
 * Stores an word value in an object with integer only
 * entries (type T_IOBJ).
 *
 ***/
#define	PutWrdVal(i,o,v) (*GetEntry(i,o,short) = v)

/***
 *
 * char PutChrVal(objid, offset, value)
 * Stores an character value in an object with integer only
 * entries (type T_IOBJ).
 *
 ***/
#define	PutChrVal(i,o,v) (*GetEntry(i,o,char) = v)

/***
 *
 * int PutIntLst(objid, offset, value)
 * Stores an integer value in an integer list (type T_ILST).
 *
 ***/
#define	PutIntLst(i,o,v) (*GetEntry(i,L_HDR+(o),int) = v)

/***
 *
 * short PutWrdLst(objid, offset, value)
 * Stores a word value in an integer list (type T_ILST).
 *
 ***/
#define	PutWrdLst(i,o,v) (*GetEntry(i,L_HDR+(o),short) = v)

/***
 *
 * char PutChrLst(objid, offset, value)
 * Stores a character value in an integer list (type T_ILST).
 *
 ***/
#define	PutChrLst(i,o,v) (*GetEntry(i,L_HDR+(o),char) = v)

/***
 *
 * General Globals
 *
 ***/
extern	OBJID	NewObj();	/* make new object */
extern	OBJID	AppIntLst();	/* append integer to list */
extern	OBJID	AppWrdLst();	/* append word to list */
extern	OBJID	AppChrLst();	/* append character to list */
extern	OBJID	AppBufLst();	/* append buffer to list */
extern	OBJID	AppLst();	/* append to list */
extern	ENTRY	NullObj;	/* null object to return */
extern	int	objdbg;

#define	_OBJ_INT	1
#endif
