/*	@(#)gpbuf_int.h 1.2 86/10/07 SMI	*/

/*
 * Copyright (c) 1986 by Sun Microsystems, Inc.
 */

/****
 *
 * GPBUF_INT.H - GP buffer internal header
 * GPBUF	GP buffer object
 * GPATTR	GP attributes
 *
 ****/
#ifndef	_GPINT
#include "vp_int.h"
#include "gpbuf.h"
#include <pixrect/gp1cmds.h>

/*
 * GP1 constants
 */
#define	GPB_MOVE_FLAG	1
#define	GPB_DRAW_FLAG	0
#define	GPB_END_FLAG	0x8000

/****
 *
 * GP Buffer allocation / posting
 *
 * NEWGPBUF(g, ptr, cmdofs, bitvec)
 * Allocates a new GP buffer from shared memory. CMDOFS gets the
 * shared memory offset, BITVEC gets the bit vector and PTR gets
 * the address of the first available command slot. An initial
 * USE_CONTEXT is put into the buffer.
 *
 * POSTGPBUF(gpb, cmdofs)
 * Post a GP buffer (EOCL command NOT added)
 *
 ****/
#define	NEWGPBUF(g, ptr, cmdofs, bitvec) {\
	while (((cmdofs) = gp1_alloc(g->gpb_shmem, 1, &(bitvec),\
		g->gpb_mindev, g->gpb_gfd)) == 0) ;\
	ptr = &((short *) g->gpb_shmem)[cmdofs];\
	*(ptr)++ = GP1_USE_CONTEXT | (g->gpb_sbindex & 0x7);\
	}

#define	POSTGPBUF(g, cmdofs)	gp1_post(g->gpb_shmem, cmdofs, g->gpb_gfd)
		
/****
 *
 * GP Buffer Management
 *
 * MAKEROOM(n) 
 * Make room for N bytes in the GP command buffer. Flush if necessary.
 *
 * ROOMFOR(n)
 * Check to see if there is room for N bytes in the current buffer.
 * Does not flush or change pointers/counters, just checks.
 *
 * USEUP(n)
 * Use up N buffer bytes. Does not put anything into the buffer,
 * just indicates N less bytes are left.
 *
 ****/
#define	ROOMFOR(n)	(gpb->gpb_left > (n))
#define	USEUP(n)	(gpb->gpb_left -= (n))
#define	MAKEROOM(n) { if (USEUP(n) <= 0) { gp_flush(gpb); USEUP(n); }}

/****
 *
 * GP Buffer PUT / GET
 *
 * PUT_xxx(p, n)
 * Stores N where P points in the buffer and increments P
 *
 * GET_xxx(p)
 * Returns where P points in the buffer and increments P
 *
 * PUT_FLOAT(p, f)	Append float to buffer
 * PUT_INT(p, i)	Append int to buffer
 * PUT_SHORT(p, i)	Append short to buffer
 * PUT_CMD(p, c, a)	Append GP command and option to buffer
 *
 * GET_FLOAT(p)		Get float from buffer
 * GET_INT(p)		Get int from buffer
 * GET_SHORT(p)		Get short from buffer
 *
 ****/
#define	GET_FLOAT(p)	(*((float *) (p))++)
#define	GET_INT(p)	(*((int *) (p))++)
#define	GET_SHORT(p)	(*((short *) (p))++)
#define	PUT_FLOAT(p, f)	(*((float *) (p))++ = f)
#define	PUT_INT(p, i)	(*((int *) (p))++ = i)
#define	PUT_SHORT(p, i)	(*((short *) (p))++ = i)
#define	PUT_CMD(p, c, a) (*((short *) (p))++ = (c | (a)))

/*
 * default sizes
 */
#define	GPB_MAXCLIP		60
#define	GPB_POLYMASK		VP_POLYMASK
#define	GPB_BYTESPERBLOCK	1024
#define	GPB_DEFAULTSIZE		1
#define	GPB_VIEWMATRIX	1
#define	GPB_XFORMMATRIX	0

#define	GPB_COORDMSK	VP_COORDMASK

/*
 * viewing and transformation flags
 */
#define	GPB_VIEW_FLAG	1	/* VIEW_ORG/VIEW_DIM attributes changed */
#define	GPB_CLIP_FLAG	2	/* CLIP_ORG/CLIP_DIM attributes changed */
#define	GPB_WIN_FLAG	4	/* WIN_ORG/WIN_DIM attributes changed */
#define	GPB_RESIZE_FLAG	8	/* window was resized */
#define	GPB_XFORM_FLAG	16	/* change transformation matrix */

/*
 * VP -> GP structure field mappings
 */
#define	gpb_type	gpb_vp.vp_type
#define	gpb_ftype	gpb_vp.vp_ftype
#define	gpb_coord	gpb_vp.vp_coord
#define	gpb_dbuf	gpb_vp.vp_dbuf
#define	gpb_pick	gpb_vp.vp_pick
#define	gpb_cindex	gpb_vp.vp_cindex
#define	gpb_textab	gpb_vp.vp_textab
#define	gpb_pattab	gpb_vp.vp_pattab
#define	gpb_wdim	gpb_vp.vp_wdim

#define	gpb_color	gpb_vp.vp_color
#define	gpb_rop		gpb_vp.vp_rop
#define	gpb_fill	gpb_vp.vp_fill
#define	gpb_visible	gpb_vp.vp_visible
#define	gpb_pen		gpb_vp.vp_pen
#define	gpb_pixbits	gpb_vp.vp_pixbits
#define	gpb_pixplanes	gpb_vp.vp_pixplanes
#define	gpb_frame	gpb_vp.vp_frame
#define	gpb_dbuf_stat	gpb_vp.vp_dbuf_stat
#define	gpb_win		gpb_vp.vp_win
#define	gpb_pick_id	gpb_vp.vp_pick_id
#define	gpb_pick_count	gpb_vp.vp_pick_count
#define	gpb_pick_org	gpb_vp.vp_pick_org
#define	gpb_pick_dim	gpb_vp.vp_pick_dim
#define	gpb_line_style	gpb_vp.vp_line_style
#define	gpb_line_width	gpb_vp.vp_line_width
#define	gpb_line_height	gpb_vp.vp_line_height
#define	gpb_line_size	gpb_vp.vp_line_size
#define	gpb_pattern	gpb_vp.vp_pattern

#define	gpb_size	gpb_attr.gpa_SIZE
#define	gpb_vertex_color gpb_attr.gpa_VERTEX_COLOR
#define	gpb_VIEW_ORG	gpb_vp.vp_view_org
#define	gpb_VIEW_DIM	gpb_vp.vp_view_dim
#define	gpb_WIN_ORG	gpb_attr.gpa_WIN_ORG
#define	gpb_WIN_DIM	gpb_attr.gpa_WIN_DIM
#define	gpb_CLIP_ORG	gpb_attr.gpa_CLIP_ORG
#define	gpb_CLIP_DIM	gpb_attr.gpa_CLIP_DIM

/*
 * GPBUF contains all the information needed to talk to the GP, build
 * vector lists, allocate shared memory, etc. This is the handle used
 * to communicate with xxx_GP routines.
 */
typedef struct
  {
   struct vp_hdr gpb_vp;	/* viewport header */
   struct _gpa	gpb_attr;	/* attributes */
   char		gpb_mindev;	/* true minor device number */
   char		gpb_flags;	/* viewing & transformation flags */
   short	gpb_left;	/* bytes in buffer */
   short	gpb_vstart;	/* vector list start code */
   short	gpb_sbindex;	/* static block index */
   short	gpb_clipid;	/* clip list ID */
   short	gpb_cmdofs;	/* command buffer memory block offset */
   int		gpb_cmdbv;	/* command buffer bit vector */
   char		*gpb_shmem;	/* GP shared memory */
   short	*gpb_ptr;	/* GP command buffer pointer */
   short	*gpb_vecsize;	/* -> vector list size */
   short	*gpb_polysize;	/* -> polygon list size */
   short	gpb_gfd;	/* GP device file descriptor */
   short	gpb_fbplanes;	/* real pixel planes mask */
   short	gpb_pat_block;	/* patterned polygon static block number */
   short	gpb_pixmask;	/* pixel color mask (for gpb_cindex offset) */
   short	gpb_matupd;	/* matrix update bits */
   short	gpb_winx;	/* window X position */
   short	gpb_winy;	/* window Y position */
   POINT	gpb_poly;	/* start of polygon */
   POINT	gpb_wold;	/* old window dimensions */
   POINT	gpb_clip_org;	/* normalized clip area origin */
   POINT	gpb_clip_dim;	/* normalized clip area dimensions */
   POINT	gpb_view_trans;	/* viewport translation factors */
   POINT	gpb_view_scale;	/* viewport scale factors */
   MATRIX	gpb_viewmatrix;	/* viewing matrix */
  } GPDATA;			/* GP buffer info */

#define	_GPINT	1
#endif
