#ifndef lint
static	char sccsid[] = "@(#)dev_gp1.c 1.2 86/10/07 SMI";
#endif

/*
 * Copyright (c) 1986 by Sun Microsystems, Inc.
 */

/* 
 * This file contains the control and viewport functions for the
 * DEV_GP1 package. DEV_GP1 is an example of how to use the GP command
 * interface. This code is not supported by Sun Microsystems, Inc.
 * 
 * DEV_GP1 is a functional interface to the GP. In general, there is
 * a routine for each (2d and 3d floating point) GP command. The package
 * asks the GP for one or more contiguous command blocks; these blocks
 * are refered to as a buffer. DEV_GP1 tries to fill the buffer as full
 * as possible before sending it to the GP. 
 * 
 * All routines
 * have the same basic format: A routine receives as parameters the 
 * argments to the GP command and checks to see if the command and its
 * arguments will fit in the buffer. If they can, they is are placed into
 * the buffer; otherwise the buffer is flushed (i.e. sent to the GP) and
 * then the command and arguments are placed in it.
 * 
 * There are a couple elements of the GP1 structure deserve comment:
 * 	bytes_left	= the number of bytes left in the buffer; note that
 * 			  GP shared memory and commands are talked about
 * 			  in terms of shorts (16 bits) but we keep track of
 * 			  bytes left because sizeof returns bytes.
 * 	list_type	= the DEV_GP1 interface allows the user to specify
 * 			  a vector list (via move and draw calls). DEV_GP1
 * 			  uses this field to determine if a list is already
 * 			  in progress. DEV_GP1 functions othere than move &
 * 			  draw set this field to GP1_NO_LIST which means that
 * 			  any subsequent move or draw calls must set up the
 * 			  list again.
 * 	cur_ptr		= this points to the word (a 16 bit short) in shared
 * 			  memory into which the next command or data is 
 * 			  will be written.
 * 
 * Here are a few areas within the package that you might want to change:
 * 1) The transformation of vectors and polygons go thru the same flush
 *    procedure as output primitives and as suchs incur a pw_lock call which
 *    is a system call. pw_lock does not need to be call for transformations
 *    since nothing is written on the screen. If you plan to do a lot of
 *    transformations, you might want to change the package so that a lock is
 *    not aquired for transforms. 
 * 2) If your application knows how many vectors or points in a polyline line
 *    it will have, you can improve the stuffing of these points into the
 *    shared memory by writing a routine that takes a list of points rather
 *    than using the current move/draw interface.
 * 
 * Common problems when programming the GP:
 * 1) Writing past the end of the command block(s). 
 * 2) Detecting that the window has changed; look at the clip id.
 * 3) Be sure to free the blocks you allocate. 
 * 4) Of course, be sure you write commands and arguments in the 
 *    correct format. A good check for this is to write a routine which will
 *    print out the contests of a buffer before it is posted to the GP. If
 *    you can read what you have written, chances are the GP can to.
 * 5) Debugging the GP can be tricky. Be aware that even though your program
 *    may be stopped in the debugger (and thus not accessing the GP) other
 *    programs, such as clocktool and perfmeter, will be accessing it. I 
 *    suggest that you do not run programs like clocktool while debugging.
 *    Also, if you are running the debugger on the same screen as the
 *    program you are debugging then the debugger will also be using the
 *    GP. This is not such a bad problem; however if you get really stuck,
 *    try running the debugger from a terminal. 
 * 
 *
 * -rlh, smi -- 9/12/86
 */

#include "dev_gp1_internal.h"
#include <stdio.h>

static	float	identity[4][4] = {		/* 3D identity matrix */
    {   1.0,	0.0,	0.0,	0.0},
    {   0.0,	1.0,	0.0,	0.0},
    {   0.0,	0.0,	1.0,	0.0},
    {   0.0,	0.0,	0.0,	1.0}
};
extern	char	*malloc();

static	void	alloc_buffers();
static	void	reset_to_defaults();

static	void	dev_gp1_update_clip_list();

/* Debugging Control */

/*******
#undef  DBG_GP1
#define DBG_GP1(a)  a

#define DO_DBG_BIT_VEC	1
******/



#ifdef DO_DBG_BIT_VEC
#define	DBG_BIT_VEC(x)	x
#else
#define	DBG_BIT_VEC(x)
#endif


/* 
 * Allocate a gp1_handle and get a static block.
 * Initialize everything...
 */
     

GP1_handle
Dev_gp1_create(pw, n_buffers, error_num)
Pixwin		*pw;		/* pixwin  */
int		n_buffers;	/* number of command buffer to allocate 
				 * at once.
				 */
int		*error_num;	/* if an error, return the error number */
{   
    register	GP1	*gp1;
    		int	ioctl_fd;	/* fd for ioctls */
		int	static_block;
		int	texture_block;
    
    if (n_buffers <= 0) 
	n_buffers = GP1_DEFAULT_NUMBER_OF_BUFFERS;
    
    if (pw->pw_pixrect->pr_ops->pro_rop != gp1_rop) {
	*error_num = DEV_GP1_ERR_NO_GP1;
	return((GP1 *)NULL);
    }
    
    ioctl_fd = gp1_d(pw->pw_clipdata->pwcd_prmulti)->ioctl_fd;
    
    if ((static_block = gp1_get_static_block(ioctl_fd)) < 0) {
	*error_num = DEV_GP1_ERR_NO_CONTEXT_BLK;
	return((GP1 *)NULL);
    }
    
    if ((texture_block = gp1_get_static_block(ioctl_fd)) < 0) {
	*error_num = DEV_GP1_ERR_NO_TEXTURE_BLK;
	return((GP1 *)NULL);
    }
    
    gp1 = (GP1 *) malloc(sizeof(GP1));
    if (gp1 == (GP1 *)NULL) {
	*error_num = DEV_GP1_ERR_NO_MEMORY;
	return((GP1 *)NULL);
    }
    
    gp1->clip_id = -1;			/* no clip list */    
    gp1->pw = pw;
    gp1->win_fd = pw->pw_clipdata->pwcd_windowfd;
    win_getsize(gp1->win_fd, &(gp1->win_size));
    win_getscreenposition(gp1->win_fd, 
			  &(gp1->win_org_x), &(gp1->win_org_y));
    
    gp1->fd = ioctl_fd;
    gp1->static_block_index = static_block;
    gp1->texture_block_index = texture_block;
    gp1->n_buffers = n_buffers;
    gp1->list_type = GP1_NO_LIST;

    gp1->minor_devno = gp1_d(pw->pw_clipdata->pwcd_prmulti)->minordev;
     
    gp1->shmem = (short *) 
	gp1_d(pw->pw_clipdata->pwcd_prmulti)->gp_shmem;
    
    alloc_buffers(gp1);		/* allocate command buffer list */
    Dev_gp1_reset_to_defaults(gp1);
    return (gp1);
}



/* 
 * Free resources used by the GP1:
 *    1) Write a cmd at the start of the 1st cmd block which
 * 	frees all the blocks. Do not flush the buffer.
 *    2) Free the static block via an ioctl.
 */

void
Dev_gp1_destroy(gp1)
register	GP1	*gp1;
{  
    register	short		*ptr;

    ptr = &(gp1->shmem[gp1->cmd_buff_offset]);
    ++ptr;				/* skip USEFRAME */
    *ptr++ = GP1_EOCL | GP1_FREEBLKS;	/* free command buffer */
    SHORTS_GET_INT(ptr, &(gp1->cmd_buff_bit_vec));
    gp1_post(gp1->shmem, gp1->cmd_buff_offset, gp1->fd);
    gp1_free_static_block(gp1->fd, gp1->static_block_index);
    gp1_free_static_block(gp1->fd, gp1->texture_block_index);
    free(gp1);
}



void
Dev_gp1_win_change(gp1)
register	GP1	*gp1;
{   
    int		new_clip_id;
    
    win_getsize(gp1->win_fd, &(gp1->win_size));
    win_getscreenposition(gp1->win_fd, 
			  &(gp1->win_org_x), &(gp1->win_org_y));
    
    pw_lock(gp1->pw, &(gp1->win_size));
    if (gp1->clip_id != 
	(new_clip_id = gp1->pw->pw_clipdata->pwcd_clipid)) {
	dev_gp1_update_clip_list(gp1);
	gp1->clip_id = new_clip_id;
    }
    pw_unlock(gp1->pw);
}


void
Dev_gp1_reset_to_defaults(gp1)
register	GP1	*gp1;
{   
    		int		attr;		/* pw attributes */
    		short		pat[1];
    register	short		*ptr;
    
    MAKE_ROOM(gp1, 3 * sizeof(short));
    ptr = gp1->cur_ptr;
    *ptr++ = GP1_SET_FB_NUM |
	gp1_d(gp1->pw->pw_clipdata->pwcd_prmulti)->cg2_index;
    *ptr++ = GP1_SET_PGON_TEX_BLK | (gp1->texture_block_index & 0x7);
    gp1->bytes_left -= 3 * sizeof(short);
    gp1->cur_ptr = ptr;
    
    pw_getattributes(gp1->pw, &attr);
    DEV_GP1_SET_WRITE_PLANES(gp1, attr);
    
    Dev_gp1_set_texture_org_screen(gp1, 0, 0);
    Dev_gp1_set_color(gp1, 1);
    Dev_gp1_set_rop_mode(gp1, PIX_SRC);
    Dev_gp1_set_hidden_surf(gp1, GP1_NOHIDDENSURF);
    Dev_gp1_set_clip_planes(gp1, 0);			/* no clipping */
    Dev_gp1_select_matrix(gp1, 0);
    Dev_gp1_set_matrix_3d(gp1, 0, identity);
    pat[0] = 0;
    Dev_gp1_set_line_texture(gp1, pat, 0, 0);
    Dev_gp1_set_line_width(gp1, 1);

    Dev_gp1_flush(gp1, TRUE);	/* send defaults and start anew */
}



static
void
alloc_buffers(gp1)
register	GP1	*gp1;
{   
    static	void	check_bit_vec();
    
    /* 
     * At this point, we should not have any blocks allocated.
     * check_bit_vec will see if some are.
     */
    DBG_BIT_VEC(check_bit_vec());
    
    while ((gp1->cmd_buff_offset = gp1_alloc(gp1->shmem,
					     gp1->n_buffers, 
					     &(gp1->cmd_buff_bit_vec),
					     gp1->minor_devno, 
					     gp1->fd)) == 0)
    	;
    
    
    /* 
     * substract 8 bytes from the total number of bytes in the buffers:
     *  2 bytes for GP1_USEFRAME
     *  2 bytes for GP1_EOCL
     *  4 bytes for the command bit vector
     */
    gp1->bytes_left = (gp1->n_buffers * GP1_BYTES_PER_BLOCK) - 8; 
    gp1->cur_ptr = &(gp1->shmem[gp1->cmd_buff_offset]);
    *(gp1->cur_ptr)++ = GP1_USEFRAME | (gp1->static_block_index & 0x7);
    
}

/* 
 * Look at the bit vector that indicates which blocks may be allocated
 * to a process and determine if it is OK. This routine only works if
 * this process is the only one using the GP1; try not to have pixrect
 * programs running -- no perfmeter or clock -- at lest hide them under
 * another window.
 */


static
void
check_bit_vec(gp1)
GP1	*gp1;
{   
    		uint	host;
    		uint	vp;
		char	*msg;
    register	uint	bit_vec;
    register	int	set;

    /* 
     * wait until the gp1 is done.
     */
    /* gp1_sync(gp1->shmem, gp1->fd); */
    
    host  = *((ushort *)gp1->shmem + 3) << 16;		/* hi-order word */
    host |= *((ushort *)gp1->shmem + 4);		/* low-order word */
    vp    = *((ushort *)gp1->shmem + 5) << 16;		/* hi-order word */
    vp   |= *((ushort *)gp1->shmem + 6);		/* low-order word */
    
    bit_vec = host ^ vp;
    
    if ((bit_vec & 0x80000000) == 0) {
	msg = "GP1:check_bit_vec -> control block bit == 0.\n";
	goto error;
    }
    if ((bit_vec & 0x0ff) != 0xff) {
	msg = "GP1:check_bit_vec -> static block bits != 0xff.\n";
	goto error;
    }
    /* turn off bits that should be 1 */
    bit_vec &= 0x7fffff00;
    bit_vec &= ~(gp1->cmd_buff_bit_vec);
    if (bit_vec != 0) {
	fprintf(stderr, "GP1:check_bit_vec -> found an allocated block .\n");
	fprintf(stderr, "\tgp1 bit vec:  0x%8.8x\n", gp1->cmd_buff_bit_vec);
	fprintf(stderr, "\tcur bit vec:  0x%8.8x\n", bit_vec);
	return;
    }
    
    return;
    
  error:
    fprintf(stderr, msg);
}




void
Dev_gp1_flush(gp1, free_blocks)
register	GP1	*gp1;
 		int		free_blocks;
{   
    register	Pixwin	*pw;
    register	int	new_clip_id;
    
    
    /* Nothing put in blocks, so return... */
    if ((gp1->cur_ptr - 1) <= &(gp1->shmem[gp1->cmd_buff_offset])) {
	gp1->list_type = GP1_NO_LIST;	
	return;
    }
    
    /* mark end of command list and free blocks - we are guaranteed that
     * there is room for the commands -- see alloc_buffers
     */
    if (free_blocks) {
	*(gp1->cur_ptr)++ = GP1_EOCL | GP1_FREEBLKS;
	SHORTS_GET_INT(gp1->cur_ptr, &(gp1->cmd_buff_bit_vec));
    }
    else {
	*(gp1->cur_ptr)++ = GP1_EOCL;
    }

    DBG_GP1(printf("in Dev_gp1_flush..\n"));
    DBG_GP1(Dev_gp1_print(gp1));
#ifdef DBG_GP1
    if ((gp1->cur_ptr - 1) >
	(&(gp1->shmem[gp1->cmd_buff_offset]) + gp1->n_buffers * 512)) {
	printf("**Warning** have gone past the end of the buffer\n");
    }
#endif
    
    pw = gp1->pw;    
    pw_lock(pw, &(gp1->win_size));
    if (gp1->clip_id != (new_clip_id = pw->pw_clipdata->pwcd_clipid)) {
	dev_gp1_update_clip_list(gp1);
	gp1->clip_id = new_clip_id;
    }
    
    gp1_post(gp1->shmem, gp1->cmd_buff_offset, gp1->fd);
    pw_unlock(pw);
    
    
    if (free_blocks)
	alloc_buffers(gp1);			/* allocate new buffers */
    else {
	/* re-use the cmd buffers */
	gp1->bytes_left = (gp1->n_buffers * GP1_BYTES_PER_BLOCK) - 8; 
	gp1->cur_ptr = &(gp1->shmem[gp1->cmd_buff_offset]);
	*(gp1->cur_ptr)++ = GP1_USEFRAME | (gp1->static_block_index & 0x7);
    }
    gp1->list_type = GP1_NO_LIST;
}



static
void
dev_gp1_update_clip_list(gp1)
register	GP1	*gp1;
{   
    register	struct gp1pr		*dmd;    
    register	short			*ptr;
    register	struct pixwin_prlist	*prl;
    register	short			*n_rects_ptr;
    register	int			n, offset;
    		int			bit_vec;
    
    /* 
     * allocate a block into which we can put the clip list.
     */
    while ((offset = gp1_alloc(gp1->shmem,
			       1, 
			       &(bit_vec),
			       gp1->minor_devno, 
			       gp1->fd)) == 0)
    	;
    
    ptr = &(gp1->shmem[offset]);
    *ptr++ = GP1_USEFRAME | (gp1->static_block_index & 0x7);
    
    /* 
     * Change the viewport to account for the case when the window
     * is on top and is moved. In that case, the clip-id changed (which
     * is why we are here, but there is no SIGWINCH generated. 
     */
    gp1->vp_offset.x -= gp1->win_org_x;
    gp1->vp_offset.y -= gp1->win_org_y;
    win_getscreenposition(gp1->win_fd,&(gp1->win_org_x),&(gp1->win_org_y));
    gp1->vp_offset.x += gp1->win_org_x;
    gp1->vp_offset.y += gp1->win_org_y;
    
    *ptr++ = GP1_SETVWP_3D;
    SHORTS_GET_FLOAT(ptr, &(gp1->vp_scale.x));
    SHORTS_GET_FLOAT(ptr, &(gp1->vp_offset.x));
    SHORTS_GET_FLOAT(ptr, &(gp1->vp_scale.y));
    SHORTS_GET_FLOAT(ptr, &(gp1->vp_offset.y));
    SHORTS_GET_FLOAT(ptr, &(gp1->vp_scale.z));
    SHORTS_GET_FLOAT(ptr, &(gp1->vp_offset.z));
     
    *ptr++ = GP1_SETCLPLST;
    n_rects_ptr = ptr++;
    prl = gp1->pw->pw_clipdata->pwcd_prl;
    n = 0;
    while (prl && (n < GP1_MAX_CLIP_RECTS)) {   /* foreach rect... */
	dmd = gp1_d(prl->prl_pixrect);		/* get pixrect */
	*ptr++ = dmd->cgpr_offset.x;		/* X, Y origin of rect */
	*ptr++ = dmd->cgpr_offset.y;
	*ptr++ = prl->prl_pixrect->pr_size.x; 	/* X, Y size of rect */
	*ptr++ = prl->prl_pixrect->pr_size.y;	
	prl = prl->prl_next;
	n++;
    }
    *n_rects_ptr = (short) n;
    
    *ptr++ = GP1_EOCL | GP1_FREEBLKS;
    SHORTS_GET_INT(ptr, &(bit_vec));
    gp1_post(gp1->shmem, offset, gp1->fd);
}



void
Dev_gp1_clear(gp1, color)
register	GP1	*gp1;
 		int		color;
{   
    register	struct pixwin_prlist	*prl;
    register	short			*ptr;
    register	struct gp1pr 		*dmd;
    register	int			rop, fbi;

    /* 
     * Lock in order to get the most up-to-date cliplist.
     */
    pw_lock(gp1->pw, &(gp1->win_size));
    
    ptr = gp1->cur_ptr;
    dmd = gp1_d(gp1->pw->pw_clipdata->pwcd_prmulti);
    fbi = dmd->cg2_index;		/* frame buffer index */
    rop = PIX_SRC | PIX_COLOR(color);	
    prl = gp1->pw->pw_clipdata->pwcd_prl;
    while (prl) {			/* for each clip list rect */
	if (IS_NO_ROOM(gp1, 11 * sizeof(short))) {
	    Dev_gp1_flush(gp1, TRUE);
	    ptr = gp1->cur_ptr;
	}
	*ptr++ = GP1_PRROPNF | (gp1->plane_mask & 0XFF);
	*ptr++ = fbi; 			/* frame buffer index */
	*ptr++ = rop;			/* rop and color */
	dmd = gp1_d(prl->prl_pixrect);	/* clip pixrect */
	*ptr++ = dmd->cgpr_offset.x;	/* X, Y origin of rect */
	*ptr++ = dmd->cgpr_offset.y;
	*ptr++ = prl->prl_pixrect->pr_size.x; /* X, Y size of rect */
	*ptr++ = prl->prl_pixrect->pr_size.y;
	*ptr++ = 0;			/* w/r to clip pixrect */
	*ptr++ = 0;
	*ptr++ = prl->prl_pixrect->pr_size.x;
	*ptr++ = prl->prl_pixrect->pr_size.y;
	prl = prl->prl_next;			/* go to next one */
	gp1->bytes_left -= 11 * sizeof(short);
    }
    gp1->cur_ptr = ptr;
    
    /* 
     * Since we have a lock, go ahead and flush.
     */
    Dev_gp1_flush(gp1, TRUE);    
    pw_unlock(gp1->pw);
}
    


void
Dev_gp1_setvp_3d(gp1, scale, offset)
GP1		*gp1;
Point3df	*scale, *offset;
{   
    register	short		*ptr;
    
    MAKE_ROOM(gp1, (sizeof(short) + (6 * sizeof(float))));
    
    ptr = gp1->cur_ptr;
    *(ptr)++ = GP1_SETVWP_3D;
    SHORTS_GET_FLOAT(ptr, &(scale->x));
    SHORTS_GET_FLOAT(ptr, &(offset->x));
    SHORTS_GET_FLOAT(ptr, &(scale->y));
    SHORTS_GET_FLOAT(ptr, &(offset->y));
    SHORTS_GET_FLOAT(ptr, &(scale->z));
    SHORTS_GET_FLOAT(ptr, &(offset->z));
    gp1->cur_ptr = ptr;
    
    Dev_gp1_flush(gp1, TRUE);
    
    gp1->vp_scale.x = scale->x;
    gp1->vp_scale.y = scale->y;
    gp1->vp_scale.z = scale->z;
    
    gp1->vp_offset.x = offset->x;
    gp1->vp_offset.y = offset->y;
    gp1->vp_offset.z = offset->z;
}



void
Dev_gp1_setvp_2d(gp1, scale, offset)
GP1		*gp1;
Point2df	*scale, *offset;
{   
    register	short		*ptr;
    
    MAKE_ROOM(gp1, (sizeof(short) + (4 * sizeof(float))));
    
    ptr = gp1->cur_ptr;
    *(ptr)++ = GP1_SETVWP_2D;
    SHORTS_GET_FLOAT(ptr, &(scale->x));
    SHORTS_GET_FLOAT(ptr, &(offset->x));
    SHORTS_GET_FLOAT(ptr, &(scale->y));
    SHORTS_GET_FLOAT(ptr, &(offset->y));
    gp1->cur_ptr = ptr;
    
    /* Dev_gp1_flush(gp1, TRUE); */
        
    gp1->vp_scale.x = scale->x;
    gp1->vp_scale.y = scale->y;
    gp1->vp_offset.x = offset->x;
    gp1->vp_offset.y = offset->y;
}
