#ifndef lint
static	char sccsid[] = "@(#)demo_2df.c 1.2 86/10/07 SMI";
#endif

/*
 * Copyright (c) 1986 by Sun Microsystems, Inc.
 */


#include <stdio.h>
#include <suntool/sunview.h>
#include <suntool/canvas.h>
#include <suntool/panel.h>
#include <suntool/icon_load.h>
#include <math.h>
#include "dev_gp1_internal.h"
#include "demo_util.h"

/* 
 * The purpose of this program is to demonstrate how to
 * use the 2d floating point funtions of the DEV_GP1 package.
 * It is fairly straight-forward to use the 3d versions of the functions;
 * in most case, you need just add a Z value.
 * 
 * The program draws a spiral which shows how to use the DEV_GP1 move and
 * draw functions. A polygon with a hole is draw either with a texture or
 * a solid color to show how to use DEV_GP1 to draw a polygon with more than
 * one boundry and how to fill it with a texture.
 * 
 * In some cases, the DEV_GP1 operation can be preformed
 * by a function call or a macro. The code in the program does it both ways
 * as an example of how each form is used. The variable "use_macros" is used to
 * tell the program which to do. This is only a means for a demostration;
 * in your application, you will most likely use only the function or
 * macro version a function but not both.
 * 
 * This program creates a SunView tool with 3 subwindows. The top two are
 * Panels though which the user controls the action of the program; the 
 * bottom one is a Canvas which is where the program draws.  The top Panel
 * is called the "Control Panel"; it contains 5 buttons (General Attributes,
 * Line Attributes, 2df Vectors, 2df Polygon, and Transform) which control
 * the operation preformed by the program. The other Panel is called 
 * the "Action Panel" and it contains Panel items specific to each action.
 * The Actions are:
 * 	General Attributes:
 * 		drawing color		- the color used to draw the polygon
 * 					  or the spiral.
 * 		use macros		- controls where functions or macros
 * 					  are used to preform the operation.
 * 	Line Attributes:
 * 		line width		- the width of the line in pixels.
 * 		line style		- one of SOLID, DASHED or DOTTED.
 * 
 * 	2df Vectors
 * 		Spiral Size		- how many points are in the spiral.
 * 		Number of reps		- how many repititions to do.
 * 		draw			- draw a spiral made of vectors.
 * 
 * 	2df Polygon
 * 		Fill style		- fill a polygon with a solid color
 * 					  or a texture.
 * 		Texture file		- name of the file which has the
 * 					  the texture -- works with icons.
 * 		draw 			- draw the polygon.
 * 
 * 	Transform
 * 		Transform Vectors	- use the GP to transform vectors.
 * 		Transform Polygons	- "   "   "  "  "         polygons.
 * 
 */



static	Pixwin		*canvas_pw;
static	GP1_handle	gp1;

static	Panel		action_panel;
static	void		new_action();

static	int		use_macros = 0;

#define	MAX_PTS		2000
static	Point2df	points[MAX_PTS];
static	int		spiral_size = 50;

/* 
 * Each action is represented by a list of Panel items. When an action button
 * is pressed any previous items are undisplayed from the action panel and
 * the items corresponding to the (new) action are displayed. To facilitate
 * the displaying and undisplaying of items, the items for each action
 * are kept in an Item_list structure which contains a list of the items
 * and the number of items in the list.
 */

typedef	struct	_Item_list {
    int		n_items;
    Panel_item	*items;
} Item_list;

static	Item_list	*cur_items = NULL;

#define	ord(e)		((int)(e))



/* 2df vector info */
typedef enum {
    V2DF_MSG, 
    V2DF_SPIRAL, 
    V2DF_REPS, 
    V2DF_DRAW, 
    V2DF_TOTAL
} V2df_item_list_type;
    
#define	v2df_msg_item		v2df_items_list[ord(V2DF_MSG)]
#define	v2df_spiral_size_item	v2df_items_list[ord(V2DF_SPIRAL)]
#define	v2df_n_reps_item	v2df_items_list[ord(V2DF_REPS)]
#define	v2df_draw_item		v2df_items_list[ord(V2DF_DRAW)]

static	Panel_item	v2df_items_list[ord(V2DF_TOTAL)];

static	int		n_reps = 50;
static	void		v2df_draw();

static	Item_list	v2df_items = {
    ord(V2DF_TOTAL), 
    v2df_items_list
};



/* Line Attributes */
typedef	enum {
    LINE_ATTR_MSG, 
    LINE_ATTR_WIDTH, 
    LINE_ATTR_STYLE, 
    LINE_ATTR_TOTAL, 
}Line_attr_item_list_type;

#define	line_attr_msg_item	line_attr_items_list[ord(LINE_ATTR_MSG)]
#define	line_attr_width_item	line_attr_items_list[ord(LINE_ATTR_WIDTH)]
#define	line_attr_style_item	line_attr_items_list[ord(LINE_ATTR_STYLE)]

static	Panel_item	line_attr_items_list[ord(LINE_ATTR_TOTAL)];

static	void		line_attr_width();
static	void		line_attr_style();
static	void		init_line_attrs();

static	Item_list	line_attr_items = {
    ord(LINE_ATTR_TOTAL), 
    line_attr_items_list
};



/* General Attributes */
typedef	enum {
    GEN_ATTR_MSG, 
    GEN_ATTR_MACROS, 
    GEN_ATTR_COLOR, 
    GEN_ATTR_TOTAL, 
}Gen_attr_item_list_type;

#define	gen_attr_msg_item	gen_attr_items_list[ord(GEN_ATTR_MSG)]
#define	gen_attr_macros_item	gen_attr_items_list[ord(GEN_ATTR_MACROS)]
#define	gen_attr_color_item	gen_attr_items_list[ord(GEN_ATTR_COLOR)]

static	Panel_item	gen_attr_items_list[ord(GEN_ATTR_TOTAL)];

static	void		gen_attr_macros();
static	void		gen_attr_color();

static	Item_list	gen_attr_items = {
    ord(GEN_ATTR_TOTAL), 
    gen_attr_items_list
};




/* Polygon info */
typedef	enum {
    PGON_MSG, 
    PGON_FILL, 
    PGON_TEX_FILE, 
    PGON_DRAW, 
    PGON_TOTAL, 
}Pgon_item_list_type;

#define	pgon_msg_item		pgon_items_list[ord(PGON_MSG)]
#define	pgon_fill_item		pgon_items_list[ord(PGON_FILL)]
#define	pgon_tex_file_item	pgon_items_list[ord(PGON_TEX_FILE)]
#define	pgon_draw_item		pgon_items_list[ord(PGON_DRAW)]

static	Panel_item	pgon_items_list[ord(PGON_TOTAL)];

static	void		pgon_fill();
static	void		pgon_draw();
static	Item_list	pgon_items = {
    ord(PGON_TOTAL), 
    pgon_items_list
};


/* Transformation */

typedef	enum {
    XFORM_MSG, 
    XFORM_VECTORS, 
    XFORM_POLYGONS, 
    XFORM_TOTAL, 
}Xform_item_list_type;

#define	xform_msg_item		xform_items_list[ord(XFORM_MSG)]
#define	xform_vectors_item	xform_items_list[ord(XFORM_VECTORS)]
#define	xform_polygon_item	xform_items_list[ord(XFORM_POLYGONS)]

static	Panel_item	xform_items_list[ord(XFORM_TOTAL)];

static	void		xform_vectors();
static	void		xform_polygon();

static	Item_list	xform_items = {
    ord(XFORM_TOTAL), 
    xform_items_list
};





main(argc, argv)
int	argc;
char	*argv[];
{
    		Frame		frame;			/* window frame */
    		Canvas		canvas;			/* graphics canvas */
    unsigned	char		cmap_red[4];		/* color values */
    unsigned	char		cmap_green[4];
    unsigned	char		cmap_blue[4];
    struct	cms_map		cmap;
    struct	colormapseg 	cms;
    
    frame = window_create(NULL, FRAME, 
			  FRAME_LABEL, "Dev_gp1 2df test", 
			  FRAME_ARGC_PTR_ARGV, &argc, argv,
			  0);
    init_panels(frame);
    
    /* 
     * Set default cms so canvas will have 4 colors: black, red, green and blue
     */
    cmap_red[0] = 0;	cmap_green[0] = 0;	cmap_blue[0] = 0;
    cmap_red[1] = 255;	cmap_green[1] = 0;	cmap_blue[1] = 0;
    cmap_red[2] = 0;	cmap_green[2] = 255;	cmap_blue[2] = 0;
    cmap_red[3] = 0;	cmap_green[3] = 0;	cmap_blue[3] = 255;
    
    cms.cms_size = 4;
    cms.cms_addr = 0;
    strcpy(cms.cms_name,"dev_gp1");
    cmap.cm_red = cmap_red;
    cmap.cm_green = cmap_green;
    cmap.cm_blue = cmap_blue;
    pw_setdefaultcms(&cms, &cmap);
    
    canvas = window_create(frame, CANVAS,
			   CANVAS_RETAINED,	FALSE,
			   CANVAS_RESIZE_PROC,	resize_func, 
			   0);
    if ((frame == NULL) || (canvas == NULL)) {
	fprintf(stderr, "can't create canvas.\n");
	exit(-1);
    }
    
    canvas_pw = canvas_pixwin(canvas);
    gp1 = init_gp1(canvas_pw);
    if (gp1 == NULL) {
	fprintf(stderr, "can't create gp1.\n");
	exit(-1);
    }
    
    generate_spiral(spiral_size);
    
    window_main_loop(frame);
    
/*     Dev_gp1_destroy(gp1); */
    
    exit(0);
}


/* 
 * This function is is not used within the program. However, it does
 * cause Dev_gp1_print to be loaded which is quite handy as it can then
 * be called from dbx.
 */

print_it() 
{   
    Dev_gp1_print(gp1);
}




init_panels(frame)
Frame	frame;
{   
    
    init_control_panel(frame);
    
    
    action_panel = window_create(frame, PANEL, 
				 PANEL_PAINT, 		PANEL_NONE, 
				 0);
    
    init_v2df_panel(action_panel);
    init_gen_attr_panel(action_panel);
    init_line_attr_panel(action_panel);
    init_pgon_panel(action_panel);
    init_xform_panel(action_panel);
    
    window_fit_height(action_panel);
}


/* 
 * Create the buttons in the Control Panel. Store the list of items 
 * associated with each button in the CLIENT_DATA of the button. When
 * the button is pressed, we will get this list and display the items.
 */

#define	CREATE_BUTTON(panel, str, x, y, proc)				\
 	panel_create_item((panel), PANEL_BUTTON, 			\
		  PANEL_ITEM_X,		PANEL_CU((x)), 			\
		  PANEL_ITEM_Y,		PANEL_CU((y)), 			\
		  PANEL_LABEL_IMAGE,					\
		  panel_button_image((panel), 				\
				     (str), strlen((str))+1, NULL),  	\
		  PANEL_NOTIFY_PROC,	(proc),				\
		  0);



init_control_panel(frame)
Frame	frame;
{   
    Panel	panel;
    Panel_item	item;
    
    panel = window_create(frame, PANEL, 
			  WIN_ROWS, 	2, 
			  0);
    item = CREATE_BUTTON(panel, "General Attrs", 0, 1, new_action); 
    panel_set(item, PANEL_CLIENT_DATA, (char *)&gen_attr_items, 0);
    
    item = CREATE_BUTTON(panel, "Line Attrs", 16, 1, new_action); 
    panel_set(item, PANEL_CLIENT_DATA, (char *)&line_attr_items, 0);
    
    item = CREATE_BUTTON(panel, "2df Vectors", 30, 1, new_action); 
    panel_set(item, PANEL_CLIENT_DATA, (char *)&v2df_items, 0);
    
    item = CREATE_BUTTON(panel, "2df Polygon", 45, 1, new_action); 
    panel_set(item, PANEL_CLIENT_DATA, (char *)&pgon_items, 0);    

    item = CREATE_BUTTON(panel, "Transform", 60, 1, new_action); 
    panel_set(item, PANEL_CLIENT_DATA, (char *)&xform_items, 0);    
}


/* 
 * This routine is called when a button is pressed in the Control Panel.
 * It undisplays the current items (i.e. those belonging to the last button)
 * and puts up the items for the new button.
 */

static
void
new_action(item)
Panel_item	item;
{   
    register	int		i;
    		Item_list	*items;
    
    items = (Item_list *) panel_get(item,  PANEL_CLIENT_DATA);
    
    if (cur_items != NULL) {
	for (i = 0; i < cur_items->n_items; i++)
	    panel_set(cur_items->items[i], PANEL_SHOW_ITEM, 0, 0);
    }
    cur_items = items;
    for (i = 0; i < cur_items->n_items; i++)
	panel_set(cur_items->items[i], PANEL_SHOW_ITEM, 1, 0);
    
    panel_paint(action_panel, PANEL_CLEAR);
}



init_gen_attr_panel(panel)
Panel	panel;
{
    char	buff[80];

    
    gen_attr_msg_item = 
	panel_create_item(panel, PANEL_MESSAGE, 
			  PANEL_ITEM_X,	PANEL_CU(0), 
			  PANEL_ITEM_Y,	PANEL_CU(0), 
			  PANEL_LABEL_STRING, "General Attributes", 
			  PANEL_SHOW_ITEM, 	0,
			  0);
    
     gen_attr_color_item = 
	 panel_create_item(panel, PANEL_CYCLE, 
			   PANEL_ITEM_X,	PANEL_CU(0), 
			   PANEL_ITEM_Y,	PANEL_CU(1), 
			   PANEL_LABEL_STRING,	"Drawing Color", 
			   PANEL_CHOICE_STRINGS, "0", "1", "2", "3", 0, 
			   PANEL_VALUE, 1, 
			   PANEL_SHOW_ITEM, 	0,
			   PANEL_NOTIFY_PROC, 	gen_attr_color,
			   0);
	 
     gen_attr_macros_item = 
	 panel_create_item(panel, PANEL_CYCLE, 
			   PANEL_ITEM_X,	PANEL_CU(20), 
			   PANEL_ITEM_Y,	PANEL_CU(1), 
			   PANEL_LABEL_STRING,	"Use Macros", 
			   PANEL_CHOICE_STRINGS, "No", "Yes", 0, 
			   PANEL_VALUE, 	0, 
			   PANEL_SHOW_ITEM, 	0,
			   PANEL_NOTIFY_PROC, 	gen_attr_macros, 
			   0);
	 
}




static
void
gen_attr_color()
{   
    int		color;
    
    color = (int)panel_get_value(gen_attr_color_item);
    Dev_gp1_set_color(gp1, color);
} 

static
void
gen_attr_macros()
{   
    use_macros = (int)panel_get_value(gen_attr_macros_item);
}



init_line_attr_panel(panel)
Panel	panel;
{
    char	buff[80];

    
    line_attr_msg_item = 
	panel_create_item(panel, PANEL_MESSAGE, 
			  PANEL_ITEM_X,	PANEL_CU(0), 
			  PANEL_ITEM_Y,	PANEL_CU(0), 
			  PANEL_LABEL_STRING, "Line Attributes", 
			  PANEL_SHOW_ITEM, 	0,
			  0);
    
    line_attr_width_item = 
	panel_create_item(panel, PANEL_CYCLE, 
			  PANEL_ITEM_X,		PANEL_CU(0), 
			  PANEL_ITEM_Y,		PANEL_CU(1), 
			  PANEL_LABEL_STRING,	"Line Width", 
			  PANEL_CHOICE_STRINGS, "1", "2", "3","4", "5", "6", 
			  	"8", "9", "10", 0,
			  PANEL_SHOW_ITEM, 	0,
			  PANEL_NOTIFY_PROC, 	line_attr_width, 
			  0);
    
    line_attr_style_item =
	panel_create_item(panel, PANEL_CYCLE, 
			  PANEL_ITEM_X,		PANEL_CU(20), 
			  PANEL_ITEM_Y,		PANEL_CU(1), 
			  PANEL_LABEL_STRING, 	"Line Style", 
			  PANEL_CHOICE_STRINGS, "Solid", "Dashed", "Dotted", 0,
			  PANEL_SHOW_ITEM, 	0,
			  PANEL_NOTIFY_PROC, 	line_attr_style, 
			  0);
	
	
	 
    init_line_attrs();
}



/* 
 * Define 3 line styles: Solid, Dashed and Dotted. Each style needs
 * a options structure and a pattern. Solid has a NULL pattern and option
 * structure. Dashed and Dotted have separate patterns but use the same
 * options.
 */

typedef struct {
    struct	pr_texture_options	ops;
    		short			*pat;
} Style;

static	short	pat_solid[] = {0};
static	short	pat_dash[] = {6, 3, 0};
static	short	pat_dot[] = {2, 2, 0};

static  struct	pr_texture_options	texture_ops = {
    1, 		/* draw start point */
    1, 		/* draw end point */
    0, 		/* balanced */
    0, 		/* do angle correction */
    0, 		/* reserved */
    0, 		/* reserved */
    0, 		/* reserved */
    0, 		/* reserved */
    0 		/* reserved */
};

static  struct	pr_texture_options	solid_ops = {0};

static	Style	styles[3];



static
void
init_line_attrs()
{   
    /* have to do this initialization at runtime... */
    
    styles[0].ops = solid_ops;
    styles[0].pat = pat_solid;
    
    styles[1].ops = texture_ops;
    styles[1].pat = pat_dash;
    
    styles[2].ops = texture_ops;
    styles[2].pat = pat_dot;
}

static
void
line_attr_width()
{   
    int		width;
    
    width = (int)panel_get_value(line_attr_width_item) + 1;
    Dev_gp1_set_line_width(gp1, width);
} 

static
void
line_attr_style()
{   
    int		style;    
    
    style = (int)panel_get_value(line_attr_style_item);
    Dev_gp1_set_line_texture(gp1, 
			     styles[style].pat, 
			     0, 
			     styles[style].ops);
}



init_v2df_panel(panel)
Panel	panel;
{
    char	buff[80];

    
    v2df_msg_item = panel_create_item(panel, PANEL_MESSAGE, 
				      PANEL_ITEM_X,	PANEL_CU(0), 
				      PANEL_ITEM_Y,	PANEL_CU(0), 
				      PANEL_LABEL_STRING, "2D flt vectors", 
				      PANEL_SHOW_ITEM, 	0, 
				      0);
    
    v2df_spiral_size_item = 
	panel_create_item(panel, PANEL_TEXT, 
			  PANEL_ITEM_X,			PANEL_CU(0), 
			  PANEL_ITEM_Y,			PANEL_CU(1), 
			  PANEL_VALUE_DISPLAY_LENGTH,	4,
			  PANEL_LABEL_STRING,		"Spiral Size:", 
			  PANEL_VALUE,	  sprintf(buff, "%d", spiral_size), 
			  PANEL_SHOW_ITEM, 	0,
			  0);
    
    v2df_n_reps_item = 
	panel_create_item(panel, PANEL_TEXT, 
			  PANEL_ITEM_X,			PANEL_CU(20), 
			  PANEL_ITEM_Y,			PANEL_CU(1), 
			  PANEL_VALUE_DISPLAY_LENGTH,	4,
			  PANEL_LABEL_STRING,		"Number of reps:", 
			  PANEL_VALUE,		sprintf(buff, "%d", n_reps), 
			  PANEL_SHOW_ITEM, 	0,
			  0);
	
    v2df_draw_item = CREATE_BUTTON(panel, "Draw", 45, 1, v2df_draw);
    panel_set(v2df_draw_item, 
	      PANEL_SHOW_ITEM, 	0,
	      0);
	
}


/* 
 * Draw a spiral as an example of how to draw vectors. For each repitition,
 * load a matrix which scales by values <= 1. This just shows how to
 * load a matrix; the resulting image is mildly interesting...
 */

static
void
v2df_draw()
{   
    register	int	i;
    register	float	scale;
    		int	new_size;
    
    sscanf((char *)panel_get_value(v2df_n_reps_item), "%d", &n_reps);
    
    sscanf((char *)panel_get_value(v2df_spiral_size_item), "%d", &new_size);
    if (new_size != spiral_size) {
	if (new_size > MAX_PTS) {
	    new_size = MAX_PTS;
	}
	generate_spiral(new_size);
	spiral_size = new_size;
    }
    
    Dev_gp1_clear(gp1, 0);
    Dev_gp1_flush(gp1, 1);

    scale = 1.0;
    for (i = 0; i < n_reps; i++) {
	load_scale_matrix(gp1, scale);
	draw_spiral();
	scale -= 0.02;
	if (scale <= -1.0) {   
	    scale = 1.0;
	    Dev_gp1_clear(gp1, 0);
	}
	Dev_gp1_flush(gp1, 1);
    }
    
}




static	Matrix2df	scale_mat = {
    {   1.0, 0.0 }, 
    {   0.0, 1.0 }, 
    {   0.0, 0.0 }
};

/* 
 * Take the scale factor and generate a 2d scale matrix. Since we
 * know all the points are in the range [-1, 1], we can just apply
 * the scale matrix directly to them.
 */

load_scale_matrix(gp1, scale)
GP1_handle	gp1;
float		scale;
{
    scale_mat[0][0] = scale_mat[1][1] = scale;
    Dev_gp1_set_matrix_2d(gp1, 0, scale_mat);
}



#define PI		3.141592653589

generate_spiral(n_pts)
int	n_pts;
{   
    
    		double	theta, d_theta;
    		double	radius;
    		double	d_radius;
    register	int	i;
    
    
    d_theta = (2.0 * PI) / 25.0;
    d_radius = 1.0/n_pts;
    for (i = 0, theta = 0.0, radius = 1.0;
	 i < n_pts; 
	 i++,  theta += d_theta, radius -= d_radius) {
	points[i].x = radius * cos(theta);
	points[i].y = radius * sin(theta);
    }
}



draw_spiral()
{   
    register	int		i;
    register	Point2df	*pt;
    	int	magic;
	


    pt = points;
    if (use_macros) {
	DEV_GP1_MOVE_2DF(gp1, pt->x, pt->y);
	pt++;
	for(i = 1; i < spiral_size; i++) {
	    DEV_GP1_DRAW_2DF(gp1, pt->x, pt->y);
	    pt++;
	}
    }
    else {
	Dev_gp1_move_2df(gp1, pt++);
	for(i = 1; i < spiral_size; i++)
	    Dev_gp1_draw_2df(gp1, pt++);
    }    

}



/* 
 * Polygon functions.
 */

init_pgon_panel(panel)
Panel	panel;
{   
    pgon_msg_item = panel_create_item(panel, PANEL_MESSAGE, 
				      PANEL_ITEM_X,	PANEL_CU(0), 
				      PANEL_ITEM_Y,	PANEL_CU(0), 
				      PANEL_LABEL_STRING, "2D flt Polygon", 
				      PANEL_SHOW_ITEM, 	0, 
				      0);
    
    pgon_fill_item = 
	panel_create_item(panel, PANEL_CYCLE, 
			  PANEL_ITEM_X,		PANEL_CU(0), 
			  PANEL_ITEM_Y,		PANEL_CU(1), 
			  PANEL_LABEL_STRING,	"Fill Style", 
			  PANEL_CHOICE_STRINGS, "Solid", "Texture", 0, 
			  PANEL_SHOW_ITEM, 	0,
			  PANEL_NOTIFY_PROC, 	pgon_fill, 
			  0);
   pgon_tex_file_item = 
       panel_create_item(panel, PANEL_TEXT, 
			 PANEL_ITEM_X,		PANEL_CU(25), 
			 PANEL_ITEM_Y,		PANEL_CU(0), 
			 PANEL_LABEL_STRING,	"Texture file:", 
			 PANEL_VALUE_DISPLAY_LENGTH,	40,
			 PANEL_SHOW_ITEM, 	0,
			 PANEL_VALUE,	"/usr/include/images/traffic.icon", 
			 0);
   pgon_draw_item = CREATE_BUTTON(panel, "Draw", 25, 1, pgon_draw);
   panel_set(pgon_draw_item, 
	     PANEL_SHOW_ITEM, 	0,
	     0);
}



static	int	pgon_tex_used = 0;

static
void
pgon_fill()
{   
    if ((int)panel_get_value(pgon_fill_item) == 1)
	pgon_tex_used = 1;
    else
	pgon_tex_used = 0;
}



Point2df	poly_pts[] = {			/* a box with a hole in it */
    {   -0.5, -0.5}, 	/* outer bound */
    {   -0.5,  0.5}, 
    {    0.5,  0.5}, 
    {    0.5, -0.5}, 
    {   -0.2, -0.2}, 	/* inner bound */
    {   -0.2,  0.2}, 
    {    0.2,  0.2}, 
    {    0.2, -0.2}
};

static	int	bounds[] = {4, 4};
static	int	n_bounds = sizeof(bounds)/sizeof(int);
static	int	n_poly_pts = 8;

static
void
pgon_draw()
{   
    Pixrect	*pr;
    char	err_msg[IL_ERRORMSG_SIZE];
    int		fill_style;
    int		i, j;
    int		room_for;
    Point2df	*pt;

    if (pgon_tex_used) {
	pr = icon_load_mpr(panel_get_value(pgon_tex_file_item), err_msg);
	if (!pr)
	    return;		/* couldn't load file... */
	if (Dev_gp1_set_polygon_texture(gp1, pr) == -1)
	    return;
	
	Dev_gp1_set_texture_org_screen(gp1, 0, 0);	
	fill_style = GP1_SHADE_TEX;
    }
    else 
	fill_style = GP1_SHADE_CONSTANT;
    
    load_scale_matrix(gp1, 1.0);
    Dev_gp1_clear(gp1, 0);
    
    if (use_macros) {
	DEV_GP1_POLY_2DF_INIT(gp1, n_bounds, n_poly_pts, room_for, fill_style);
	if (!room_for) {
	    fprintf(stderr, "cannot fit the polygon!");
	    return;
	}
	pt = poly_pts;
	for (j = 0; j < n_bounds; j++) {
	    DEV_GP1_POLY_2DF_ADD_BOUND(gp1);
	    for (i = 0; i < bounds[j]; i++, pt++) {
		DEV_GP1_POLY_2DF_ADD_VERTEX(gp1, pt->x, pt->y);
	    }
	}
    }
    else {
	Dev_gp1_polygon_2df(gp1, n_bounds, bounds, poly_pts, fill_style);
    }
    
    Dev_gp1_flush(gp1, 1);
}


init_xform_panel(panel)
Panel	panel;
{   
    
    xform_msg_item = panel_create_item(panel, PANEL_MESSAGE, 
				       PANEL_ITEM_X,	PANEL_CU(0), 
				       PANEL_ITEM_Y,	PANEL_CU(0), 
				       PANEL_LABEL_STRING, 
				        "Tranform and read back", 
				       PANEL_SHOW_ITEM, 	0, 
				       0);
   xform_vectors_item = CREATE_BUTTON(panel, "Transform Vectors", 
				      0, 1, xform_vectors);
   panel_set(xform_vectors_item, 
	     PANEL_SHOW_ITEM, 	0,
	     0);
   
   xform_polygon_item = CREATE_BUTTON(panel, "Transform Polygons", 
				      25, 1, xform_polygon);
   panel_set(xform_polygon_item, 
	     PANEL_SHOW_ITEM, 	0,
	     0);
}



/* 
 * This function demostrates how to use the Dev_gp1 to transform vectors.
 * 
 * Draw 2d World Coordinate vectors using the GP. Next have the GP
 * transform (and clip)  the vectors to screen space 
 * and read back the vectors. Then, using pw_vector, draw the transformed
 * vectors, in a different color, and if all goes well, the transformed
 * vectors will hit the same pixels as the WC vectors.
 */

#define NUM_VECS	20

static
void
xform_vectors()
{   
    		Point2df	vf1[NUM_VECS];	/* floating point world  */
    		Point2df	vf2[NUM_VECS];	/* coordinate vectors */
    		Point2di	vi1[NUM_VECS];	/* xformed integer screen */
    		Point2di	vi2[NUM_VECS];	/* coordinate vectors */
    register	float		x, incr;
		int		room_for;	/* number of vectors that
						 * can be xformed at one time.
						 */
		int		clip;		/* result of the clipping
						 * operation.
						 */
    register	int		i;
    		int		ox, oy;
    
    
    /* 
     * Generate the vectors: vertical lines. 
     */
    incr = 1.6 / NUM_VECS;
    for (x = -0.8, i = 0; i < NUM_VECS; i++, x += incr) {
	vf1[i].x = x;
	vf1[i].y = -0.8;
	
	vf2[i].x = x;
	vf2[i].y = 0.8;
    }
    
    load_scale_matrix(gp1, 1.0);
    Dev_gp1_clear(gp1, 0);
    DEV_GP1_SET_COLOR(gp1, 1);
    
    for (i = 0; i < NUM_VECS; i++) {
	DEV_GP1_MOVE_2DF(gp1, vf1[i].x, vf1[i].y);
	DEV_GP1_DRAW_2DF(gp1, vf2[i].x, vf2[i].y);
    }
    
    Dev_gp1_flush(gp1, 1);
    sleep(2);		/* give the operator a chance to see the vectors */
    
    DEV_GP1_XVECTOR_2DF_BEGIN(gp1, room_for);
    if (room_for < NUM_VECS) {
	fprintf(stderr, "oh no! not enough room to transform %d vectors\n", 
	       NUM_VECS);
	return;
    }
    for (i = 0; i < NUM_VECS; i++) {
	DEV_GP1_XVECTOR_2DF_ADD_VECTOR(vf1[i].x, vf1[i].y,
				       vf2[i].x, vf2[i].y);
    }
    
    DEV_GP1_XVECTOR_2DF_XFORM(gp1);
    
    for (i = 0; i < NUM_VECS; i++) {
	DEV_GP1_XVECTOR_2DF_GET_VECTOR(gp1,clip,
				       vi1[i].x, vi1[i].y, 
				       vi2[i].x, vi2[i].y);
    }
    DEV_GP1_XVECTOR_2DF_END;
    
    
    ox = gp1->win_org_x;
    oy = gp1->win_org_y;
    for (i = 0; i < NUM_VECS; i++) {
	pw_vector(gp1->pw, 
		  (vi1[i].x - ox), 
		  (vi1[i].y - oy), 
		  (vi2[i].x - ox), 
		  (vi2[i].y - oy), 
		  PIX_SRC, 2);
    }
    
}


/* 
 * Draw the same polygon as 2df Polygon uses. Then transform it and read
 * it back. Then draw it using pw. 
 */

static
void
xform_polygon()
{   
    		int		tmp_tex_used;
    register	int		i, j;
    register	Point2df	*pt;
    		int		room_for;
		int		ox, oy;
    		int		num_pts;
    		int		new_n_bounds;
    		int		new_bounds[2];
    		struct pr_pos	new_pts[8];
    register	struct pr_pos	*new_pt;
    
    
    
    tmp_tex_used = pgon_tex_used;
    pgon_tex_used = 0;	/* draw a solid polygon */
    
    Dev_gp1_clear(gp1, 0);
    DEV_GP1_SET_COLOR(gp1, 1);
    
    pgon_draw();		/* draw a polygon. */
    sleep(2);			/* admire it. */
    
    DEV_GP1_XPOLY_2DF_BEGIN(gp1, n_bounds, n_poly_pts, room_for);
    if (!room_for) {
	fprintf(stderr, 
		"oh no! not enough room to xform polygon with %d verts\n", 
		n_poly_pts);
	return;
    }
    
    pt = poly_pts;
    for (j = 0; j < n_bounds; j++) {
	DEV_GP1_XPOLY_2DF_ADD_BOUND(gp1);
	for (i = 0; i < bounds[j]; i++, pt++) {
	    DEV_GP1_XPOLY_2DF_ADD_VERTEX(gp1, pt->x, pt->y);
	}
    }
    
    DEV_GP1_XPOLY_2DF_XFORM(gp1, new_n_bounds);
    if (new_n_bounds != n_bounds) {
	fprintf(stderr, 
		"xformed poly has %d new bounds!\n", new_n_bounds);
	return;
    }
    
    ox = gp1->win_org_x;
    oy = gp1->win_org_y;
    new_pt = new_pts;
    for (j = 0; j < new_n_bounds; j++) {
	DEV_GP1_XPOLY_2DF_GET_BOUND(gp1, num_pts);    
	for (i = 0; i < num_pts; i++) {
	    DEV_GP1_XPOLY_2DF_GET_VERTEX(new_pt->x, new_pt->y);
	    new_pt->x -= ox;
	    new_pt->y -= oy;
	    new_pt++;
	}
    }
    
    DEV_GP1_XPOLY_2DF_END;
    
    pw_polygon_2(gp1->pw, 0, 0, n_bounds, bounds, new_pts, 
		 (PIX_SRC | PIX_COLOR(2)),  NULL, NULL, NULL);
    
    pgon_tex_used =  tmp_tex_used;
}

    
