#include <param.h>
#include <systm.h>
#include <mount.h>
#include <ino.h>
#include <reg.h>
#include <buf.h>
#include <seg.h>
#include <filsys.h>
#include <dir.h>
#include <signal.h>
#include <user.h>
#include <inode.h>
#include <file.h>
#include <conf.h>
#include <stat.h>
#include <fcntl.h>
#include <errno.h>

extern struct inode inode[];
extern struct user u;
extern char b[];
extern struct mount mount[];

/*
 * the fstat system call.
 */
fstat()
{
	register struct file *fp;
	register struct a
	{
		int	fdes;
		struct stat *sb;
	} *uap;

	uap = (struct a *)u.u_ap;
	fp = getf(uap->fdes);
	if(fp == NULL)
		return;
	stat1(fp->f_inode, uap->sb);
}

/*
 * the stat system call.
 */
stat()
{
	register struct inode *ip;
	register struct a
	{
		char	*fname;
		struct stat *sb;
	} *uap;

	uap = (struct a *)u.u_ap;
	ip = namei(uchar, 0);
	if(ip == NULL)
		return;
	stat1(ip, uap->sb);
	iput(ip);
}

/*
 * The basic routine for fstat and stat:
 * get the inode and pass appropriate parts back.
 */
stat1(ip, ub)
register struct inode *ip;
struct stat *ub;
{
	register struct dinode *dp;
	register struct buf *bp;
	struct stat ds;

	iupdat(ip, time, time);
	/*
	 * first copy from inode table
	 */
	ds.st_dev = ip->i_dev;
	ds.st_ino = ip->i_number;
	ds.st_mode = ip->i_mode;
	ds.st_nlink = ip->i_nlink;
	ds.st_uid = ip->i_uid;
	ds.st_gid = ip->i_gid;
	ds.st_rdev = getdevno(ip);
	ds.st_size = ip->i_size;
	/*
	 * next the dates in the disk
	 */
	bp = bread(ip->i_dev, itod(ip->i_number));
	ka5->r[0] = baddr(bp);
	dp = &b;
	dp += itoo(ip->i_number);
	ds.st_atime = dp->di_atime;
	ds.st_mtime = dp->di_mtime;
	ds.st_ctime = dp->di_ctime;
	brelse(bp);
	if (copyout((caddr_t)&ds, (caddr_t)ub, sizeof(ds)) < 0)
		u.u_error = EFAULT;
}

/*
 * the mount system call.
 */
smount()
{
	dev_t dev;
	register struct inode *ip;
	register struct mount *mp;
	struct mount *smp;
	register struct filsys *fp;
	struct buf *bp;
	register struct a
	{
		char	*fspec;
		char	*freg;
		int	ronly;
	} *uap;

	uap = (struct a *)u.u_ap;
	if (! suser())
		return;
	dev = getmdev();
	if(u.u_error)
		return;
	u.u_dirp = (caddr_t)uap->freg;
	ip = namei(uchar, 0);
	if(ip == NULL)
		return;
	if (ip->i_count != 1 || (ip->i_mode & IFMT) != IFDIR)
		goto out;
	smp = NULL;
	for(mp = &mount[0]; mp < &mount[NMOUNT]; mp++)
	{
		if(mp->m_bufp != NULL)
		{
			if(dev == mp->m_dev)
				goto out;
		} else
		if(smp == NULL)
			smp = mp;
	}
	mp = smp;
	if(mp == NULL)
		goto out;
	(*bdevsw[major(dev)].d_open)(dev, uap->ronly ? FREAD : (FREAD|FWRITE));
	if(u.u_error)
		goto out;
	bp = bread(dev, SUPERB);
	if(u.u_error)
	{
		brelse(bp);
		goto out1;
	}
	bp->b_flags |= B_SUPER;
	mp->m_inodp = ip;
	mp->m_dev = dev;
	mp->m_bufp = bp;
	ka5->r[0] = baddr(bp);
	fp = &b;
	fp->s_ilock = 0;
	fp->s_flock = 0;
	fp->s_ronly = (uap->ronly != 0);
	brelse(bp);
	ip->i_flag |= IMOUNT;
	prele(ip);
	if (dev == pipedev && uap->ronly == 0)
		pdevmnt++;
	return;

out:
	u.u_error = EBUSY;
out1:
	iput(ip);
}

/*
 * the umount system call.
 */
sumount()
{
	dev_t dev;
	register struct inode *ip;
	register struct mount *mp;
	struct buf *bp;
	char stillopen;
	register struct a
	{
		char	*fspec;
	};

	if (! suser())
		return;
	dev = getmdev();
	if(u.u_error)
		return;
	xumount(dev);	/* remove unused sticky files from text table */
	update();
	for(mp = &mount[0]; mp < &mount[NMOUNT]; mp++)
		if(mp->m_bufp != NULL && dev == mp->m_dev)
			goto found;
	u.u_error = EINVAL;
	return;

found:
	stillopen = 0;
	for(ip = &inode[0]; ip < &inode[NINODE]; ip++)
	{
		if(ip->i_number != 0 && dev == ip->i_dev)
		{
			u.u_error = EBUSY;
			return;
		}
		else if (ip->i_count && (ip->i_mode & IFMT) == IFBLK &&
		    getdevno(ip) == dev)
			stillopen++;
	}
	ip = mp->m_inodp;
	ip->i_flag &= ~IMOUNT;
	plock(ip);
	iput(ip);
	bp = mp->m_bufp;
	mp->m_bufp = NULL;
	bp->b_flags &= ~B_SUPER;
	brelse(bp);
	if (stillopen == 0)
	{
		register struct filsys *fp;

		ka5->r[0] = baddr(bp);
		fp = &b;
		(*bdevsw[major(dev)].d_close)(dev,
			fp->s_ronly ? FREAD : (FREAD|FWRITE));
		binval(dev);
	}
	if (dev == pipedev)
		pdevmnt = 0;
}

/*
 * Common code for mount and umount.
 * Check that the user's argument is a reasonable
 * thing on which to mount, and return the device number if so.
 */
dev_t
getmdev()
{
	dev_t dev;
	register struct inode *ip;

	ip = namei(uchar, 0);
	if(ip == NULL)
		return(NODEV);
	if((ip->i_mode&IFMT) != IFBLK)
		u.u_error = ENOTBLK;
	dev = getdevno(ip);
	if(major(dev) >= nblkdev)
		u.u_error = ENXIO;
	iput(ip);
	return(dev);
}

/*
 *	dup system call
 *
 *	superseded by fcntl (below), but remains for historical reasons
 */
dup()
{
	register struct file *fp;
	register struct a
	{
		int	fdes;
	} *uap;
	register i;

	uap = (struct a *)u.u_ap;
	fp = getf(uap->fdes);
	if (fp == NULL)
		return;
	if ((i = ufalloc(0)) < 0)
		return;
	u.u_ofile[i] = fp;
	fp->f_count++;
}

/*
 *	The file control system call
 */
fcntl()
{
	register struct file *fp;
	register struct a
	{
		int	fdes;
		int	cmd;
		int	arg;
	} *uap;
	register i;

	uap = (struct a *)u.u_ap;
	if (uap->cmd != F_UNLOCK && (fp = getf(uap->fdes)) == NULL)
		return;		/* only unlock does not need a file desc. */
	switch(uap->cmd)
	{
	case F_DUPFD:
		i = uap->arg;
		if (i < 0 || i >= NOFILE)
			break;
		if ((i = ufalloc(i)) < 0)
			return;
		u.u_ofile[i] = fp;
		fp->f_count++;
		return;

	case F_GETFD:
		u.u_r.r_r0 = (unsigned)u.u_ofile[uap->fdes] & 01;
		return;

	case F_SETFD:
		if (uap->arg)
			(unsigned)u.u_ofile[uap->fdes] |= 01;
		else
			(unsigned)u.u_ofile[uap->fdes] &= ~01;
		return;

	case F_GETFL:
		u.u_r.r_r0 = fp->f_flag;
		return;

	case F_SETFL:
		fp->f_flag &= (FREAD|FWRITE);	/* no-one may alter these */
		if (u.u_uid)
			uap->arg &= ~O_FREE;	/* superuser only */
		fp->f_flag |= uap->arg & ~(FREAD|FWRITE);
		return;

	case F_RLOCK:
		if (fp->f_flag & FREAD)
		{
			lock(fp->f_inode, L_RLOCK);
			return;
		}
		break;

	case F_WLOCK:
		if (fp->f_flag & FWRITE)
		{
			lock(fp->f_inode, L_WLOCK);
			return;
		}
		break;

	case F_UNLOCK:
		unlock();
		return;

	}

	u.u_error = EINVAL;
}

lock(ip, type)
register struct inode *ip;
register char type;
{
	if(u.u_locki != NULL)
		unlock();
	if((ip->i_mode&IFMT) != IFLOK && (ip->i_mode&IFMT) != IFALK)
	{
		u.u_error = EBADF;
		return(1);
	}
	if(access(ip, IWRITE))
	{
		/** Ignore locking if no write permission **/
		u.u_error = 0;
		u.u_lockf = L_PHLOK;
		u.u_locki = ip;
		return 0;
	}
	while(ip->i_lockf & (L_WLOCK|L_WAIT))
	{
		ip->i_lockf |= L_WAIT;
		sleep((caddr_t)&ip->i_lockf, PLOCK);
	}
	switch(type)
	{
	case L_RLOCK:
		ip->i_lockc++;
		ip->i_lockf |= type;
		u.u_lockf = type;
		break;

	case L_WLOCK:
		if(save(u.u_qsav))	/* catch half nelsons */
		{
			if(u.u_error == 0)
				u.u_error = EINTR;
			ip->i_lockf &= ~L_WAIT;
			wakeup((caddr_t)&ip->i_lockf);
			return(1);
		}
		while(ip->i_lockf&L_RLOCK)
		{
			ip->i_lockf |= L_WAIT;
			sleep((caddr_t)&ip->i_lockc, PLOCK);
		}
		ip->i_lockf |= type;
		u.u_lockf = type;
		break;
	}
	u.u_locki = ip;
	return(0);
}

unlock()
{
	register struct inode *ip;

	if((ip = u.u_locki) == NULL)
	{
		u.u_error = EINVAL;
		return;
	}
	if(u.u_lockf == L_RLOCK)
	{
		if(--ip->i_lockc == 0)
		{
			if(ip->i_lockf&L_WAIT)
				wakeup((caddr_t)&ip->i_lockc);
			ip->i_lockf &= ~L_RLOCK;
		}
	}
	else if(u.u_lockf == L_WLOCK)
	{
		if(ip->i_lockf&L_WAIT)
			wakeup((caddr_t)&ip->i_lockf);
		ip->i_lockf = 0;
	}
	u.u_lockf = 0;
	u.u_locki = NULL;
}
