/*
 *	EMUL-8: a pdp8e emulator.
 *
 *	Author:
 *		Bill Haygood
 *		41832 Ernest Road
 *		Loon Lake, WA 99148-9607
 *		Internet: billh@comtch.iea.com
 *		Voice/AnsMach/FAX \
 *			or	   509-233-2555
 *		  Cellular/Pager  /
 *
 *	Copyright 1992, 1993, 1994 by the author with all rights reserved.
 *
 *	Miscellaneous routines.
 */
#include "pdp8.h"
#include "pdp8regs.h"
#include "pdp8eae.h"
#include "pdp8iot.h"
#include "pdp8itab.h"
#include "pdp8opr.h"
#include "pdp8txt.h"
/* -------------------------------------------------------------------- */
/*
 *	Print a header to display the registers.
 */
INT header (INT newline, INT cputrace, INT fpptrace)
{
/*
            1  1 1    2     2                4    5     5    6    6   7   7
0      7    2  5 7    2     8                5    0     6    1    6   0   4
DI PC  Code Instruction  (Dir) ((Indir))    Flags L AC   MQ   SR  EAE SC GTF
  FPC                              INDX STAT FAC
107606 7763 CLA MQL MQA DAD 61234 (23456701) 0010 00000 0000 0000  A  00 F
 */
    chrsout [0] = '\0' ;
    if (newline == 0)
	strcpy (chrsout, "\r\n") ;
    if (cputrace)
    {
	strcat (chrsout, "Flags Bits:  0:Link  1:GTF  2:IReq  3:IInh"
				"  4:IEna  5:UF  6-8:IF  9-11:DF\r\n") ;
	strcat (chrsout, "DI PC  Code Instruction  (Dir) ((Indir))"
					"    Flags L AC   MQ   SR") ;
	if (EAE)
	    strcat (chrsout, "  EAE SC GTF") ;
	strcat (chrsout, "\r\n") ;
    }
    if (fpptrace)
    {
	strcat (&chrsout [strlen (chrsout)],
		"  FPC                              INDX STAT FAC\r\n") ;
    }
    if (tracefile)
	fwrite (chrsout, 1, strlen (chrsout), tracefile) ;
    else
	fprintf (stderr, "%s", chrsout) ;
    return TRUE ;
}
/* -------------------------------------------------------------------- */
/*
 *	Print a header to display the registers.
 */
VOID outline (VOID)
{
/*
            1  1 1    2     2                4    5     5    6    6   7   7
0      7    2  5 7    2     8                5    0     6    1    6   0   4
ID PC  Code Instruction  (Dir) ((Indir))    Flags L AC   MQ   SR  EAE SC GTF
107600 6211 CDF 1                            0010 00000 0000 0000  B  00  F
107601 1776 TADI 7776 (2345)((6712))         0010 06712 0000 0000  B  00  F
107602 7776 SPA SNA SZL CLA OSR HLT          0010 00000 0000 0000  B  00  F
107603 7767 CLA MQL MQA SCA MUY 1234         0010 00000 0000 0000  B  00  F
107604 7765 CLA MQL MQA DST 61234            0010 00000 0000 0000  B  00  F
107606 7763 CLA MQL MQA DAD 61234 (23456701) 0010 00000 0000 0000  B  00  F
 */
    INT i = strlen (chrsout) ;
    while (i < 45)
	chrsout [i++] = ' ' ;
    sprintf (&chrsout[45], "%04o %05o %04o %04o",
			(((AC & 010000) >> 1)
			| (GTF << 10)
			| ((io_flags & int_mask) ? BIT2 : 0)
			| (int_inh << 8)
			| (int_ena << 7)
			| (UF >> 6)
			| (IF >> 9)
			| (DF >> 12)),
			  AC, MQ, SR) ;
    if (EAE)
    {
	sprintf (pdp8exec, "  %c  %02o", EAE, SC) ;
	strcat (chrsout, pdp8exec) ;
	if (EAE == 'B')
	    strcat (chrsout, GTF ? "  T" : "  F") ;
    }
    strcat (chrsout, "\r\n") ;
    if (tracefile)
	fwrite (chrsout, 1, strlen (chrsout), tracefile) ;
    else
	fprintf (stderr, "%s", chrsout) ;
}
/* -------------------------------------------------------------------- */
/*
 *	Store double to UWORD array w.
 */
VOID storefpp (DOUBLE d, UWORD *w)
{
    LONG y0, y1 ;
    INT exp ;

    if (d)
    {
	d = frexp (d, &exp) ;
	*w = exp & 07777 ;
	d = ldexp (d, 31) ;
	y0 = d ;
	*(w + 1) = (y0 >> 20) & 07777 ;
	*(w + 2) = (y0 >> 8) & 07777 ;
	if (fpp_data == EP)
	{
	    if (d -= y0)
	    {
		d = ldexp (d, 28) ;
		y1 = d ;
		*(w + 3) = ((y0 << 4) & 07760) | ((y1 >> 24) & 017) ;
		*(w + 4) = (y1 >> 12) & 07777 ;
		*(w + 5) = y1 & 07777 ;
	    }
	    else
		*(w + 3) = *(w + 4) = *(w + 5) = 0 ;
	}
    }
    else
    {
	*w++ = 0, *w++ = 0, *w = 0 ;
	if (fpp_data == EP)
	    *++w = 0, *++w = 0, *++w = 0 ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Convert a pdp8 fpp floating point datum to double.
 *	w points to fpp exponent word.
 */
DOUBLE fpp2d (UWORD *w)
{
    LONG y0 ;
    ULONG y1 ;
    INT nz ;

    if ((nz = *(w + 1) & 06000) == 02000 || nz == 04000)
    {
	y0 = (*(w + 1) << 20) | (*(w + 2) << 8) ;
	if (fpp_data == EP)
	{
	    y0 |= *(w + 3) >> 4 ;
	    y1 = ((*(w + 3) & 0xf) << 28) | (*(w + 4) << 16)
						| (*(w + 5) << 4) ;
	    if (y0 < 0)
	    {
		y0 = ~y0 ;
		if ((y1 = -y1) == 0)
		    y0++ ;
	    }
	}
	else
	    y1 = 0 ;
	order.d = ldexp ((DOUBLE) y0, -31) ;
	if (y1)
	    order.d += ldexp ((DOUBLE) y1, -63) ;
	if (nz & 04000)
	    order.d = -order.d ;
	if ((nz = *w) & 04000)
	    nz |= ~07777 ;
	order.d = ldexp (order.d, nz) ;
    }
    else
	order.d = 0.0 ;
    return order.d ;
}
/* --------------------------------------------------------------------	*/
/*
 *	Output an FPP datum in decimal format.
 *	e.g. : -3.14159265358979
 *
 *	If 'w' is NULL, output content of the FAC, otherwise
 *	use address supplied.
 */
VOID outfpp (UWORD *w, INT posit)
{
    LONG j ;
    INT i, norm = TRUE ;
/*
 *	Print to 'chrsout' and delete trailing zeroes.
 */
    if ((*(w + 1) & 04000) == 0)
	chrsout [posit++] = ' ' ;
    if (fpp_data == DP)
    {
	j = (((LONG) *(w + 1)) << 12) | *(w + 2) ;
	if (j & 040000000)
	    j |= ~077777777 ;
	posit += sprintf (&chrsout [posit], "%d", j) ;
    }
    else				/* FP || EP			*/
    {
	DOUBLE doub = fpp2d (w) ;
	j = *(w + 1) | *(w + 2) ;
	if (fpp_data == EP)
	    j |= *(w + 3) | *(w + 4) | *(w + 5) ;
	if (j == 0)
	    j = 02000 ;
	else
	{
	    if ((j = *(w + 1)) & 04000)
		j = (- j) & 07777 ;
	}
	if (j & 02000)			/* w is normalized		*/
	{
	    if (!doub || (abs (doub) < 10.0))
	    {
		posit += sprintf (&chrsout [posit], (fpp_data == EP)
					? "%17.15f" : "%8.6f", doub) ;
		for (i = posit - 1 ; i && chrsout [i] == '0' ; i--)
		    chrsout [i] = '\0' ;
	    }
	    else
	    {
		posit += sprintf (&chrsout [posit], (fpp_data == EP)
					? "%18.15e" : "%8.6e", doub) ;
		i = posit - 1 ;
		while (chrsout [i] != 'E' && chrsout [i] != 'e')
		    i-- ;
		j = i - 1 ;
		while (chrsout [j] == '0')
		    j-- ;
		j++ ;
		while (chrsout [j] != '\0')
		    chrsout [j++] = chrsout [i++] ;
	    }
	    posit = strlen (chrsout) ;
	}
	else
	{
	    if (*(w + 1) & 04000)
		posit += sprintf (&chrsout [posit], " ") ;
	    posit += sprintf (&chrsout [posit], "unnormalized") ;
	    norm = FALSE ;
	}
    }
    if (fpp_run && norm)
	sprintf (&chrsout [posit], (fpp_data == DP) ?  "(DP)"
				: (fpp_data == FP) ? "(FP)" : "(EP)") ;
    strcat (chrsout, "\r\n") ;
    if (tracefile)
	fwrite (chrsout, 1, strlen (chrsout), tracefile) ;
    else
	fprintf (stderr, "%s", chrsout) ;
}
/* --------------------------------------------------------------------	*/
/*
 *	Output byte(s) to output file.
 */
#define ULBUFSIZ	(1 << 10)
VOID out_ul (ULONG ulong, INT writeflag)
{
    STATIC UBYTE outbuff [ULBUFSIZ] ;
    STATIC INT outptr = 0 ;

    if (writeflag)
    {
	switch (writeflag & 0xf)
	{
	    case 4 :
		outbuff [outptr++] = ulong >> 24 ;
		outbuff [outptr++] = ulong >> 16 ;
		outbuff [outptr++] = ulong >> 8 ;
		outbuff [outptr++] = ulong ;
		fwrite (outbuff, 1, outptr, pdp8file) ;
		break ;
	    case 3 :
		outbuff [outptr++] = ulong >> 24 ;
		outbuff [outptr++] = ulong >> 16 ;
		outbuff [outptr++] = ulong >> 8 ;
		fwrite (outbuff, 1, outptr, pdp8file) ;
		break ;
	    case 2 :
		outbuff [outptr++] = ulong >> 24 ;
		outbuff [outptr++] = ulong >> 16 ;
		fwrite (outbuff, 1, outptr, pdp8file) ;
		break ;
	    case 1 :
		outbuff [outptr++] = ulong >> 24 ;
		fwrite (outbuff, 1, outptr, pdp8file) ;
		break ;
	    case 0 :
		if (outptr)
		    fwrite (outbuff, 1, outptr, pdp8file) ;
		break ;
	    default :
		break ;
	}
    }
    else
    {
	outbuff [outptr++] = ulong >> 24 ;
	outbuff [outptr++] = ulong >> 16 ;
	outbuff [outptr++] = ulong >> 8 ;
	outbuff [outptr++] = ulong ;
	if (!(outptr &= ULBUFSIZ - 1))
	    fwrite (outbuff, 1, ULBUFSIZ, pdp8file) ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Output a bit stream to a file.
 */
VOID outbits (ULONG bits, INT length)
{
    STATIC INT avail = 32 ;
    STATIC ULONG ulong = 0 ;

    if (length)
    {
	if (avail == length)
	{
	    ulong |= bits ;
	    out_ul (ulong, FALSE) ;
	    avail = 32 ;
	    ulong = 0 ;
	}
	else
	{
	    if (avail > length)
	    {
		ulong |= bits << (avail - length) ;
		avail -= length ;
	    }
	    else
	    {
		length -= avail ;
		ulong |= bits >> length ;
		out_ul (ulong, FALSE) ;
		bits &= (1 << length) - 1 ;
		avail = 32 - length ;
		ulong = bits << avail ;
	    }
	}
    }
    else
    {
	if (avail == 32)
	    length = 0x80 ;
	else
	{
	    if (avail)
		length = 1 ;
	    if (avail < 25)
		length = 2 ;
	    if (avail < 17)
		length = 3 ;
	    if (avail < 9)
		length = 4 ;
	}
	out_ul (ulong, length) ;
    }
}
/* --------------------------------------------------------------------	*/
/*
 *	Input four bytes from a file.
 */
ULONG in_ul (VOID)
{
    STATIC ULONG ulong ;
    STATIC UBYTE inbuff [ULBUFSIZ] ;
    STATIC INT inptr = 0 ;

    if (!inptr)
	fread (inbuff, 1, ULBUFSIZ, pdp8file) ;
    ulong = (inbuff [inptr++] << 24) | (inbuff [inptr++] << 16)
		| (inbuff [inptr++] << 8) | inbuff [inptr++] ;
    inptr &= ULBUFSIZ - 1 ;
    return ulong ;
}
/* -------------------------------------------------------------------- */
/*
 *	Input a bit stream from a file.
 */
UWORD inbits (INT length)
{
    STATIC INT avail = 0 ;
    STATIC ULONG ulong = 0 ;
    UWORD mask = (1 << length) - 1 ;
    UWORD val ;

    if (!avail)
    {
	ulong = in_ul () ;
	avail = 32 ;
    }
    if (avail == length)
    {
	val = ulong & mask ;
	avail = 0 ;
    }
    else if (avail > length)
    {
	avail -= length ;
	val = (ulong >> avail) & mask ;
    }
    else
    {
	val = (ulong << (length - avail)) & mask ;
	length -= avail ;
	ulong = in_ul () ;
	avail = 32 - length ;
	val |= ulong >> (32 - length) ;
    }
    return val ;
}
/* -------------------------------------------------------------------- */
/*
 *	Do RK8-E write cylinder.
 */
VOID rk8writ (INT unit)
{
    if (fseek (rk8 [unit].drive, rk8 [unit].cyl * RK8BUFSIZ, SEEK_SET))
	rk8status |= BIT3 ;
    if (fwrite ((UBYTE *) rk8 [unit].buff, 1, RK8BUFSIZ,
					 rk8 [unit].drive) != RK8BUFSIZ)
	rk8status |= BIT10 ;
    rk8 [unit].dirty = FALSE ;
}
/* -------------------------------------------------------------------- */
/*
 *	Do RK8-E read cylinder.
 */
VOID rk8read (INT unit)
{
    if (rk8 [unit].cyl >= rk8 [unit].maxcyl)
    {
	ULONG *l = (ULONG *) rk8 [unit].buff ;
	INT i = RK8BUFSIZ / 16 ;
/*
 *	Clear the unit's cylinder buffer.
 */
	while (i--)
	    *l++ = 0 , *l++ = 0 , *l++ = 0 , *l++ = 0 ;
/*
 *	Write zeroed cylinders until desired cylinder reached.
 */
	i = rk8 [unit].cyl ;
	rk8 [unit].cyl = rk8 [unit].maxcyl ;
	rk8 [unit].maxcyl = i ;
	while (rk8 [unit].cyl < rk8 [unit].maxcyl)
	    rk8writ (unit) , rk8 [unit].cyl++ ;
	rk8 [unit].maxcyl++ ;
    }
    else
    {
	if (fseek (rk8 [unit].drive, rk8 [unit].cyl * RK8BUFSIZ, SEEK_SET))
	    rk8status |= BIT3 ;
	if (fread ((UBYTE *) rk8 [unit].buff, 1, RK8BUFSIZ,
					rk8 [unit].drive) != RK8BUFSIZ)
	    rk8status |= BIT4 ;
    }
}
/* -------------------------------------------------------------------- */
/*
 *	Do RK8-E I/O (read or write an rk8e sector or half-sector).
 */
INT rk8io (INT unit, INT block, UBYTE half, UBYTE rwflag, UINT memadr)
{
    UBYTE *b = rk8 [unit].buff + (block & 0x1f) * 384 ;
    UWORD *w = base + memadr ;
    INT retval = (half) ? 128 : 256 ;
    INT words = retval >> 3 ;
/*
 *	Get specified cylinder into cylinder buffer.
 */
    if (rk8 [unit].cyl != (block >> 5))
    {
	if (rk8 [unit].dirty)
	    rk8writ (unit) ;
	rk8 [unit].cyl = block >> 5 ;
	rk8read (unit) ;
    }
    if (rwflag)
    {
/*
 *	Write a sector to the rk8e disk file.  The file packing scheme:
 *		<- 8 bits->
 *		+---------+	0H => High order of pdp8 word 0
 *		|   0H    |	0L => Low order of pdp8 word 0
 *		+----+----+	1H => High order of pdp8 word 1
 *		| 0L | 1H |	1L => Low order of pdp8 word 1
 *		+----+----+
 *		|   1L    |	In other words, pdp8 words are written
 *		+----+----+	in bit order (high to low).
 */
	while (words--)
	{
	    *b++ = *w >> 4 ;
	    *b   = *w++ << 4 , *b++ |= *w >> 8 ;
	    *b++ = *w++ ;
	    *b++ = *w >> 4 ;
	    *b   = *w++ << 4 , *b++ |= *w >> 8 ;
	    *b++ = *w++ ;
	    *b++ = *w >> 4 ;
	    *b   = *w++ << 4 , *b++ |= *w >> 8 ;
	    *b++ = *w++ ;
	    *b++ = *w >> 4 ;
	    *b   = *w++ << 4 , *b++ |= *w >> 8 ;
	    *b++ = *w++ ;
	}
	if (half)
	{
	    ULONG *l = (ULONG *) b ;
	    for (words = 12 ; words ; words--)
		*l++ = 0 , *l++ = 0 , *l++ = 0 , *l++ = 0 ;
	}
	rk8 [unit].dirty = TRUE ;
    }
    else
    {
/*
 *	Read a sector from the rk8e disk file.
 */
	while (words--)
	{
	    *w = *b++ << 4 , *w++ |= *b >> 4 ;
	    *w = (*b++ & 0xf) << 8 , *w++ |= *b++ ;
	    *w = *b++ << 4 , *w++ |= *b >> 4 ;
	    *w = (*b++ & 0xf) << 8 , *w++ |= *b++ ;
	    *w = *b++ << 4 , *w++ |= *b >> 4 ;
	    *w = (*b++ & 0xf) << 8 , *w++ |= *b++ ;
	    *w = *b++ << 4 , *w++ |= *b >> 4 ;
	    *w = (*b++ & 0xf) << 8 , *w++ |= *b++ ;
	}
    }
    return retval ;
}
/* -------------------------------------------------------------------- */
/*
 *	Do RX50 write track.
 */
VOID rx5writ (INT unit)
{
}
/* -------------------------------------------------------------------- */
/*
 *	Do RX50 read track.
 */
VOID rx5read (INT unit)
{
}
/* -------------------------------------------------------------------- */
/*
 *	Do RX50 I/O (read or write an rx50 sector).
 */
INT rx5io (INT unit, INT block, UBYTE half, UBYTE rwflag, UINT memadr)
{
    return 0 ;
}
/* -------------------------------------------------------------------- */
STATIC VOID (*eaetab []) (VOID) =
{
    i7401, i7403, i7405, i7407, i7411, i7413, i7415, i7417,
    i7421, i7423, i7425, i7427, i7431, i7433, i7435, i7437,
    i7441, i7443, i7445, i7447, i7451, i7453, i7455, i7457,
    i7461, i7463, i7465, i7467, i7471, i7473, i7475, i7477,
    i7501, i7503, i7505, i7507, i7511, i7513, i7515, i7517,
    i7521, i7523, i7525, i7527, i7531, i7533, i7535, i7537,
    i7541, i7543, i7545, i7547, i7551, i7553, i7555, i7557,
    i7561, i7563, i7565, i7567, i7571, i7573, i7575, i7577,
    i7601, i7603, i7605, i7607, i7611, i7613, i7615, i7617,
    i7621, i7623, i7625, i7627, i7631, i7633, i7635, i7637,
    i7641, i7643, i7645, i7647, i7651, i7653, i7655, i7657,
    i7661, i7663, i7665, i7667, i7671, i7673, i7675, i7677,
    i7701, i7703, i7705, i7707, i7711, i7713, i7715, i7717,
    i7721, i7723, i7725, i7727, i7731, i7733, i7735, i7737,
    i7741, i7743, i7745, i7747, i7751, i7753, i7755, i7757,
    i7761, i7763, i7765, i7767, i7771, i7773, i7775, i7777
} ;
/* -------------------------------------------------------------------- */
/*
 *	Load instruction vectors for the EAE.
 */
VOID loadEAE (VOID)
{
    INT i, j ;

    for (i = 07401, j = 0 ; i < 010000 ; i += 2)
	itab [i] = eaetab [j++] ;
    for (i = 017401, j = 0 ; i < 020000 ; i += 2)
	itab [i] = eaetab [j++] ;
}
/* -------------------------------------------------------------------- */
/*
 *	Load instruction vectors for specified clock.
 */
INT loadCLOCK (VOID)
{
    INT valid = TRUE ;

    switch (clocktype)
    {
	case DK8EA :
	case DK8EC :
	    if (itab [06131] == i7000 && itab [06132] == i7000
						&& itab [06133] == i7000)
	    {
		itab [06131] = i6131 ;
		itab [06132] = i6132 ;
		itab [06133] = i6133 ;
	    }
	    else
		valid = FALSE ;
	    break ;
	case DK8EP :
	    printf ("DK8-EP not presently supported.\n") ;
	    break ;
	case PDP8A :
	case VT78 :
	case VT278 :
	    if (itab [06135] == i7000 && itab [06136] == i7000
						&& itab [06137] == i7000)
	    {
		itab [06135] = i6135 ;
		itab [06136] = i6136 ;
		itab [06137] = i6137 ;
	    }
	    else
		valid = FALSE ;
	    break ;
	default:
	    printf ("Unknown Real Time Clock type\n") ;
	    valid = FALSE ;
    }
    return valid ;
}
/* -------------------------------------------------------------------- */
STATIC VOID (*fpptab []) () =
{
    i6551, i6552, i6553, i6554, i6555, i6556, i6557, i6567
} ;
/* -------------------------------------------------------------------- */
/*
 *	Load instruction vectors for the FPP.
 */
INT loadFPP (VOID)
{
    INT i = 06551, j = 0 ;
    if (itab [06551] == i7000 && itab [06552] == i7000
		&& itab [06553] == i7000 && itab [06554] == i7000
		&& itab [06555] == i7000 && itab [06556] == i7000
		&& itab [06557] == i7000 && itab [06567] == i7000)
    {
	while (j < 7)
	    itab [i++] = fpptab [j++] ;
        itab [06567] = fpptab [j] ;
	return TRUE ;
    }
    else
	return 0 ;
}
/* -------------------------------------------------------------------- */
STATIC VOID (*rx50tab []) () =
{
    rxsel, rxlcd, rxxdr, rxstr, rxser, rxsdn, rxintr, rxinit
} ;
/* -------------------------------------------------------------------- */
STATIC BYTE *rx50txt [] =
{
    "SEL", "LCD", "XDR", "STR", "SER", "SDN", "INTR", "INIT"
} ;
/* -------------------------------------------------------------------- */
/*
 *	Load instruction and text vectors for RX50 floppy disk drive.
 */
INT loadRX50 (INT ioadr)
{
    INT j = 0 ;

    ioadr = 06000 + (ioadr << 3) ;
    if (ioadr >= 06000 && ioadr <= 06777
		&& itab [ioadr] == i7000 && itab [ioadr + 1] == i7000
		&& itab [ioadr + 2] == i7000 && itab [ioadr + 3] == i7000
		&& itab [ioadr + 4] == i7000 && itab [ioadr + 5] == i7000
		&& itab [ioadr + 6] == i7000 && itab [ioadr + 7] == i7000)
    {
	while (j < 010)			/* Load text and instr vectors	*/
	{
	    iottxt [ioadr & 0777] = rx50txt [j] ;
	    itab [ioadr++] = fpptab [j++] ;
	}
	ioadr = (ioadr - 06010) >> 3 ;
    }
    else
	ioadr = 0 ;
    return ioadr ;
}
/* -------------------------------------------------------------------- */
STATIC VOID (*auxvdutab []) () =
{
    iakcf, iaksf, iakcc, iakrs, iakie, iakrb,
    iatfl, iatsf, iatcf, iatpc, iatsk, iatls
} ;
/* -------------------------------------------------------------------- */
/*
 *	Load text and instruction vectors for the AUX VDU.
 */
INT loadAUXVDU (INT ioadr)
{
    ioadr = 06000 + (ioadr << 3) ;
    if (ioadr >= 06000 && ioadr <= 06777
	    && itab [ioadr] == i7000 && itab [ioadr + 1] == i7000
	    && itab [ioadr + 2] == i7000 && itab [ioadr + 4] == i7000
	    && itab [ioadr + 5] == i7000 && itab [ioadr + 6] == i7000
	    && itab [ioadr + 010] == i7000 && itab [ioadr + 012] == i7000
	    && itab [ioadr + 012] == i7000 && itab [ioadr + 014] == i7000
	    && itab [ioadr + 015] == i7000 && itab [ioadr + 016] == i7000)
    {					/* Load text and instr vectors	*/
	iottxt [ioadr & 0777] = "AKCF" ;
	itab [ioadr++] = iakcf ;
	iottxt [ioadr & 0777] = "AKSF" ;
	itab [ioadr++] = iaksf ;
	iottxt [ioadr & 0777] = "AKCC" ;
	itab [ioadr++] = iakcc ;
	ioadr++ ;
	iottxt [ioadr & 0777] = "AKRS" ;
	itab [ioadr++] = iakrs ;
	iottxt [ioadr & 0777] = "AKIE" ;
	itab [ioadr++] = iakie ;
	iottxt [ioadr & 0777] = "AKRB" ;
	itab [ioadr++] = iakrb ;
	ioadr++ ;
	iottxt [ioadr & 0777] = "ATFL" ;
	itab [ioadr++] = iatfl ;
	iottxt [ioadr & 0777] = "ATSF" ;
	itab [ioadr++] = iatsf ;
	iottxt [ioadr & 0777] = "ATCF" ;
	itab [ioadr++] = iatcf ;
	ioadr++ ;
	iottxt [ioadr & 0777] = "ATPC" ;
	itab [ioadr++] = iatpc ;
	iottxt [ioadr & 0777] = "ATSK" ;
	itab [ioadr++] = iatsk ;
	iottxt [ioadr & 0777] = "ATLS" ;
	itab [ioadr] = iatls ;
	ioadr = (ioadr >> 3) & 077 ;
    }
    else
	ioadr = 0 ;
    return ioadr ;
}
/* -------------------------------------------------------------------- */
/*
 *	Load special emulator instruction and text vectors.
 */
INT loadEMUL8 (INT ioadr)
{
    if (ioadr)
    {
	ioadr = 06000 + (ioadr << 3) ;
	if (ioadr >= 06000 && ioadr <= 06777
		&& itab [ioadr] == i7000 && itab [ioadr + 1] == i7000
		&& itab [ioadr + 2] == i7000 && itab [ioadr + 3] == i7000
		&& itab [ioadr + 4] == i7000 && itab [ioadr + 5] == i7000
		&& itab [ioadr + 6] == i7000 && itab [ioadr + 7] == i7000)
	{
	    itab [ioadr] = ishell ;
	    iottxt [ioadr & 0777] = "SHELL" ;
	    itab [ioadr + 6] = izexit ;
	    iottxt [(ioadr + 6) & 0777] = "ZEXIT" ;
	    itab [ioadr + 7] = iac2sr ;
	    iottxt [(ioadr + 7) & 0777] = "AC2SR" ;
	    ioadr = (ioadr >> 3) & 077 ;
	}
	else
	    ioadr = 0 ;
    }
    return ioadr ;
}
/* -------------------------------------------------------------------- */
INT validate (VOID)
{
    UWORD dateword = ((*(base + 07777) & 0600) << 5)
					| (*(base + 017666) & 07777) ;
    INT j = (dateword & 07400) >> 8 ;
    if (j && j <= 12)
    {
	j = (dateword >> 3) & 037 ;
	switch ((dateword & 07400) >> 8)
	{
	    case  2 :
		if (j > 29)			/* Feb			*/
		    dateword = 0 ;
		break ;
	    case  4 : case  6 : case  9 : case 11 :
		if (j > 30)			/* Apr, Jun, Sep, Nov	*/
		    dateword = 0 ;
	    default :				/* Jan, Mar, May, Jul	*/
		break ;				/* Aug, Oct, Dec	*/
	}
    }
    else
	dateword = 0 ;
    return dateword ;
}
/* -------------------------------------------------------------------- */
