	Page	58,132
	Title	TTY.ASM 	TTY Interface Routines
;******************************************************************************
;
;   Name:	TTY.ASM 	TTY Interface Routines
;
;   Group:	Emulator
;
;   Revision:	1.00
;
;   Date:	January 30, 1988
;
;   Author:	Randy W. Spurlock
;
;******************************************************************************
;
;  Module Functional Description:
;
;		This module contains all the code for the TTY interface
;	used by the Apple emulator.
;
;******************************************************************************
;
;  Changes:
;
;    DATE     REVISION				DESCRIPTION
;  --------   --------	-------------------------------------------------------
;   1/30/88	1.00	Original
;
;******************************************************************************
	Page
;
;  Public Declarations
;
	Public	TTY_Init		; Initialize TTY mode routine
	Public	TTY_Reset		; Reset TTY mode routine
	Public	Write_TTY		; Write TTY routine
	Public	Set_Position		; Set current position routine
	Public	Set_Row 		; Set current row routine
	Public	Set_Column		; Set current column routine
	Public	Set_Attribute		; Set current attribute routine
	Public	Set_Foreground		; Set current foreground routine
	Public	Set_Background		; Set current background routine
	Public	Set_Cursor		; Set cursor position routine
	Public	Set_Type		; Set cursor type routine
	Public	Get_Position		; Get current position routine
	Public	Get_Row 		; Get current row routine
	Public	Get_Column		; Get current column routine
	Public	Get_Attribute		; Get current attribute routine
	Public	Get_Foreground		; Get current foreground routine
	Public	Get_Background		; Get current background routine
	Public	Get_Cursor		; Get cursor position routine
	Public	Get_Type		; Get cursor type routine
	Public	Cursor_Off		; Turn cursor off routine
	Public	Cursor_On		; Turn cursor on routine
;
;  External Declarations
;
	Extrn	Scroll_Up:Near		; Scroll window up routine     (WINDOW)
	Extrn	Scroll_Down:Near	; Scroll window down routine   (WINDOW)
	Extrn	Clear_Window:Near	; Clear window routine	       (WINDOW)
	Extrn	Set_Window:Near 	; Set current window routine   (WINDOW)
	Extrn	Blink_Off:Near		; Turn blink off routine	(VIDEO)
	Extrn	Blink_On:Near		; Turn blink on routine 	(VIDEO)
	Extrn	CGA_Restore:Near	; CGA restore screen routine	  (CGA)
	Extrn	EGA_Restore:Near	; EGA restore screen routine	  (EGA)
	Extrn	EGA_Setup:Near		; EGA graphics setup routine	  (EGA)
	Extrn	System_Flag:Byte	; Apple emulator system flag byte(DATA)
	Extrn	Video_Flag:Byte 	; Video system flag byte	 (DATA)
	Extrn	Current_Window:Word	; Pointer to current window	 (DATA)
	Extrn	Screen_Window:Word	; Entire video screen window	 (DATA)
;
;  LOCAL Equates
;
TEXT_MODE	Equ	00h		; Text mode value (40 x 25)
GRAPH_MODE	Equ	04h		; Graphics mode value (320 x 200)
BACK_SHIFT	Equ	04h		; Background attribute shift value
CURSOR_MASK	Equ	07h		; Cursor type mask value
ATTRIBUTE_MASK	Equ	0Fh		; Foreground/background attribute mask
TAB_SIZE	Equ	04h		; Tab stop size value (4)
TAB_MASK	Equ	03h		; Tab count mask value
VIDEO_MEMORY	Equ	0B800h		; Start of video memory segment
;
;  Define any include files needed
;
	Include 	Macros.inc	; Include the macro definitions
	Include 	Equates.inc	; Include the equate definitions
	Include 	Strucs.inc	; Include the structure definitions
	.286c				; Include 80286 instructions
	Page
;
;  Define the emulator code segment
;
Emulate Segment Word Public 'EMULATE'   ; Emulator code segment
	Assume	cs:Emulate, ds:Nothing, es:Nothing
	Subttl	TTY_Init	Initialize TTY Mode Routine
	Page	+
;******************************************************************************
;
;	TTY_Init()
;
;		Save the required registers
;		Set text mode 0 (40x25)
;		Set current window to entire screen
;		Call routine to turn off cursor
;		Call routine to turn blink off
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
TTY_Init	Proc	Near		; Initialize TTY mode procedure
	Save	ax,si			; Save the required registers
	mov	ah,SET_MODE		; Get set mode video sub-function code
	mov	al,TEXT_MODE		; Get the 40x25 text mode value
	int	VIDEO			; Set video mode to 40x25 text mode
	lea	si,cs:[Screen_Window]	; Get pointer to screen window structure
	call	Set_Window		; Call routine to set current window
	call	Cursor_Off		; Call routine to turn cursor off
	call	Blink_Off		; Call routine to turn blink off
	Restore ax,si			; Restore the required registers
	ret				; Return to the caller
TTY_Init	Endp			; End of the TTY_Init procedure
	Subttl	TTY_Reset	Reset TTY Mode Routine
	Page	+
;******************************************************************************
;
;	TTY_Reset(RAM_Space, Video_Segment)
;
;		Save the required registers
;		Set current window to entire screen
;		Call routine to turn blink on
;		Call routine to turn cursor off
;		If this is a CGA type video
;			Call routine to restore CGA screen
;		Else this is an EGA type video
;			Call routine to restore EGA screen
;		Endif for video type
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - 65C02 RAM space
;		ES    - Video memory segment
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
TTY_Reset	Proc	Near		; Reset TTY mode procedure
	Save	si			; Save the required registers
	lea	si,cs:[Screen_Window]	; Get pointer to screen window structure
	call	Set_Window		; Call routine to set current window
	call	Blink_On		; Call routine to turn blink on
	call	Cursor_Off		; Call routine to turn cursor off
	test	cs:[System_Flag],EGA_TYPE
	jnz	Reset_EGA		; Jump if this is an EGA type video
Reset_CGA:
	call	CGA_Restore		; Call routine to restore CGA
	jmp	Short Reset_Exit	; Go return to the caller
Reset_EGA:
	call	EGA_Setup		; Call routine to setup EGA
	call	EGA_Restore		; Call routine to restore EGA
Reset_Exit:
	Restore si			; Restore the required registers
	ret				; Return to the caller
TTY_Reset	Endp			; End of the TTY_Reset procedure
	Subttl	Write_TTY	Write TTY Routine
	Page	+
;******************************************************************************
;
;	Write_TTY(Character)
;
;		Save the required registers
;		If special character
;			Call routine to handle special character
;		Else
;			Get pointer to current window
;			Calculate the current screen position
;			If position is inside current window
;				Get current screen colors (Fore./Back.)
;				Put requested character on the screen
;			Endif
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Character to write
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Write_TTY	Proc	Near		; Write TTY procedure
	Save	ax,bx,cx,dx,si,di,es	; Save the required registers
	mov	bx,cs			; Setup access to
	mov	es,bx			;		  character table
	mov	si,cs:[Current_Window]	; Get pointer to current window
	mov	cx,SPECIAL_LENGTH	; Get length of special character table
	lea	di,cs:[Special_Table]	; Get pointer to character table
	repne	scasb			; Check for a special character
	jne	Write_Character 	; Jump if NOT a special character
	lea	ax,cs:[Special_Table]	; Calculate
	sub	di,ax			;	    matching
	dec	di			;		     character value
	shl	di,1			; Convert matching number to table index
	call	cs:[di + Special_Jump]	; Call correct special character routine
	jmp	Short Write_Exit	; Go return control to the caller
Write_Character:
	mov	cx,cs:[si.Current_Row]	; Get the current row number
	add	cx,cs:[si.Upper_Left_Row]
	cmp	cx,cs:[si.Lower_Right_Row]
	ja	Write_Exit		; Jump if not in the current window
	mov	bx,cx			; Compute
	shl	bx,2			;	  row number
	add	bx,cx			;		     times five (*80)
	add	bx,VIDEO_MEMORY 	; Compute actual video memory segment
	mov	es,bx			; Setup to access video memory
	mov	di,cs:[si.Current_Col]	; Get the current column number
	add	di,cs:[si.Upper_Left_Col]
	cmp	di,cs:[si.Lower_Right_Col]
	ja	Write_Exit		; Jump if not in the current window
	shl	di,1			; Compute column number times two
	mov	ah,Byte Ptr cs:[si.Current_Back]
	shl	ah,BACK_SHIFT		; Shift background into position
	or	ah,Byte Ptr cs:[si.Current_Fore]
	stosw				; Store character into memory
	mov	bx,cs:[si.Current_Row]	; Get the current
	mov	cx,cs:[si.Current_Col]	;		  row/column values
	inc	cx			; Increment the column number
	mov	ax,cx			; Get the new column number
	add	ax,cs:[si.Upper_Left_Col]
	cmp	ax,cs:[si.Lower_Right_Col]
	jbe	Write_Update		; Jump if column number valid
	xor	cx,cx			; Reset current column number
	inc	bx			; Increment the row number
	mov	ax,bx			; Get the new row number
	add	ax,cs:[si.Upper_Left_Row]
	cmp	ax,cs:[si.Lower_Right_Row]
	jbe	Write_Exit		; Jump if row number valid
	dec	bx			; Restore the original row number
	mov	ax,1			; Get to scroll window up
	call	Scroll_Up		; Call routine to scroll the window
Write_Update:
	mov	cs:[si.Current_Row],bx	; Update the current
	mov	cs:[si.Current_Col],cx	;		     row/column values
Write_Exit:
	Restore ax,bx,cx,dx,si,di,es	; Restore the required registers
	ret				; Return to the caller
Write_TTY	Endp			; End of the Write_TTY procedure
	Subttl	Special_NULL	Special NULL Character Routine
	Page	+
;******************************************************************************
;
;	Special_NULL(Current_Window)
;
;		Return to the caller
;
;	Registers on Entry:
;
;		CS:SI - Pointer to the current window structure
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		DI    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Special_NULL	Proc	Near		; Special NULL character procedure
	ret				; Return to the caller
Special_NULL	Endp			; End of the Special_NULL procedure
	Subttl	Special_BELL	Special BELL Character Routine
	Page	+
;******************************************************************************
;
;	Special_BELL(Current_Window)
;
;
;		Return to the caller
;
;	Registers on Entry:
;
;		CS:SI - Pointer to the current window structure
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		DI    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Special_BELL	Proc	Near		; Special BELL character procedure
	Save				; Save the required registers



	Restore 			; Restore the required registers
	ret				; Return to the caller
Special_BELL	Endp			; End of the Special_BELL procedure
	Subttl	Special_BS	Special BS Character Routine
	Page	+
;******************************************************************************
;
;	Special_BS(Current_Window)
;
;		If current column number is not zero (Window edge)
;			Decrement the current column number
;			Calculate the current screen position
;			Get current screen colors (Foreground/Background)
;			Put space character on the screen
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		CS:SI - Pointer to the current window structure
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		DI    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Special_BS	Proc	Near		; Special BS character procedure
	mov	cx,cs:[si.Current_Col]	; Get the current column number
	jcxz	BS_Exit 		; Jump if currently at window edge
	dec	cs:[si.Current_Col]	; Decrement the current column value
	mov	ax,cs:[si.Current_Row]	; Get the current row number
	add	ax,cs:[si.Upper_Left_Row]
	cmp	ax,cs:[si.Lower_Right_Row]
	ja	BS_Exit 		; Jump if not in the current window
	mov	bx,ax			; Compute
	shl	bx,2			;	  row number
	add	ax,bx			;		     times five (*80)
	add	ax,VIDEO_MEMORY 	; Compute actual video memory segment
	mov	es,ax			; Setup to access video memory
	mov	di,cs:[si.Current_Col]	; Get the current column number
	add	di,cs:[si.Upper_Left_Col]
	cmp	di,cs:[si.Lower_Right_Col]
	ja	BS_Exit 		; Jump if not in the current window
	shl	di,1			; Compute column number times two
	mov	al,SPACE		; Setup to write a space character
	mov	ah,Byte Ptr cs:[si.Current_Back]
	shl	ah,BACK_SHIFT		; Shift background into position
	or	ah,Byte Ptr cs:[si.Current_Fore]
	stosw				; Store character into memory
BS_Exit:
	ret				; Return to the caller
Special_BS	Endp			; End of the Special_BS procedure
	Subttl	Special_HT	Special HT Character Routine
	Page	+
;******************************************************************************
;
;	Special_HT(Current_Window)
;
;		Calculate number of pad characters required (Spaces)
;		Calculate the current screen position
;		Get current screen colors (Foreground/Background)
;		Update the current column position
;		While more pad characters to display
;			Put pad character on the screen (Space)
;			Decrement the pad counter
;		EndWhile
;		Return to the caller
;
;	Registers on Entry:
;
;		CS:SI - Pointer to the current window structure
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		DI    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Special_HT	Proc	Near		; Special HT character procedure
	mov	ax,cs:[si.Current_Col]	; Get the current column position
	mov	bx,ax			; Save the current column position
	and	ax,TAB_MASK		; Compute
	mov	cx,TAB_SIZE		;	  the actual
	sub	cx,ax			;		     tab count value
	add	bx,cs:[si.Upper_Left_Col]
	add	bx,cx			; Compute the actual new column value
	sub	bx,cs:[si.Lower_Right_Col]
	jbe	HT_Continue		; Jump if room for the space characters
	cmp	bx,TAB_SIZE		; Check for room for partial tab
	ja	HT_Exit 		; Jump if no room for the tab
	mov	cx,TAB_SIZE		; Update the actual
	mov	cx,bx			;		    pad count value
HT_Continue:
	mov	ax,cs:[si.Current_Row]	; Get the current row number
	add	ax,cs:[si.Upper_Left_Row]
	cmp	ax,cs:[si.Lower_Right_Row]
	ja	HT_Exit 		; Jump if not in the current window
	mov	bx,ax			; Compute
	shl	bx,2			;	  row number
	add	ax,bx			;		     times five (*80)
	add	ax,VIDEO_MEMORY 	; Compute actual video memory segment
	mov	es,ax			; Setup to access video memory
	mov	di,cs:[si.Current_Col]	; Compute the actual
	shl	di,1			;		     memory address
	mov	al,SPACE		; Setup to write a space character
	mov	ah,Byte Ptr cs:[si.Current_Back]
	shl	ah,BACK_SHIFT		; Shift background into position
	or	ah,Byte Ptr cs:[si.Current_Fore]
	add	cs:[si.Current_Col],cx	; Update the current column position
	rep	stosw			; Store the horizontal pad characters
HT_Exit:
	ret				; Return to the caller
Special_HT	Endp			; End of the Special_HT procedure
	Subttl	Special_LF	Special LF Character Routine
	Page	+
;******************************************************************************
;
;	Special_LF(Current_Window)
;
;		Save the required registers
;		If not on the bottom row
;			Increment the current row value
;		Else
;			Call routine to scroll current window up
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		CS:SI - Pointer to the current window structure
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		DI    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Special_LF	Proc	Near		; Special LF character procedure
	mov	ax,cs:[si.Current_Col]	; Get the current column number
	add	ax,cs:[si.Upper_Left_Col]
	cmp	ax,cs:[si.Lower_Right_Col]
	ja	LF_Exit 		; Jump if not in the current window
	mov	ax,cs:[si.Current_Row]	; Get the current row number
	add	ax,cs:[si.Upper_Left_Row]
	cmp	ax,cs:[si.Lower_Right_Row]
	ja	LF_Exit 		; Jump if not in the current window
	jb	LF_Update		; Jump if not on the bottom row
	mov	ax,1			; Setup to scroll the current window
	call	Scroll_Up		; Call routine to scroll window up
	jmp	Short LF_Exit		; Go return control to the caller
LF_Update:
	inc	cs:[si.Current_Row]	; Increment to the next row
LF_Exit:
	ret				; Return to the caller
Special_LF	Endp			; End of the Special_LF procedure
	Subttl	Special_VT	Special VT Character Routine
	Page	+
;******************************************************************************
;
;	Special_VT(Current_Window)
;
;		Save the required registers
;		If not on the bottom row
;			Increment the current row value
;		Else
;			Call routine to scroll current window up
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		CS:SI - Pointer to the current window structure
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		DI    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Special_VT	Proc	Near		; Special VT character procedure
	mov	ax,cs:[si.Current_Col]	; Get the current column number
	add	ax,cs:[si.Upper_Left_Col]
	cmp	ax,cs:[si.Lower_Right_Col]
	ja	VT_Exit 		; Jump if not in the current window
	mov	ax,cs:[si.Current_Row]	; Get the current row number
	add	ax,cs:[si.Upper_Left_Row]
	cmp	ax,cs:[si.Lower_Right_Row]
	ja	VT_Exit 		; Jump if not in the current window
	jb	VT_Update		; Jump if not on the bottom row
	mov	ax,1			; Setup to scroll the current window
	call	Scroll_Up		; Call routine to scroll window up
	jmp	Short VT_Exit		; Go return control to the caller
VT_Update:
	inc	cs:[si.Current_Row]	; Increment to the next row
VT_Exit:
	ret				; Return to the caller
Special_VT	Endp			; End of the Special_VT procedure
	Subttl	Special_FF	Special FF Character Routine
	Page	+
;******************************************************************************
;
;	Special_FF(Current_Window)
;
;		Save the required registers
;		Call routine to clear the screen
;		Set row position to zero
;		Set column position to zero
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		CS:SI - Pointer to the current window structure
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		DI    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Special_FF	Proc	Near		; Special FF character procedure
	call	Clear_Window		; Call routine to clear the window
	xor	ax,ax			; Setup to zero current position (Home)
	mov	cs:[si.Current_Row],ax	; Zero the current
	mov	cs:[si.Current_Col],ax	;		   row/column position
	ret				; Return to the caller
Special_FF	Endp			; End of the Special_FF procedure
	Subttl	Special_CR	Special CR Character Routine
	Page	+
;******************************************************************************
;
;	Special_CR(Current_Window)
;
;		Save the required registers
;		Set current column position to zero
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		CS:SI - Pointer to the current window structure
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		DI    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Special_CR	Proc	Near		; Special CR character procedure
	xor	ax,ax			; Setup to zero column position
	mov	cs:[si.Current_Col],ax	; Zero current column position value
	ret				; Return to the caller
Special_CR	Endp			; End of the Special_CR procedure
	Subttl	Set_Position	Set Current Position Routine
	Page	+
;******************************************************************************
;
;	Set_Position(Row, Column)
;
;		Save the required registers
;		Get pointer to the current window
;		If new row position is valid
;			If new column position is valid
;				Set current row to new row
;				Set current column to new column
;			Endif
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - New row number
;		BX    - New column number
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Set_Position	Proc	Near		; Set current position procedure
	Save	dx,si			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to current window
	or	ax,ax			; Check for negative row value
	js	Position_Exit		; Jump if the row value is negative
	mov	dx,ax			; Setup to compute actual row value
	add	dx,cs:[si.Upper_Left_Row]
	cmp	dx,cs:[si.Lower_Right_Row]
	ja	Position_Exit		; Jump if new row value is invalid
	or	bx,bx			; Check for negative column value
	js	Position_Exit		; Jump if the column value is negative
	mov	dx,bx			; Setup to compute actual column value
	add	dx,cs:[si.Upper_Left_Col]
	cmp	dx,cs:[si.Lower_Right_Col]
	ja	Position_Exit		; Jump if new column value is invalid
	mov	cs:[si.Current_Row],ax	; Update the current
	mov	cs:[si.Current_Col],bx	;		     row/column values
Position_Exit:
	Restore dx,si			; Restore the required registers
	ret				; Return to the caller
Set_Position	Endp			; End of the Set_Position procedure
	Subttl	Set_Row 	Set Current Row Position Routine
	Page	+
;******************************************************************************
;
;	Set_Row(Row)
;
;		Save the required registers
;		Get pointer to the current window
;		If new row position is valid
;			Set current row to new row
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - New row number
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Set_Row 	Proc	Near		; Set current row position procedure
	Save	dx,si			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to current window
	or	ax,ax			; Check for negative row value
	js	Row_Exit		; Jump if the row value is negative
	mov	dx,ax			; Setup to compute actual row value
	add	dx,cs:[si.Upper_Left_Row]
	cmp	dx,cs:[si.Lower_Right_Row]
	ja	Row_Exit		; Jump if new row value is invalid
	mov	cs:[si.Current_Row],ax	; Update the current row value
Row_Exit:
	Restore dx,si			; Restore the required registers
	ret				; Return to the caller
Set_Row 	Endp			; End of the Set_Row procedure
	Subttl	Set_Column	Set Current Column Position Routine
	Page	+
;******************************************************************************
;
;	Set_Column(Column)
;
;		Save the required registers
;		Get pointer to the current window
;		If new column position is valid
;			Set current column to new column
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - New column number
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Set_Column	Proc	Near		; Set current column position procedure
	Save	dx,si			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to current window
	or	ax,ax			; Check for negative column value
	js	Column_Exit		; Jump if the column value is negative
	mov	dx,ax			; Setup to compute actual column value
	add	dx,cs:[si.Upper_Left_Col]
	cmp	dx,cs:[si.Lower_Right_Col]
	ja	Column_Exit		; Jump if new column value is invalid
	mov	cs:[si.Current_Col],ax	; Update the current column value
Column_Exit:
	Restore dx,si			; Restore the required registers
	ret				; Return to the caller
Set_Column	Endp			; End of the Set_Column procedure
	Subttl	Set_Attribute	Set Current Attribute Routine
	Page	+
;******************************************************************************
;
;	Set_Attribute(Foreground, Background)
;
;		Save the required registers
;		Get pointer to the current window
;		Set current foreground color to new value
;		Set current background color to new value
;		Mask off the foreground value
;		Mask off the background value
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - New foreground color
;		BX    - New background color
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Set_Attribute	Proc	Near		; Set current attribute procedure
	Save	si			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to the current window
	mov	cs:[si.Current_Fore],ax ; Save new foreground
	mov	cs:[si.Current_Back],bx ;		      background colors
	and	cs:[si.Current_Fore],ATTRIBUTE_MASK
	and	cs:[si.Current_Back],ATTRIBUTE_MASK
	Restore si			; Restore the required registers
	ret				; Return to the caller
Set_Attribute	Endp			; End of the Set_Attribute procedure
	Subttl	Set_Foreground	Set Current Foreground Routine
	Page	+
;******************************************************************************
;
;	Set_Foreground(Foreground)
;
;		Save the required registers
;		Get pointer to the current window
;		Set current foreground color to new value
;		Mask off the foreground value
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - New foreground color
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Set_Foreground	Proc	Near		; Set current foreground procedure
	Save	si			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to the current window
	mov	cs:[si.Current_Fore],ax ; Save new foreground color value
	and	cs:[si.Current_Fore],ATTRIBUTE_MASK
	Restore si			; Restore the required registers
	ret				; Return to the caller
Set_Foreground	Endp			; End of the Set_Foreground procedure
	Subttl	Set_Background	Set Current Background Routine
	Page	+
;******************************************************************************
;
;	Set_Background(Background)
;
;		Save the required registers
;		Get pointer to the current window
;		Set current background color to new value
;		Mask off the background value
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - New background color
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Set_Background	Proc	Near		; Set current background procedure
	Save	si			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to the current window
	mov	cs:[si.Current_Back],ax ; Save new background color value
	and	cs:[si.Current_Back],ATTRIBUTE_MASK
	Restore si			; Restore the required registers
	ret				; Return to the caller
Set_Background	Endp			; End of the Set_Background procedure
	Subttl	Set_Cursor	Set Cursor Position Routine
	Page	+
;******************************************************************************
;
;	Set_Cursor(Row, Column)
;
;		Save the required registers
;		Get pointer to the current window
;		If new row position is valid
;			If new column position is valid
;				Save the new cusor row and column positions
;				Set new cursor position
;			Endif
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Cursor row number
;		BX    - Cursor column number
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Set_Cursor	Proc	Near		; Set cursor position procedure
	Save	ax,bx,cx,dx,si		; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to current window
	or	ax,ax			; Check for negative row value
	js	Cursor_Exit		; Jump if the row value is negative
	mov	cx,ax			; Setup to compute actual row value
	add	cx,cs:[si.Upper_Left_Row]
	cmp	cx,cs:[si.Lower_Right_Row]
	ja	Cursor_Exit		; Jump if new row value is invalid
	or	bx,bx			; Check for negative column value
	js	Cursor_Exit		; Jump if the column value is negative
	mov	dx,bx			; Setup to compute actual column value
	add	dx,cs:[si.Upper_Left_Col]
	cmp	dx,cs:[si.Lower_Right_Col]
	ja	Cursor_Exit		; Jump if new column value is invalid
	mov	cs:[si.Cursor_Row],ax	; Save new cursor
	mov	cs:[si.Cursor_Col],bx	;		  row and column values
	mov	dh,cl			; Setup the cursor
	mov	dl,bl			;		   row and column values
	xor	bh,bh			; Setup for video page zero
	mov	ah,WRITE_CURSOR 	; Get write cursor position function
	int	VIDEO			; Set the new cursor position
Cursor_Exit:
	Restore ax,bx,cx,dx,si		; Restore the required registers
	ret				; Return to the caller
Set_Cursor	Endp			; End of the Set_Cursor procedure
	Subttl	Set_Type	Set Cursor Type Routine
	Page	+
;******************************************************************************
;
;	Set_Type(Start, End)
;
;		Save the required registers
;		Get pointer to the current window
;		Mask off the cursor start value
;		Mask off the cursor end value
;		Save the new cursor type value
;		Set the new cursor type value
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Cursor start row number (0 - 7)
;		BX    - Cursor end row number (0 - 7)
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Set_Type	Proc	Near		; Set cursor type procedure
	Save	ax,cx,si		; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to the current window
	mov	ch,al			; Get and mask
	and	ch,CURSOR_MASK		;	       cursor start value
	mov	cl,bl			; Get and mask
	and	cl,CURSOR_MASK		;	       cursor end value
	mov	cs:[si.Cursor_Type],cx	; Save the new cursor type value
	mov	ah,CURSOR_TYPE		; Get set cursor type function code
	int	VIDEO			; Set the new cursor type value
	Restore ax,cx,si		; Restore the required registers
	ret				; Return to the caller
Set_Type	Endp			; End of the Set_Type procedure
	Subttl	Get_Position	Get Current Position Routine
	Page	+
;******************************************************************************
;
;	Get_Position()
;
;		Save the required registers
;		Get pointer to the current window
;		Get the current row value
;		Get the current column value
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		AX    - Current row number
;		BX    - Current column number
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Position	Proc	Near		; Get current position procedure
	Save	si			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to the current window
	mov	ax,cs:[si.Current_Row]	; Get the current
	mov	bx,cs:[si.Current_Col]	;		  row/column values
	Restore si			; Restore the required registers
	ret				; Return to the caller
Get_Position	Endp			; End of the Get_Position procedure
	Subttl	Get_Row 	Get Current Row Position Routine
	Page	+
;******************************************************************************
;
;	Get_Row()
;
;		Save the required registers
;		Get pointer to the current window
;		Get the current row value
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		AX    - Current row number
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Row 	Proc	Near		; Get current row position procedure
	Save	si			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to the current window
	mov	ax,cs:[si.Current_Row]	; Get the current row value
	Restore si			; Restore the required registers
	ret				; Return to the caller
Get_Row 	Endp			; End of the Get_Row procedure
	Subttl	Get_Column	Get Current Column Position Routine
	Page	+
;******************************************************************************
;
;	Get_Column()
;
;		Save the required registers
;		Get pointer to the current window
;		Get the current column value
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		AX    - Current column number
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Column	Proc	Near		; Get current column position procedure
	Save	si			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to the current window
	mov	ax,cs:[si.Current_Col]	; Get the current column value
	Restore si			; Restore the required registers
	ret				; Return to the caller
Get_Column	Endp			; End of the Get_Column procedure
	Subttl	Get_Attribute	Get Current Attribute Routine
	Page	+
;******************************************************************************
;
;	Get_Attribute()
;
;		Save the required registers
;		Get pointer to the current window
;		Get the current foreground color
;		Get the current background color
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		AX    - Current foreground color
;		BX    - Current background color
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Attribute	Proc	Near		; Get current attribute procedure
	Save	si			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to the current window
	mov	ax,cs:[si.Current_Fore] ; Get foreground/
	mov	bx,cs:[si.Current_Back] ;		 background colors
	Restore si			; Restore the required registers
	ret				; Return to the caller
Get_Attribute	Endp			; End of the Get_Attribute procedure
	Subttl	Get_Foreground	Get Current Foreground Routine
	Page	+
;******************************************************************************
;
;	Get_Foreground()
;
;		Save the required registers
;		Get pointer to the current window
;		Get the current foreground color
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		AX    - Current foreground color
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Foreground	Proc	Near		; Get current foreground procedure
	Save	si			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to the current window
	mov	ax,cs:[si.Current_Fore] ; Get the current foreground color
	Restore si			; Restore the required registers
	ret				; Return to the caller
Get_Foreground	Endp			; End of the Get_Foreground procedure
	Subttl	Get_Background	Get Current Background Routine
	Page	+
;******************************************************************************
;
;	Get_Background()
;
;		Save the required registers
;		Get pointer to the current window
;		Get the current background color
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		AX    - Current background color
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Background	Proc	Near		; Get current background procedure
	Save	si			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to the current window
	mov	ax,cs:[si.Current_Back] ; Get the current background color
	Restore si			; Restore the required registers
	ret				; Return to the caller
Get_Background	Endp			; End of the Get_Background procedure
	Subttl	Get_Cursor	Get Cursor Position Routine
	Page	+
;******************************************************************************
;
;	Get_Cursor()
;
;		Save the required registers
;		Get pointer to the current window
;		Get current cursor position
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		AX    - Cursor row number
;		BX    - Cursor column number
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Cursor	Proc	Near		; Get cursor position procedure
	Save	si			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to current window
	mov	ax,cs:[si.Cursor_Row]	; Get current cursor
	mov	bx,cs:[si.Cursor_Col]	;		     row and column
	Restore si			; Restore the required registers
	ret				; Return to the caller
Get_Cursor	Endp			; End of the Get_Cursor procedure
	Subttl	Get_Type	Get Cursor Type Routine
	Page	+
;******************************************************************************
;
;	Get_Type()
;
;		Save the required registers
;		Get pointer to the current window
;		Get current cursor type
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		AX    - Cursor start row number (0 - 7)
;		BX    - Cursor end row number (0 - 7)
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Type	Proc	Near		; Get cursor type procedure
	Save	si,cx			; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to the current window
	mov	cx,cs:[si.Cursor_Type]	; Get the current cursor type value
	xor	ax,ax			; Setup to get cursor
	xor	bx,bx			;		      start and end
	mov	al,ch			; Get cursor start
	mov	bl,cl			;		   and end values
	Restore si,cx			; Restore the required registers
	ret				; Return to the caller
Get_Type	Endp			; End of the Get_Type procedure
	Subttl	Cursor_Off	Turn Cursor Off Routine
	Page	+
;******************************************************************************
;
;	Cursor_Off()
;
;		Save the required registers
;		Get pointer to current window structure
;		If current cursor type not already off
;			Save the current cursor type value
;			Get set cursor type function code
;			Get cursor off start and end values
;			Save the new cursor start and end values
;		Endif
;		Do BIOS video function to set cursor type (Cursor Off)
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Cursor_Off	Proc	Near		; Turn cursor off procedure
	Save	ax,cx,si		; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to current window
	mov	cx,cs:[si.Cursor_Type]	; Get the current cursor type value
	cmp	cx,OFF_CURSOR		; Check for cursor already off
	je	Off_Set 		; Jump if cursor is already off
	mov	cs:[si.Cursor_Save],cx	; Save the current cursor type value
	mov	cx,OFF_CURSOR		; Get the cursor off type value
	mov	cs:[si.Cursor_Type],cx	; Save new cursor start and end values
Off_Set:
	mov	ah,CURSOR_TYPE		; Get set cursor type function code
	int	VIDEO			; Turn the cursor off
	Restore ax,cx,si		; Restore the required registers
	ret				; Return to the caller
Cursor_Off	Endp			; End of the Cursor_Off procedure
	Subttl	Cursor_On	Turn Cursor On Routine
	Page	+
;******************************************************************************
;
;	Cursor_On()
;
;		Save the required registers
;		Get pointer to current window structure
;		If the current cursor type is off
;			Get set cursor type function code
;			Get the saved cursor start and end values
;			Save the new cursor start and end values
;		Endif
;		Do BIOS video function to set cursor type (Cursor On)
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Cursor_On	Proc	Near		; Turn cursor on procedure
	Save	ax,cx,si		; Save the required registers
	mov	si,cs:[Current_Window]	; Get pointer to current window
	mov	cx,cs:[si.Cursor_Type]	; Get the current cursor type value
	cmp	cx,OFF_CURSOR		; Check for cursor currently off
	jne	On_Set			; Jump if cursor is already on
	mov	cx,cs:[si.Cursor_Save]	; Get the saved cursor start and end
	mov	cs:[si.Cursor_Type],cx	; Save new cursor start and end values
On_Set:
	mov	ah,CURSOR_TYPE		; Get set cursor type sub-function code
	int	VIDEO			; Turn the cursor on
	Restore ax,cx,si		; Restore the required registers
	ret				; Return to the caller
Cursor_On	Endp			; End of the Cursor_On procedure
	Subttl			TTY Interface Data Areas
	Page	+
;******************************************************************************
;
;	Define any data areas needed by the TTY interface
;
;******************************************************************************
Special_Table	Equ	This Byte	; Special character table
		Db	NULL		; Null character
		Db	BELL		; Bell character
		Db	BS		; Backspace character
		Db	HT		; Horizontal tab character
		Db	LF		; Line feed character
		Db	VT		; Vertical tab character
		Db	FF		; Form feed character
		Db	CR		; Carriage return character
SPECIAL_LENGTH	Equ	This Byte - Special_Table
Special_Jump	Equ	This Word	; Special character jump table
		Dw	Special_NULL	; Null character routine
		Dw	Special_BELL	; Bell character routine
		Dw	Special_BS	; Backspace character routine
		Dw	Special_HT	; Horizontal tab character routine
		Dw	Special_LF	; Line feed character routine
		Dw	Special_VT	; Vertical tab character routine
		Dw	Special_FF	; Form feed character routine
		Dw	Special_CR	; Carriage return character routine
;******************************************************************************
;
;	Define the end of the Emulator Code Segment
;
;******************************************************************************
Emulate Ends
	End				; End of the TTY module
