	Page	58,132
	Title	SERIAL.ASM	Serial Port Controller
;******************************************************************************
;
;   Name:	SERIAL.ASM	Serial Port Controller
;
;   Group:	Emulator
;
;   Revision:	1.00
;
;   Date:	January 30, 1988
;
;   Author:	Randy W. Spurlock
;
;******************************************************************************
;
;  Module Functional Description:
;
;		This module contains all the code for the Apple
;	serial port controller.
;
;******************************************************************************
;
;  Changes:
;
;    DATE     REVISION				DESCRIPTION
;  --------   --------	-------------------------------------------------------
;   1/30/88	1.00	Original
;
;******************************************************************************
	Page
;
;  Public Declarations
;
	Public	Serial_ID		; Serial port ID string
	Public	Serial_Init		; Serial port initialize routine
	Public	Serial_Ctrl		; Serial port control routine
	Public	Serial_Rd		; Serial port read routine
	Public	Serial_Wrt		; Serial port write routine
	Public	Serial_Mem_Rd		; Serial port memory read routine
	Public	Serial_Mem_Wrt		; Serial port memory write routine
	Public	Serial_Exp_Rd		; Serial port expansion read routine
	Public	Serial_Exp_Wrt		; Serial port expansion write routine
	Public	Serial_Reset		; Serial port reset routine
	Public	Serial_Data		; Serial port data segment pointers
;
;  External Declarations
;
	Extrn	Get_Parameter:Near	; Get parameter routine       (SUPPORT)
	Extrn	Upper_Case:Near 	; Convert to upper case       (SUPPORT)
	Extrn	Match_Parameter:Near	; Match parameter routine     (SUPPORT)
	Extrn	Move_Block:Near 	; Move memory block routine   (SUPPORT)
	Extrn	Slot_Address:Near	; Get expansion slot address   (DEVICE)
	Extrn	Assignment:Near 	; Assignment test routine      (CONFIG)
	Extrn	Unknown:Near		; Unknown parameter routine    (CONFIG)
	Extrn	Noise:Near		; Skip noise routine	       (CONFIG)
	Extrn	Slot_Address:Near	; Get expansion slot address   (DEVICE)
	Extrn	Error:Near		; Apple emulator error routine	(APPLE)
	Extrn	Exit:Near		; Apple emulator exit routine	(APPLE)
	Extrn	RAM_Space:Word		; RAM space segment value	 (DATA)
	Extrn	Serial_ROM:Word 	; Apple serial port ROM file name(DATA)
	Extrn	Current_Owner:Byte	; Current owner slot number	 (DATA)
	Extrn	Delimit_Input:Byte	; Input delimiter table 	 (DATA)
	Extrn	Parm_Buffer:Byte	; Parameter buffer storage area  (DATA)
	Extrn	System_Flag:Byte	; Apple emulator system flag byte(DATA)
	Extrn	ERR_NO_SERIAL_FILE:Abs	; No serial ROM file error code  (DATA)
	Extrn	ERR_BAD_SERIAL_FILE:Abs ; Bad serial ROM file error code (DATA)
	Extrn	ERR_BAD_SERIAL_IMAGE:Abs; Bad serial ROM image error code(DATA)
	Extrn	ERR_NO_SERIAL_PORT:Abs	; No serial port error code	 (DATA)
	Extrn	ERR_NO_MEMORY:Abs	; Not enough memory error code	 (DATA)
	Extrn	ERR_SERIAL_PORT:Abs	; Bad serial port error code	 (DATA)
	Extrn	ERR_BAUD_RATE:Abs	; Bad baud rate error code	 (DATA)
	Extrn	ERR_STOP_BITS:Abs	; Bad stop bits error code	 (DATA)
	Extrn	ERR_DATA_LENGTH:Abs	; Bad data length error code	 (DATA)
	Extrn	ERR_PARITY:Abs		; Bad parity error code 	 (DATA)
	Extrn	ERR_PULLUP:Abs		; Bad pullup error code 	 (DATA)
;
;  LOCAL Equates
;
NEXT_ADDRESS	Equ	02h		; Next base port address offset
IO_MASK 	Equ	0Fh		; Serial card I/O control bits mask
DCD_SHIFT	Equ	01h		; DCD signal shift value (Move to carry)
DSR_SHIFT	Equ	02h		; DSR signal shift value (Move to carry)
DEF_CONTROL	Equ	08h		; Default 1200 baud, 8 data, 1 stop
DEF_COMMAND	Equ	09h		; Default None, Normal, DTR, RTS
DEF_PULLUP	Equ	0B0h		; Default DCD, DSR, CTS (Pull up)
DLAB		Equ	80h		; Divisor latch address bit
BREAK		Equ	40h		; Set break control bit
BREAK_CHECK	Equ	0Ch		; Break check mask and test value
LCR_MASK	Equ	78h		; Line control register mask value
ALT_MASK	Equ	87h		; Alternate line control register mask
MCR_MASK	Equ	0ECh		; Modem control register mask value
CTRL_SIZE	Equ	((Size Serial_Card) + 0Fh) Shr 4
PARITY_MASK	Equ	0E0h		; ACIA parity mask value
PARITY_SHIFT	Equ	005h		; ACIA parity shift value
PARITY_ALIGN	Equ	002h		; ACIA to 8250 parity alignment shift
BAUD_MASK	Equ	00Fh		; ACIA baud rate mask value
BAUD_SHIFT	Equ	000h		; ACIA baud rate shift value
LENGTH_MASK	Equ	060h		; ACIA data length mask value
LENGTH_SHIFT	Equ	005h		; ACIA data length shift value
STOP_MASK	Equ	080h		; ACIA stop bits mask value
STOP_SHIFT	Equ	007h		; ACIA stop bits shift value
CONTROL_MASK	Equ	0E0h		; ACIA control mask value
CONTROL_SHIFT	Equ	005h		; ACIA control shift value
COMMAND_MASK	Equ	01Fh		; ACIA command mask value
COMMAND_SHIFT	Equ	000h		; ACIA command shift value
SERIAL_SIZE	Equ	0800h		; Apple serial port ROM size (2k Bytes)
LOCAL_SIZE	Equ	0100h		; Local serial ROM size (Bytes)
LOCAL_OFFSET	Equ	0700h		; Local serial ROM image offset
PORT_MASK	Equ	0001h		; Port number mask value
SERIAL_OFFSET	Equ	0000h		; Serial port base address BIOS offset
EXP_ADDRESS	Equ	0C800h		; Expansion memory address
;
;  Define the 8250 Registers
;
RECEIVER_DATA	Equ	00h		; Receiver buffer data register (DLAB=0)
TRANSMIT_DATA	Equ	00h		; Transmit buffer data register (DLAB=0)
DIVISOR_LSB	Equ	00h		; Divisor latch (LSB) register (DLAB=1)
DIVISOR_MSB	Equ	01h		; Divisor latch (MSB) register (DLAB=1)
INT_ENABLE	Equ	01h		; Interrupt enable register (DLAB=0)
INTERRUPT_ID	Equ	02h		; Interrupt ID register
LINE_CONTROL	Equ	03h		; Line control register
MODEM_CONTROL	Equ	04h		; Modem control register
LINE_STATUS	Equ	05h		; Line status register
MODEM_STATUS	Equ	06h		; Modem status register
;
;  Define any include files needed
;
	Include 	Macros.inc	; Include the macro definitions
	Include 	Equates.inc	; Include the equate definitions
	Include 	Strucs.inc	; Include the structure definitions
	.286c				; Include 80286 instructions
	Page
;
;  Define the emulator code segment
;
Emulate Segment Word Public 'EMULATE'   ; Emulator code segment
	Assume	cs:Emulate, ds:Nothing, es:Nothing
	Subttl	Serial_Init	Serial Port Initialization
	Page	+
;******************************************************************************
;
;	Serial_Init(Data, Offset, Slot_Number)
;
;		Save the required registers
;		Save the current parse position
;		Call routine to get expansion slot address
;		If serial ROM space has NOT been allocated yet
;			Try to allocate memory for serial ROM image
;			If errors allocating memory
;				Set error code to not enough memory
;				Call the error routine
;				Call routine to exit the emulator
;			Endif
;			Save the serial ROM image address
;			Try to open the Serial Controller ROM data file
;			If no errors opening the Serial Controller ROM file
;				Try to read in the Serial Controller ROM image
;				If errors reading the Serial Controller ROM
;					Set error code to bad Serial ROM file
;					Call the error routine
;					Call routine to exit the emulator
;				Endif
;				Close the Serial Controller ROM file
;			Else error opening the Serial Controller ROM file
;				Set error code to bad Serial ROM file
;				Call the error routine
;				Call routine to exit the emulator
;			Endif
;		Endif for ROM space allocated
;		Copy local ROM image into slot specific address
;		Try to allocate memory for serial controller
;		If no errors allocating memory
;			Save address of serial controller data area
;			Allocate serial controller base port address
;			If no serial port exists at this address
;				Set error code to no serial port
;				Call the error routine
;				Call routine to exit the emulator
;			Endif
;			Call routine to set control register to default
;			Call routine to set command register to default
;			Set the default pull up values
;			If configuration data is present
;				Call serial port configuration routine
;			Endif
;		Else not enough memory available
;			Set error code to not enough memory
;			Call the error routine
;			Call routine to exit the emulator
;		Endif
;		Update the current parse position
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Slot number (0 - 7)
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (65C02 RAM segment)
;
;	Registers on Exit:
;
;		AX-CX - Destroyed
;		DX    - Updated to next parse offset
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Serial_Init	Proc	Near		; Serial port initialization procedure
	Save	bp,ds,es		; Save the required register values
	push	si			; Save the configuration data pointer
	mov	bp,dx			; Save the current parse offset value
	mov	di,ax			; Get the disk controller slot number
	shl	di,1			; Convert slot number to table index
	call	Slot_Address		; Call routine to get slot address
	mov	ax,ds			; Setup access to
	mov	es,ax			;		  the 65C02 RAM space
	mov	ax,cs:[Serial_Address]	; Get the serial ROM save pointer
	or	ax,ax			; Check for serial ROM already read
	jnz	Transfer_ROM		; Jump if serial ROM already read in
	mov	ah,ALLOCATE_MEMORY	; Get the allocate memory function code
	mov	bx,SERIAL_SIZE Shr 4	; Get number of paragraphs to allocate
	int	DOS			; Try to allocate the ROM save area
	jnc	Open_Serial_ROM 	; Jump if no errors allocating space
Memory_Error:
	mov	al,ERR_NO_MEMORY	; Get not enough memory error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit the emulator
Open_Serial_ROM:
	mov	cs:[Serial_Address],ax	; Save the serial ROM segment address
	mov	ax,cs			; Get the current code segment value
	mov	ds,ax			; Setup to open the disk ROM file
	mov	ah,OPEN_FILE		; Get the open file function code
	mov	al,READ_ONLY		; Get the read file access code
	lea	dx,ds:[Serial_ROM]	; Get pointer to serial ROM file name
	int	DOS			; Try to open the serial ROM file
	jnc	Read_Serial_ROM 	; Jump if no errors opening file
	mov	al,ERR_NO_SERIAL_FILE	; Get no serial ROM file error code
Serial_ROM_Error:
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Read_Serial_ROM:
	mov	ds,cs:[Serial_Address]	; Setup the serial ROM segment address
	mov	bx,ax			; Move file handle to BX register
	mov	ah,READ_FILE		; Get read file function code
	xor	dx,dx			; Setup the data buffer address
	mov	cx,SERIAL_SIZE		; Get the serial ROM image size (2k)
	int	DOS			; Try to read the serial ROM image
	mov	cx,ax			; Save ROM image size in CX register
	mov	al,ERR_BAD_SERIAL_FILE	; Get bad serial ROM file error code
	jc	Serial_ROM_Error	; Jump if error trying to read the file
	cmp	cx,SERIAL_SIZE		; Check for all of image read in
	mov	al,ERR_BAD_SERIAL_IMAGE ; Get bad serial ROM image error code
	jne	Serial_ROM_Error	; Jump if part of image missing
	mov	ah,CLOSE_FILE		; Get close file function code
	int	DOS			; Close the disk ROM file
Transfer_ROM:
	push	di			; Save the slot index value
	mov	ds,cs:[Serial_Address]	; Get serial ROM segment address
	mov	di,si			; Get the local ROM address
	mov	si,LOCAL_OFFSET 	; Get local ROM start offset
	mov	cx,LOCAL_SIZE Shr 1	; Get the local ROM size (Words)
	rep	movsw			; Put local ROM into position
	pop	di			; Restore the slot index value
Allocate_Area:
	mov	ah,ALLOCATE_MEMORY	; Get the allocate memory function code
	mov	bx,CTRL_SIZE		; Get number of paragraphs to allocate
	int	DOS			; Try to allocate disk controller space
	jc	Memory_Error		; Jump if errors allocating space
	mov	cs:[di + Serial_Data],ax; Save serial controller segment address
	mov	bx,BIOS 		; Setup access to
	mov	ds,bx			;		  the BIOS data segment
	mov	bx,cs:[Base_Address]	; Setup to get next base port address
	mov	bx,Word Ptr ds:[bx]	; Get the next serial base port address
	mov	ds,ax			; Setup serial controller address
	mov	ds:[Serial_Base],bx	; Save the serial base port address
	add	cs:[Base_Address],NEXT_ADDRESS
	or	bx,bx			; Check for serial port present
	jnz	Initialize_Values	; Jump if serial port actually present
	mov	al,ERR_NO_SERIAL_PORT	; Get no serial port error code
Serial_Port_Error:
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Initialize_Values:
	mov	ax,es			; Restore access to
	mov	ds,ax			;		    65C02 RAM space
	mov	es,cs:[di + Serial_Data]; Setup the serial card data segment
	mov	es:[Serial_Control],DEF_CONTROL
	mov	es:[Serial_Command],DEF_COMMAND
	mov	es:[Serial_Pullup],DEF_PULLUP
	pop	si			; Restore configuration data pointer
	test	cs:[System_Flag],CONFIG_DATA
	jz	Serial_Initialize	; Jump if no configuration data present
	call	Serial_Config		; Call the serial configuration routine
Serial_Initialize:
	mov	al,es:[Serial_Control]	; Get the default control value
	call	Write_Control		; Call routine to write control value
	mov	al,es:[Serial_Command]	; Get the default command value
	call	Write_Command		; Call routine to write command value
Serial_Exit:
	mov	dx,bp			; Update the current parse offset
	Restore bp,ds,es		; Restore the required register values
	ret				; Return to the caller
Serial_Init	Endp			; End of the Serial_Init procedure
	Subttl	Serial_Config	Serial Controller Configuration Routine
	Page	+
;******************************************************************************
;
;	Serial_Config(Controller_Data, Pointer, Offset)
;
;		Save the required registers
;		Setup access to the configuration data
;		Save the controller card segment
;		While not at the next section
;			Call routine to get the next parameter
;			If a null parameter was given
;				If delimiter type is comment
;					Call routine to eat this comment
;				Endif
;				If delimiter type is section
;					Exit the while loop
;				Endif
;			Else valid parameter was given
;				Call routine to convert to uppercase
;				Get pointer to parameter table
;				Call routine to check for parameter match
;				If no parameter match
;					Call routine for unknown parameter
;				Endif
;				Save the actual matching value
;				Call routine to check for assignment
;				Call routine to process the parameter
;			Endif
;		Endwhile
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		SI    - Pointer to configuration data (In 65C02 RAM Space)
;		BP    - Current parse offset
;		ES    - Controller data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;		BP    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Serial_Config	Proc	Near		; Serial configuration procedure
	Save	ds,es			; Save the required registers
	mov	dx,bp			; Setup the current parse offset
	mov	ds,cs:[RAM_Space]	; Setup access to the 65C02 RAM space
	mov	bp,es			; Save the controller data segment
	mov	ax,cs			; Setup access to
	mov	es,ax			;		  the parse tables
Serial_Loop:
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,EOF			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Input]	; Get pointer to input delimiter table
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	call	Get_Parameter		; Call routine to get next parameter
	jc	Config_Done		; Jump if no more parameters
	jnz	Check_Parameter 	; Jump if a non-null parameter
Check_Type:
	test	ah,SECTION_TYPE 	; Check for section type delimiter
	jnz	Config_Done		; Jump if next section reached
	test	ah,COMMENT_TYPE 	; Check for a comment type delimiter
	jz	Serial_Loop		; Jump if this is not a comment
	call	Noise			; Call routine to skip this noise
	jmp	Short Serial_Loop	; Go continue the parsing
Check_Parameter:
	call	Upper_Case		; Call routine to convert to uppercase
	Save	ax,si,ds		; Save the required register values
	mov	bx,es			; Setup to access
	mov	ds,bx			;		  parameter match table
	lea	si,ds:[Match_Table]	; Get pointer to start of match table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get the match table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	mov	bl,al			; Save the matching parameter number
	Restore ax,si,ds		; Restore the required register values
	jnc	Process_Parameter	; Jump if a parameter match was found
Parameter_Error:
	call	Unknown 		; Call the unknown parameter routine
Config_Done:
	jmp	Short Config_Exit	; Go return control to the caller
Process_Parameter:
	xor	bh,bh			; Convert match number
	shl	bx,1			;		       to table index
	push	bx			; Save the matching parameter value
	call	Assignment		; Call routine to check for assignment
	pop	bx			; Restore the matching parameter value
	call	cs:[Parm_Table + bx]	; Call routine to handle parameter
	jmp	Short Check_Type	; Go check the delimiter type value
Config_Exit:
	dec	dx			; Update the current
	mov	bp,dx			;		     parse offset
	Restore ds,es			; Restore the required registers
	ret				; Return to the caller
Serial_Config	Endp			; End of the Serial_Config procedure
	Subttl	Port_Option	Port Parameter Option
	Page	+
;******************************************************************************
;
;	Port_Option(Controller_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the next parameter
;		If a valid parameter is given (non-null)
;			Call routine to convert parameter to uppercase
;			Get pointer to port table
;			Call routine to check for parameter match
;			If no parameter match
;				Get invalid port error code
;				Call routine to print the error message
;				Call routine to exit the emulator
;			Else valid port parameter
;				Mask off all but desired port bits
;				Convert port number to offset
;				Get the actual serial port base address
;				If serial port actually exists
;					Save the serial base port address
;				Else no serial port
;					Get no serial port error code
;					Call routine to print error
;					Call routine to exit emulator
;				Endif
;			Endif
;		Else invalid port
;			Get invalid port error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		SI    - Pointer to configuration data (In 65C02 RAM Space)
;		BP    - Controller data segment
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Port_Option	Proc	Near		; Port parameter option procedure
	Save	si,di,es		; Save the required registers
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,EOF			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Input]	; Get pointer to input delimiter table
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	call	Get_Parameter		; Call routine to get next parameter
	ja	Check_Port		; Jump if valid parameter given
Bad_Port:
	mov	al,ERR_SERIAL_PORT	; Get bad serial port error code
Port_Error:
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Check_Port:
	call	Upper_Case		; Call routine to convert to uppercase
	Save	ax,si,ds		; Save the required register values
	mov	bx,es			; Setup to access
	mov	ds,bx			;		  parameter match table
	lea	si,ds:[Port_Table]	; Get pointer to start of port table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get the port table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	mov	bl,al			; Save the matching parameter number
	Restore ax,si,ds		; Restore the required register values
	jc	Bad_Port		; Jump if no port match found
	and	bx,PORT_MASK		; Mask off all but valid port bits
	shl	bx,1			; Convert port value to offset value
	push	ds			; Save the current data segment
	mov	cx,BIOS 		; Setup access to
	mov	ds,cx			;		  the BIOS data segment
	mov	al,ERR_NO_SERIAL_PORT	; Get no serial port error code
	mov	cx,ds:[SERIAL_OFFSET+bx]; Get the actual serial port address
	pop	ds			; Restore the current data segment
	jcxz	Port_Error		; Jump if no actual serial port
	mov	es,bp			; Setup the controller segment
	mov	es:[Serial_Base],cx	; Save the serial base port address
Port_Exit:
	Restore si,di,es		; Restore the required registers
	ret				; Return to the caller
Port_Option	Endp			; End of the Port_Option procedure
	Subttl	Baud_Option	Baud Rate Parameter Option
	Page	+
;******************************************************************************
;
;	Baud_Option(Controller_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the next parameter
;		If a valid parameter is given (non-null)
;			Call routine to convert parameter to uppercase
;			Get pointer to baud rate table
;			Call routine to check for parameter match
;			If no parameter match
;				Get invalid baud rate error code
;				Call routine to print the error message
;				Call routine to exit the emulator
;			Else valid baud rate parameter
;				Mask off the current baud rate bits (Control)
;				Shift baud rate value into position
;				Update the serial control value
;			Endif
;		Else invalid baud rate
;			Get invalid baud rate error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		SI    - Pointer to configuration data (In 65C02 RAM Space)
;		BP    - Controller data segment
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Baud_Option	Proc	Near		; Baud rate parameter option procedure
	Save	si,di,es		; Save the required registers
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,EOF			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Input]	; Get pointer to input delimiter table
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	call	Get_Parameter		; Call routine to get next parameter
	ja	Check_Baud		; Jump if valid parameter given
Baud_Error:
	mov	al,ERR_BAUD_RATE	; Get bad baud rate error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Check_Baud:
	call	Upper_Case		; Call routine to convert to uppercase
	Save	ax,si,ds		; Save the required register values
	mov	bx,es			; Setup to access
	mov	ds,bx			;		  parameter match table
	lea	si,ds:[Baud_Table]	; Get pointer to start of baud table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get the baud table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	mov	bl,al			; Save the matching parameter number
	Restore ax,si,ds		; Restore the required register values
	jc	Baud_Error		; Jump if no baud rate match found
	mov	es,bp			; Setup the controller segment
	and	es:[Serial_Control],Not BAUD_MASK
	shl	bl,BAUD_SHIFT		; Shift baud rate value into position
	or	es:[Serial_Control],bl	; Update the serial control value
Baud_Exit:
	Restore si,di,es		; Restore the required registers
	ret				; Return to the caller
Baud_Option	Endp			; End of the Baud_Option procedure
	Subttl	Stop_Option	Stop Bits Parameter Option
	Page	+
;******************************************************************************
;
;	Stop_Option(Controller_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the next parameter
;		If a valid parameter is given (non-null)
;			Call routine to convert parameter to uppercase
;			Get pointer to stop bits table
;			Call routine to check for parameter match
;			If no parameter match
;				Get invalid stop bits error code
;				Call routine to print the error message
;				Call routine to exit the emulator
;			Else valid stop bits parameter
;				Mask off the current stop bits (Control)
;				Shift stop bits value into position
;				Update the serial control value
;			Endif
;		Else invalid stop bits
;			Get invalid stop bits error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		SI    - Pointer to configuration data (In 65C02 RAM Space)
;		BP    - Controller data segment
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Stop_Option	Proc	Near		; Stop bits parameter option procedure
	Save	si,di,es		; Save the required registers
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,EOF			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Input]	; Get pointer to input delimiter table
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	call	Get_Parameter		; Call routine to get next parameter
	ja	Check_Stop		; Jump if valid parameter given
Stop_Error:
	mov	al,ERR_STOP_BITS	; Get bad stop bits error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Check_Stop:
	call	Upper_Case		; Call routine to convert to uppercase
	Save	ax,si,ds		; Save the required register values
	mov	bx,es			; Setup to access
	mov	ds,bx			;		  parameter match table
	lea	si,ds:[Stop_Table]	; Get pointer to start of stop table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get the stop table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	mov	bl,al			; Save the matching parameter number
	Restore ax,si,ds		; Restore the required register values
	jc	Stop_Error		; Jump if no stop bits match found
	mov	es,bp			; Setup the controller segment
	and	es:[Serial_Control],Not STOP_MASK
	shl	bl,STOP_SHIFT		; Shift stop bits value into position
	or	es:[Serial_Control],bl	; Update the serial control value
Stop_Exit:
	Restore si,di,es		; Restore the required registers
	ret				; Return to the caller
Stop_Option	Endp			; End of the Stop_Option procedure
	Subttl	Length_Option	Data Length Parameter Option
	Page	+
;******************************************************************************
;
;	Length_Option(Controller_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the next parameter
;		If a valid parameter is given (non-null)
;			Call routine to convert parameter to uppercase
;			Get pointer to data length table
;			Call routine to check for parameter match
;			If no parameter match
;				Get invalid data length error code
;				Call routine to print the error message
;				Call routine to exit the emulator
;			Else valid data length parameter
;				Mask off the current data length (Control)
;				Shift data length value into position
;				Update the serial control value
;			Endif
;		Else invalid data length
;			Get invalid data length error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		SI    - Pointer to configuration data (In 65C02 RAM Space)
;		BP    - Controller data segment
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Length_Option	Proc	Near		; Data length parameter option procedure
	Save	si,di,es		; Save the required registers
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,EOF			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Input]	; Get pointer to input delimiter table
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	call	Get_Parameter		; Call routine to get next parameter
	ja	Check_Length		; Jump if valid parameter given
Length_Error:
	mov	al,ERR_DATA_LENGTH	; Get bad data length error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Check_Length:
	call	Upper_Case		; Call routine to convert to uppercase
	Save	ax,si,ds		; Save the required register values
	mov	bx,es			; Setup to access
	mov	ds,bx			;		  parameter match table
	lea	si,ds:[Length_Table]	; Get pointer to start of length table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get length table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	mov	bl,al			; Save the matching parameter number
	Restore ax,si,ds		; Restore the required register values
	jc	Length_Error		; Jump if no data length match found
	mov	es,bp			; Setup the controller segment
	and	es:[Serial_Control],Not LENGTH_MASK
	shl	bl,LENGTH_SHIFT 	; Shift data length value into position
	or	es:[Serial_Control],bl	; Update the serial control value
Length_Exit:
	Restore si,di,es		; Restore the required registers
	ret				; Return to the caller
Length_Option	Endp			; End of the Length_Option procedure
	Subttl	Parity_Option	Parity Parameter Option
	Page	+
;******************************************************************************
;
;	Parity_Option(Controller_Data, Pointer, Offset)
;
;		Save the required registers
;		Call routine to get the next parameter
;		If a valid parameter is given (non-null)
;			Call routine to convert parameter to uppercase
;			Get pointer to parity table
;			Call routine to check for parameter match
;			If no parameter match
;				Get invalid parity error code
;				Call routine to print the error message
;				Call routine to exit the emulator
;			Else valid parity parameter
;				Mask off the current parity bits (Command)
;				Shift parity value into position
;				Update the serial command value
;			Endif
;		Else invalid parity
;			Get invalid parity error code
;			Call routine to print the error message
;			Call routine to exit the emulator
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		SI    - Pointer to configuration data (In 65C02 RAM Space)
;		BP    - Controller data segment
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Parity_Option	Proc	Near		; Parity parameter option procedure
	Save	si,di,es		; Save the required registers
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,EOF			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Input]	; Get pointer to input delimiter table
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	call	Get_Parameter		; Call routine to get next parameter
	ja	Check_Parity		; Jump if valid parameter given
Parity_Error:
	mov	al,ERR_PARITY		; Get bad parity error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Check_Parity:
	call	Upper_Case		; Call routine to convert to uppercase
	Save	ax,si,ds		; Save the required register values
	mov	bx,es			; Setup to access
	mov	ds,bx			;		  parameter match table
	lea	si,ds:[Parity_Table]	; Get pointer to start of parity table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get parity table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	mov	bl,al			; Save the matching parameter number
	Restore ax,si,ds		; Restore the required register values
	jc	Parity_Error		; Jump if no parity match found
	mov	es,bp			; Setup the controller segment
	and	es:[Serial_Command],Not PARITY_MASK
	shl	bl,PARITY_SHIFT 	; Shift parity value into position
	or	es:[Serial_Command],bl	; Update the serial command value
Parity_Exit:
	Restore si,di,es		; Restore the required registers
	ret				; Return to the caller
Parity_Option	Endp			; End of the Parity_Option procedure
	Subttl	Pullup_Option	Pullup Parameter Option
	Page	+
;******************************************************************************
;
;	Pullup_Option(Controller_Data, Pointer, Offset)
;
;		Save the required registers
;		While there are more parameters (Not EOD/EOL/Comment)
;		If a valid parameter is given (non-null)
;			Call routine to get the next parameter
;			Call routine to convert parameter to uppercase
;			Get pointer to pullup table
;			Call routine to check for parameter match
;			If no parameter match
;				Get invalid pullup error code
;				Call routine to print the error message
;				Call routine to exit the emulator
;			Else valid pullup parameter
;				Convert match number to table index
;				Look up the correct line mask for this signal
;				Update the serial pullup value (Logical OR)
;			Endif
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		SI    - Pointer to configuration data (In 65C02 RAM Space)
;		BP    - Controller data segment
;
;	Registers on Exit:
;
;		AH    - Last delimiter type found
;		BX-CX - Destroyed
;		DX    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Pullup_Option	Proc	Near		; Pullup parameter option procedure
	Save	si,di,es		; Save the required registers
	mov	es,bp			; Setup the controller segment
	xor	al,al			; Default to no
	mov	es:[Serial_Pullup],al	;		lines pulled up
Pullup_Loop:
	mov	ax,cs			; Setup access to
	mov	es,ax			;		  the data segment
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,EOF			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Input]	; Get pointer to input delimiter table
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	call	Get_Parameter		; Call routine to get next parameter
	ja	Check_Pullup		; Jump if valid parameter given
	jz	Pullup_Loop		; Jump if this is a null parameter
Pullup_Error:
	mov	al,ERR_PULLUP		; Get bad pullup error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Check_Pullup:
	call	Upper_Case		; Call routine to convert to uppercase
	Save	ax,si,ds		; Save the required register values
	mov	bx,es			; Setup to access
	mov	ds,bx			;		  parameter match table
	lea	si,ds:[Pullup_Table]	; Get pointer to start of pullup table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get pullup table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	mov	bl,al			; Save the matching parameter number
	Restore ax,si,ds		; Restore the required register values
	jc	Pullup_Error		; Jump if no pullup match found
	xor	bh,bh			; Convert match number to table index
	mov	al,es:[Line_Table + bx] ; Get the line mask for this signal
	mov	es,bp			; Setup the controller segment
	or	es:[Serial_Pullup],al	; Set line mask to pull up this signal
	test	ah,(END_OF_DATA + END_OF_LINE + COMMENT_TYPE)
	jz	Pullup_Loop		; Loop till end of pullup values
Pullup_Exit:
	Restore si,di,es		; Restore the required registers
	ret				; Return to the caller
Pullup_Option	Endp			; End of the Pullup_Option procedure
	Subttl	Serial_Ctrl	Serial Port Control
	Page	+
;******************************************************************************
;
;	Serial_Ctrl(RAM_Space, Slot_Number)
;
;
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Slot number (0 - 7)
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Serial_Ctrl	Proc	Near		; Serial port control procedure

	ret				; Return to the caller
Serial_Ctrl	Endp			; End of the Serial_Ctrl procedure
	Subttl	Serial_Rd	Serial Port Read
	Page	+
;******************************************************************************
;
;	Serial_Rd(Effective_Address, Slot_Index)
;
;		Save the required registers
;		Setup the serial card data segment
;		Get the serial card I/O control bits (From effective address)
;		Call the correct routine to handle the read
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		BP    - Slot index (Slot number * 2)
;		DS:DI - 65C02 Effective address
;
;	Registers on Exit:
;
;		AL    - Read value
;		BP    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Serial_Rd	Proc	Near		; Serial port read procedure
	Save	bx,es			; Save the required registers
	mov	es,cs:[bp + Serial_Data]; Setup the serial card data segment
	mov	ax,di			; Get the effective address
	and	ax,IO_MASK		; Mask off all but the I/O control bits
	mov	bx,ax			; Move control bits to BX register
	shl	bx,1			; Convert control bits to table index
	call	cs:[bx + Read_Table]	; Call correct routine to handle read
Serial_Rd_Exit:
	Restore bx,es			; Restore the required registers
	ret				; Return to the caller
Serial_Rd	Endp			; End of the Serial_Rd procedure
	Subttl	Serial_Wrt	Serial Port Write
	Page	+
;******************************************************************************
;
;	Serial_Wrt(Effective_Address, Slot_Index, Write_Value)
;
;		Save the required registers
;		Setup the serial card data segment
;		Get the serial card I/O control bits (From effective address)
;		Call the correct routine to handle the write
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Write value
;		BP    - Slot index (Slot number * 2)
;		DS:DI - 65C02 Effective address
;
;	Registers on Exit:
;
;		BP    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Serial_Wrt	Proc	Near		; Serial port write procedure
	Save	bx,es			; Save the required registers
	mov	es,cs:[bp + Serial_Data]; Setup the serial card data segment
	mov	bx,di			; Get the effective address
	and	bx,IO_MASK		; Mask off all but the I/O control bits
	shl	bx,1			; Convert control bits to table index
	call	cs:[bx + Write_Table]	; Call correct routine to handle write
Serial_Wrt_Exit:
	Restore bx,es			; Restore the required registers
	ret				; Return to the caller
Serial_Wrt	Endp			; End of the Serial_Wrt procedure
	Subttl	Serial_Mem_Rd	Serial Port Memory Read
	Page	+
;******************************************************************************
;
;	Serial_Mem_Rd(Effective_Address)
;
;		Read the memory location value (Byte)
;		Return to the caller
;
;	Registers on Entry:
;
;		DS:DI - 65C02 Effective address
;
;	Registers on Exit:
;
;		AL    - Memory value
;
;******************************************************************************
		Even			; Force procedure to even address
Serial_Mem_Rd	Proc	Near		; Serial port memory read procedure
	mov	al,ds:[di]		; Read the memory location
	ret				; Return to the caller
Serial_Mem_Rd	Endp			; End of the Serial_Mem_Rd procedure
	Subttl	Serial_Mem_Wrt	Serial Port Memory Write
	Page	+
;******************************************************************************
;
;	Serial_Mem_Wrt(Effective_Address, Memory_Value)
;
;		Return to the caller (ROM is not writable)
;
;	Registers on Entry:
;
;		AL    - Memory value
;		DS:DI - 65C02 Effective address
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Serial_Mem_Wrt	Proc	Near		; Serial port memory write procedure
	ret				; Return to the caller
Serial_Mem_Wrt	Endp			; End of the Serial_Mem_Wrt procedure
	Subttl	Serial_Exp_Rd	Serial Port Expansion Read
	Page	+
;******************************************************************************
;
;	Serial_Exp_Rd(Effective_Address)
;
;		Read the memory location value (Byte)
;		Return to the caller
;
;	Registers on Entry:
;
;		DS:DI - 65C02 Effective address
;
;	Registers on Exit:
;
;		AL    - Memory value
;
;******************************************************************************
		Even			; Force procedure to even address
Serial_Exp_Rd	Proc	Near		; Serial port expansion read procedure
	mov	al,ds:[di]		; Read the memory location
	ret				; Return to the caller
Serial_Exp_Rd	Endp			; End of the Serial_Exp_Rd procedure
	Subttl	Serial_Exp_Wrt	Serial Port Expansion Write
	Page	+
;******************************************************************************
;
;	Serial_Exp_Wrt(Effective_Address, Memory_Value)
;
;		Return to the caller (ROM is NOT writable)
;
;	Registers on Entry:
;
;		AL    - Memory value
;		DS:DI - 65C02 Effective address
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Serial_Exp_Wrt	Proc	Near		; Serial port expansion write procedure
	ret				; Return to the caller
Serial_Exp_Wrt	Endp			; End of the Serial_Exp_Wrt procedure
	Subttl	Serial_Reset	Serial Port Reset Routine
	Page	+
;******************************************************************************
;
;	Serial_Reset()
;
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Serial_Reset	Proc	Near		; Serial port reset procedure

	ret				; Return to the caller
Serial_Reset	Endp			; End of the Serial_Reset procedure
	Subttl	Serial_Update	Serial Port Expansion Update
	Page	+
;******************************************************************************
;
;	Serial_Update(Slot_Index, Serial_Segment)
;
;
;		Return to the caller
;
;	Registers on Entry:
;
;		BP    - Slot index (Slot number * 2)
;		DS:DI - 65C02 Effective address
;		ES    - Serial card segment
;
;	Registers on Exit:
;
;		BX    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Serial_Update	Proc	Near		; Serial port expansion update procedure
	Save	cx,si,di,ds,es		; Save the required register values
	mov	bx,bp			; Get the slot index value
	shr	bx,1			; Compute the actual slot number
	cmp	bl,cs:[Current_Owner]	; Check slot number against current
	je	Update_Exit		; Jump if slot number still current
	mov	cs:[Current_Owner],bl	; Update the current owner value
	mov	bx,ds			; Setup access to
	mov	es,bx			;		  the 65C02 RAM space
	mov	cx,SERIAL_SIZE Shr 1	; Get the serial ROM size (Words)
	xor	si,si			; Setup to move the serial ROM
	mov	di,EXP_ADDRESS		; Get start of the expansion memory
	mov	ds,cs:[Serial_Address]	; Get segment of the serial ROM image
	rep	movsw			; Move the serial ROM into position
Update_Exit:
	Restore cx,si,di,ds,es		; Restore the required register values
	ret				; Return to the caller
Serial_Update	Endp			; End of the Serial_Update procedure
	Subttl	Read_Data	Read Serial Port Data
	Page	+
;******************************************************************************
;
;	Read_Data(Slot_Index, Serial_Segment)
;
;		Get the serial port base address value
;		Increment to the receive data register
;		Read the serial receive register
;		Call routine to perform expansion update
;		Return to the caller
;
;	Registers on Entry:
;
;		BP    - Slot index (Slot number * 2)
;		DS:DI - 65C02 Effective address
;		ES    - Serial card segment
;
;	Registers on Exit:
;
;		AL    - Serial port data value
;		BX    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Read_Data	Proc	Near		; Read serial port data procedure
	Save	dx			; Save the required registers
	mov	dx,es:[Serial_Base]	; Get the serial port base I/O address
	add	dx,RECEIVER_DATA	; Increment to the receive data register
	in	al,dx			; Read the serial receive data register
	call	Serial_Update		; Call routine to update exp. memory
	Restore dx			; Restore the required registers
	ret				; Return to the caller
Read_Data	Endp			; End of the Read_Data procedure
	Subttl	Read_Status	Read Serial Port Status
	Page	+
;******************************************************************************
;
;	Read_Status(Slot_Index, Serial_Segment)
;
;		Get the serial port base address value
;		Increment to the line status register
;		Read the serial status register
;		Translate the serial status value
;		Increment to the modem status register
;		Read the modem status value
;		Pullup the requested signals (Use pullup value)
;		Invert signal values to correct polarity (Active low)
;		Put DCD* and DSR* values into status byte
;		Call routine to perform expansion update
;		Return to the caller
;
;	Registers on Entry:
;
;		BP    - Slot index (Slot number * 2)
;		DS:DI - 65C02 Effective address
;		ES    - Serial card segment
;
;	Registers on Exit:
;
;		AL    - Serial port status value
;		AH    - Destroyed
;		BX    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Read_Status	Proc	Near		; Read serial port status procedure
	Save	dx			; Save the required registers
	mov	dx,es:[Serial_Base]	; Get the serial port base I/O address
	add	dx,LINE_STATUS		; Increment to the line status register
	in	al,dx			; Read the serial line status register
	xor	bh,bh			; Setup to translate
	mov	bl,al			;		     the line status
	mov	ah,cs:[Status_Table+bx] ; Translate the line status value
	add	dx,(MODEM_STATUS - LINE_STATUS)
	in	al,dx			; Read the serial modem status register
	or	al,es:[Serial_Pullup]	; Pullup the requested signal values
	not	al			; Invert signals to correct polarity
	xchg	al,ah			; Exchange line and modem status values
	shl	ah,DCD_SHIFT		; Put DCD value into carry flag
	rcr	al,1			; Rotate DCD value into status byte
	shl	ah,DSR_SHIFT		; put DSR value into carry flag
	rcr	al,2			; Rotate DSR value into status byte
	call	Serial_Update		; Call routine to update exp. memory
	Restore dx			; Restore the required registers
	ret				; Return to the caller
Read_Status	Endp			; End of the Read_Status procedure
	Subttl	Read_Command	Read Serial Port Command
	Page	+
;******************************************************************************
;
;	Read_Command(Slot_Index, Serial_Segment)
;
;		Get the current command value
;		Call routine to perform expansion update
;		Return to the caller
;
;	Registers on Entry:
;
;		BP    - Slot index (Slot number * 2)
;		DS:DI - 65C02 Effective address
;		ES    - Serial card segment
;
;	Registers on Exit:
;
;		AL    - Serial port command value
;		BX    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Read_Command	Proc	Near		; Read serial port command procedure
	mov	al,es:[Serial_Command]	; Get the current serial command value
	call	Serial_Update		; Call routine to update exp. memory
	ret				; Return to the caller
Read_Command	Endp			; End of the Read_Command procedure
	Subttl	Read_Control	Read Serial Port Control
	Page	+
;******************************************************************************
;
;	Read_Control(Slot_Index, Serial_Segment)
;
;		Get the current control value
;		Call routine to perform expansion update
;		Return to the caller
;
;	Registers on Entry:
;
;		BP    - Slot index (Slot number * 2)
;		DS:DI - 65C02 Effective address
;		ES    - Serial card segment
;
;	Registers on Exit:
;
;		AL    - Serial port control value
;		BX    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Read_Control	Proc	Near		; Read serial port control procedure
	mov	al,es:[Serial_Control]	; Get the current serial control value
	call	Serial_Update		; Call routine to update exp. memory
	ret				; Return to the caller
Read_Control	Endp			; End of the Read_Control procedure
	Subttl	Write_Data	Write Serial Port Data
	Page	+
;******************************************************************************
;
;	Write_Data(Value, Slot_Index, Serial_Segment)
;
;		Get the serial port base address value
;		Increment to the transmit data register
;		Write the serial transmit register
;		Call routine to perform expansion update
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Memory value (Data)
;		BP    - Slot index (Slot number * 2)
;		DS:DI - 65C02 Effective address
;		ES    - Serial card segment
;
;	Registers on Exit:
;
;		BX    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Write_Data	Proc	Near		; Write serial port data procedure
	Save	dx			; Save the required registers
	mov	dx,es:[Serial_Base]	; Get the serial port base I/O address
	add	dx,TRANSMIT_DATA	; Increment to transmit data register
	out	dx,al			; Write serial transmit data register
	call	Serial_Update		; Call routine to update exp. memory
	Restore dx			; Restore the required registers
	ret				; Return to the caller
Write_Data	Endp			; End of the Write_Data procedure
	Subttl	Write_Status	Write Serial Port Status
	Page	+
;******************************************************************************
;
;	Write_Status(Value, Slot_Index, Serial_Segment)
;
;		Call routine to perform expansion update
;		Return to the caller (Status register NOT writable)
;
;	Registers on Entry:
;
;		AL    - Memory value (Status)
;		BP    - Slot index (Slot number * 2)
;		DS:DI - 65C02 Effective address
;		ES    - Serial card segment
;
;	Registers on Exit:
;
;		BX    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Write_Status	Proc	Near		; Write serial port status procedure
	call	Serial_Update		; Call routine to update exp. memory
	ret				; Return to the caller
Write_Status	Endp			; End of the Write_Status procedure
	Subttl	Write_Command	Write Serial Port Command
	Page	+
;******************************************************************************
;
;	Write_Command(Value, Slot_Index, Serial_Segment)
;
;		Save the new command value
;		Get the serial port base address value
;		Get the new command value
;		Mask off all but the command bits
;		Increment to the modem control register
;		Get the current modem control value
;		Mask off all the bits to change
;		Update the modem control value
;		Output the new modem control value
;		Increment to the line control register
;		Get the current line control value
;		Mask off all the bits to change
;		Get the new command value
;		Make a copy of the new command value
;		Mask off all but the parity bits
;		Shift the parity bits into position
;		Update the line control value
;		Mask off all but the break check bits
;		If a break request is set
;			Set the break control bit
;		Endif
;		Output the new line control value
;		Call routine to perform expansion update
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Memory value (Command)
;		BP    - Slot index (Slot number * 2)
;		DS:DI - 65C02 Effective address
;		ES    - Serial card segment
;
;	Registers on Exit:
;
;		BX    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Write_Command	Proc	Near		; Write serial port command procedure
	Save	ax,dx			; Save the required register values
	mov	es:[Serial_Command],al	; Save the new serial command value
	mov	dx,es:[Serial_Base]	; Get the serial port base I/O address
	mov	bl,al			; Get the new serial command value
	and	bx,COMMAND_MASK 	; Mask off all but the command bits
	add	dx,MODEM_CONTROL	; Increment to modem control register
	in	al,dx			; Get the current modem control value
	and	al,MCR_MASK		; Mask off the bits to change
	or	al,cs:[Command_Table+bx]; Update the modem control value
	out	dx,al			; Write the new modem control value
	add	dx,(LINE_CONTROL - MODEM_CONTROL)
	in	al,dx			; Get the current line control value
	and	al,ALT_MASK		; Mask off the bits to change
	mov	bl,es:[Serial_Command]	; Get the new serial command value
	mov	bh,bl			; Save a copy of the command value
	and	bl,PARITY_MASK		; Mask off all but the new parity bits
	shr	bl,PARITY_ALIGN 	; Shift parity bits into position
	or	al,bl			; Update the line control value
	and	bh,BREAK_CHECK		; Check for transmit
	cmp	bh,BREAK_CHECK		;		     break request
	jne	Output			; Jump if no break transmit request
	or	al,BREAK		; Turn on the set break control bit
Output:
	out	dx,al			; Write the new line control value
	call	Serial_Update		; Call routine to update exp. memory
	Restore ax,dx			; Restore the required register values
	ret				; Return to the caller
Write_Command	Endp			; End of the Write_Command procedure
	Subttl	Write_Control	Write Serial Port Control
	Page	+
;******************************************************************************
;
;	Write_Control(Value, Slot_Index, Serial_Segment)
;
;		Save the new control value
;		Get the serial port base address value
;		Get the new control value
;		Mask off all but the baud rate bits
;		Increment to the line control register
;		Get the current line control value
;		Set the divisor latch address bit
;		Get the new baud rate divisor value
;		If a baud rate divisor specified (Non-zero)
;			Increment to the divisor LSB register
;			Output the new divisor LSB value
;			Increment to the divisor MSB register
;			Output the new divisor MSB value
;		Endif
;		Increment to the line control register
;		Get the current line control value
;		Mask off all the bits to change (Clear DLAB bit)
;		Get the new control value
;		Mask off all but the control bits
;		Update the line control value
;		Output the new line control value
;		Call routine to perform expansion update
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Memory value (Control)
;		BP    - Slot index (Slot number * 2)
;		DS:DI - 65C02 Effective address
;		ES    - Serial card segment
;
;	Registers on Exit:
;
;		BX    - Destroyed
;		ES    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Write_Control	Proc	Near		; Write serial port control procedure
	Save	ax,dx			; Save the required register values
	mov	es:[Serial_Control],al	; Save the new serial control value
	mov	dx,es:[Serial_Base]	; Get the serial port base I/O address
	mov	bl,al			; Get the new serial control value
	and	bx,BAUD_MASK		; Mask off all but the baud rate
	shl	bx,1			; Convert baud rate to table index
	add	dx,LINE_CONTROL 	; Increment to the line control register
	in	al,dx			; Get the current line control value
	or	al,DLAB 		; Set the divisor latch address bit
	out	dx,al			; Setup to output the divisor value
	mov	ax,cs:[Rate_Table + bx] ; Get the new baud rate divisor value
	or	ax,ax			; Check for external clock specified
	jz	Set_Control		; Jump if external clock specified
	add	dx,(DIVISOR_LSB - LINE_CONTROL)
	out	dx,al			; Output LSB of divisor value
	add	dx,(DIVISOR_MSB - DIVISOR_LSB)
	xchg	al,ah			; Setup to output MSB of divisor value
	out	dx,al			; Output MSB of divisor value
Set_Control:
	add	dx,(LINE_CONTROL - DIVISOR_MSB)
	in	al,dx			; Get the current line control value
	and	al,LCR_MASK		; Mask off the bits to change
	mov	bl,es:[Serial_Control]	; Get the new serial control value
	and	bx,CONTROL_MASK 	; Mask off all but the new control bits
	shr	bx,CONTROL_SHIFT	; Convert control bits to table index
	or	al,cs:[Control_Table+bx]; Update the line control value
	out	dx,al			; Write the new line control value
	call	Serial_Update		; Call routine to update exp. memory
	Restore ax,dx			; Restore the required register values
	ret				; Return to the caller
Write_Control	Endp			; End of the Write_Control procedure
	Page
;******************************************************************************
;
;	Define the serial port data structures
;
;******************************************************************************
Serial_Data	Equ	This Word	; Define the serial port data pointers
		Slot_Data	<>	; Pointers to serial port data areas
Base_Address	Equ	This Word	; Base port address pointer (ROM BIOS)
		Dw	SERIAL_OFFSET	; Starting address for serial port 0
Serial_Address	Equ	This Word	; Define the serial ROM save pointer
		Dw	0000h		; Pointer to serial ROM segment
Read_Table	Equ	This Word	; Define the serial card read table
		Dw	Serial_Update	; Location C0x0h read routine address
		Dw	Serial_Update	; Location C0x1h read routine address
		Dw	Serial_Update	; Location C0x2h read routine address
		Dw	Serial_Update	; Location C0x3h read routine address
		Dw	Serial_Update	; Location C0x4h read routine address
		Dw	Serial_Update	; Location C0x5h read routine address
		Dw	Serial_Update	; Location C0x6h read routine address
		Dw	Serial_Update	; Location C0x7h read routine address
		Dw	Read_Data	; Location C0x8h read routine address
		Dw	Read_Status	; Location C0x9h read routine address
		Dw	Read_Command	; Location C0xAh read routine address
		Dw	Read_Control	; Location C0xBh read routine address
		Dw	Serial_Update	; Location C0xCh read routine address
		Dw	Serial_Update	; Location C0xDh read routine address
		Dw	Serial_Update	; Location C0xEh read routine address
		Dw	Serial_Update	; Location C0xFh read routine address
Write_Table	Equ	This Word	; Define the serial card write table
		Dw	Serial_Update	; Location C0x0h write routine address
		Dw	Serial_Update	; Location C0x1h write routine address
		Dw	Serial_Update	; Location C0x2h write routine address
		Dw	Serial_Update	; Location C0x3h write routine address
		Dw	Serial_Update	; Location C0x4h write routine address
		Dw	Serial_Update	; Location C0x5h write routine address
		Dw	Serial_Update	; Location C0x6h write routine address
		Dw	Serial_Update	; Location C0x7h write routine address
		Dw	Write_Data	; Location C0x8h write routine address
		Dw	Write_Status	; Location C0x9h write routine address
		Dw	Write_Command	; Location C0xAh write routine address
		Dw	Write_Control	; Location C0xBh write routine address
		Dw	Serial_Update	; Location C0xCh write routine address
		Dw	Serial_Update	; Location C0xDh write routine address
		Dw	Serial_Update	; Location C0xEh write routine address
		Dw	Serial_Update	; Location C0xFh write routine address
Rate_Table	Equ	This Word	; Baud rate divisor table
		Dw	0000h		; Divisor value for external clock
		Dw	0900h		; Divisor value for 50	  baud
		Dw	0600h		; Divisor value for 75	  baud
		Dw	0418h		; Divisor value for 110   baud
		Dw	035Ch		; Divisor value for 134   baud
		Dw	0300h		; Divisor value for 150   baud
		Dw	0180h		; Divisor value for 300   baud
		Dw	00C0h		; Divisor value for 600   baud
		Dw	0060h		; Divisor value for 1200  baud
		Dw	0040h		; Divisor value for 1800  baud
		Dw	0030h		; Divisor value for 2400  baud
		Dw	0020h		; Divisor value for 3600  baud
		Dw	0018h		; Divisor value for 4800  baud
		Dw	0010h		; Divisor value for 7200  baud
		Dw	000Ch		; Divisor value for 9600  baud
		Dw	0006h		; Divisor value for 19200 baud
Control_Table	Equ	This Byte	; ACIA control register table (LCR)
		Db	03h		; LCR value - 8 data bits/1 stop bit
		Db	02h		; LCR value - 7 data bits/1 stop bit
		Db	01h		; LCR value - 6 data bits/1 stop bit
		Db	00h		; LCR value - 5 data bits/1 stop bit
		Db	07h		; LCR value - 8 data bits/2 stop bits
		Db	06h		; LCR value - 7 data bits/2 stop bits
		Db	05h		; LCR value - 6 data bits/2 stop bits
		Db	04h		; LCR value - 5 data bits/1 1/2 stop bit
Command_Table	Equ	This Byte	; ACIA command register table (MCR)
		Db	00h		; MCR value - DTR  low, RTS  low, Normal
		Db	01h		; MCR value - DTR high, RTS  low, Normal
		Db	00h		; MCR value - DTR  low, RTS  low, Normal
		Db	01h		; MCR value - DTR high, RTS  low, Normal
		Db	02h		; MCR value - DTR  low, RTS  low, Normal
		Db	03h		; MCR value - DTR high, RTS high, Normal
		Db	02h		; MCR value - DTR  low, RTS high, Normal
		Db	03h		; MCR value - DTR high, RTS high, Normal
		Db	02h		; MCR value - DTR  low, RTS high, Normal
		Db	03h		; MCR value - DTR high, RTS high, Normal
		Db	02h		; MCR value - DTR  low, RTS high, Normal
		Db	03h		; MCR value - DTR high, RTS high, Normal
		Db	02h		; MCR value - DTR  low, RTS high, Break
		Db	03h		; MCR value - DTR high, RTS high, Break
		Db	02h		; MCR value - DTR  low, RTS high, Break
		Db	03h		; MCR value - DTR high, RTS high, Break
		Db	10h		; MCR value - DTR  low, RTS  low, Echo
		Db	11h		; MCR value - DTR high, RTS  low, Echo
		Db	10h		; MCR value - DTR  low, RTS  low, Echo
		Db	11h		; MCR value - DTR high, RTS  low, Echo
		Db	02h		; MCR value - DTR  low, RTS high, Normal
		Db	03h		; MCR value - DTR high, RTS high, Normal
		Db	02h		; MCR value - DTR  low, RTS high, Normal
		Db	03h		; MCR value - DTR high, RTS high, Normal
		Db	02h		; MCR value - DTR  low, RTS high, Normal
		Db	03h		; MCR value - DTR high, RTS high, Normal
		Db	02h		; MCR value - DTR  low, RTS high, Normal
		Db	03h		; MCR value - DTR high, RTS high, Normal
		Db	02h		; MCR value - DTR  low, RTS high, Break
		Db	03h		; MCR value - DTR high, RTS high, Break
		Db	02h		; MCR value - DTR  low, RTS high, Break
		Db	03h		; MCR value - DTR high, RTS high, Break
Status_Table	Equ	This Byte	; Graphics video bit reversal table
		Status	<00000000b>	; Translated IBM status of 00000000b
		Status	<00000001b>	; Translated IBM status of 00000001b
		Status	<00000010b>	; Translated IBM status of 00000010b
		Status	<00000011b>	; Translated IBM status of 00000011b
		Status	<00000100b>	; Translated IBM status of 00000100b
		Status	<00000101b>	; Translated IBM status of 00000101b
		Status	<00000110b>	; Translated IBM status of 00000110b
		Status	<00000111b>	; Translated IBM status of 00000111b
		Status	<00001000b>	; Translated IBM status of 00001000b
		Status	<00001001b>	; Translated IBM status of 00001001b
		Status	<00001010b>	; Translated IBM status of 00001010b
		Status	<00001011b>	; Translated IBM status of 00001011b
		Status	<00001100b>	; Translated IBM status of 00001100b
		Status	<00001101b>	; Translated IBM status of 00001101b
		Status	<00001110b>	; Translated IBM status of 00001110b
		Status	<00001111b>	; Translated IBM status of 00001111b
		Status	<00010000b>	; Translated IBM status of 00010000b
		Status	<00010001b>	; Translated IBM status of 00010001b
		Status	<00010010b>	; Translated IBM status of 00010010b
		Status	<00010011b>	; Translated IBM status of 00010011b
		Status	<00010100b>	; Translated IBM status of 00010100b
		Status	<00010101b>	; Translated IBM status of 00010101b
		Status	<00010110b>	; Translated IBM status of 00010110b
		Status	<00010111b>	; Translated IBM status of 00010111b
		Status	<00011000b>	; Translated IBM status of 00011000b
		Status	<00011001b>	; Translated IBM status of 00011001b
		Status	<00011010b>	; Translated IBM status of 00011010b
		Status	<00011011b>	; Translated IBM status of 00011011b
		Status	<00011100b>	; Translated IBM status of 00011100b
		Status	<00011101b>	; Translated IBM status of 00011101b
		Status	<00011110b>	; Translated IBM status of 00011110b
		Status	<00011111b>	; Translated IBM status of 00011111b
		Status	<00100000b>	; Translated IBM status of 00100000b
		Status	<00100001b>	; Translated IBM status of 00100001b
		Status	<00100010b>	; Translated IBM status of 00100010b
		Status	<00100011b>	; Translated IBM status of 00100011b
		Status	<00100100b>	; Translated IBM status of 00100100b
		Status	<00100101b>	; Translated IBM status of 00100101b
		Status	<00100110b>	; Translated IBM status of 00100110b
		Status	<00100111b>	; Translated IBM status of 00100111b
		Status	<00101000b>	; Translated IBM status of 00101000b
		Status	<00101001b>	; Translated IBM status of 00101001b
		Status	<00101010b>	; Translated IBM status of 00101010b
		Status	<00101011b>	; Translated IBM status of 00101011b
		Status	<00101100b>	; Translated IBM status of 00101100b
		Status	<00101101b>	; Translated IBM status of 00101101b
		Status	<00101110b>	; Translated IBM status of 00101110b
		Status	<00101111b>	; Translated IBM status of 00101111b
		Status	<00110000b>	; Translated IBM status of 00110000b
		Status	<00110001b>	; Translated IBM status of 00110001b
		Status	<00110010b>	; Translated IBM status of 00110010b
		Status	<00110011b>	; Translated IBM status of 00110011b
		Status	<00110100b>	; Translated IBM status of 00110100b
		Status	<00110101b>	; Translated IBM status of 00110101b
		Status	<00110110b>	; Translated IBM status of 00110110b
		Status	<00110111b>	; Translated IBM status of 00110111b
		Status	<00111000b>	; Translated IBM status of 00111000b
		Status	<00111001b>	; Translated IBM status of 00111001b
		Status	<00111010b>	; Translated IBM status of 00111010b
		Status	<00111011b>	; Translated IBM status of 00111011b
		Status	<00111100b>	; Translated IBM status of 00111100b
		Status	<00111101b>	; Translated IBM status of 00111101b
		Status	<00111110b>	; Translated IBM status of 00111110b
		Status	<00111111b>	; Translated IBM status of 00111111b
		Status	<01000000b>	; Translated IBM status of 01000000b
		Status	<01000001b>	; Translated IBM status of 01000001b
		Status	<01000010b>	; Translated IBM status of 01000010b
		Status	<01000011b>	; Translated IBM status of 01000011b
		Status	<01000100b>	; Translated IBM status of 01000100b
		Status	<01000101b>	; Translated IBM status of 01000101b
		Status	<01000110b>	; Translated IBM status of 01000110b
		Status	<01000111b>	; Translated IBM status of 01000111b
		Status	<01001000b>	; Translated IBM status of 01001000b
		Status	<01001001b>	; Translated IBM status of 01001001b
		Status	<01001010b>	; Translated IBM status of 01001010b
		Status	<01001011b>	; Translated IBM status of 01001011b
		Status	<01001100b>	; Translated IBM status of 01001100b
		Status	<01001101b>	; Translated IBM status of 01001101b
		Status	<01001110b>	; Translated IBM status of 01001110b
		Status	<01001111b>	; Translated IBM status of 01001111b
		Status	<01010000b>	; Translated IBM status of 01010000b
		Status	<01010001b>	; Translated IBM status of 01010001b
		Status	<01010010b>	; Translated IBM status of 01010010b
		Status	<01010011b>	; Translated IBM status of 01010011b
		Status	<01010100b>	; Translated IBM status of 01010100b
		Status	<01010101b>	; Translated IBM status of 01010101b
		Status	<01010110b>	; Translated IBM status of 01010110b
		Status	<01010111b>	; Translated IBM status of 01010111b
		Status	<01011000b>	; Translated IBM status of 01011000b
		Status	<01011001b>	; Translated IBM status of 01011001b
		Status	<01011010b>	; Translated IBM status of 01011010b
		Status	<01011011b>	; Translated IBM status of 01011011b
		Status	<01011100b>	; Translated IBM status of 01011100b
		Status	<01011101b>	; Translated IBM status of 01011101b
		Status	<01011110b>	; Translated IBM status of 01011110b
		Status	<01011111b>	; Translated IBM status of 01011111b
		Status	<01100000b>	; Translated IBM status of 01100000b
		Status	<01100001b>	; Translated IBM status of 01100001b
		Status	<01100010b>	; Translated IBM status of 01100010b
		Status	<01100011b>	; Translated IBM status of 01100011b
		Status	<01100100b>	; Translated IBM status of 01100100b
		Status	<01100101b>	; Translated IBM status of 01100101b
		Status	<01100110b>	; Translated IBM status of 01100110b
		Status	<01100111b>	; Translated IBM status of 01100111b
		Status	<01101000b>	; Translated IBM status of 01101000b
		Status	<01101001b>	; Translated IBM status of 01101001b
		Status	<01101010b>	; Translated IBM status of 01101010b
		Status	<01101011b>	; Translated IBM status of 01101011b
		Status	<01101100b>	; Translated IBM status of 01101100b
		Status	<01101101b>	; Translated IBM status of 01101101b
		Status	<01101110b>	; Translated IBM status of 01101110b
		Status	<01101111b>	; Translated IBM status of 01101111b
		Status	<01110000b>	; Translated IBM status of 01110000b
		Status	<01110001b>	; Translated IBM status of 01110001b
		Status	<01110010b>	; Translated IBM status of 01110010b
		Status	<01110011b>	; Translated IBM status of 01110011b
		Status	<01110100b>	; Translated IBM status of 01110100b
		Status	<01110101b>	; Translated IBM status of 01110101b
		Status	<01110110b>	; Translated IBM status of 01110110b
		Status	<01110111b>	; Translated IBM status of 01110111b
		Status	<01111000b>	; Translated IBM status of 01111000b
		Status	<01111001b>	; Translated IBM status of 01111001b
		Status	<01111010b>	; Translated IBM status of 01111010b
		Status	<01111011b>	; Translated IBM status of 01111011b
		Status	<01111100b>	; Translated IBM status of 01111100b
		Status	<01111101b>	; Translated IBM status of 01111101b
		Status	<01111110b>	; Translated IBM status of 01111110b
		Status	<01111111b>	; Translated IBM status of 01111111b
		Status	<10000000b>	; Translated IBM status of 10000000b
		Status	<10000001b>	; Translated IBM status of 10000001b
		Status	<10000010b>	; Translated IBM status of 10000010b
		Status	<10000011b>	; Translated IBM status of 10000011b
		Status	<10000100b>	; Translated IBM status of 10000100b
		Status	<10000101b>	; Translated IBM status of 10000101b
		Status	<10000110b>	; Translated IBM status of 10000110b
		Status	<10000111b>	; Translated IBM status of 10000111b
		Status	<10001000b>	; Translated IBM status of 10001000b
		Status	<10001001b>	; Translated IBM status of 10001001b
		Status	<10001010b>	; Translated IBM status of 10001010b
		Status	<10001011b>	; Translated IBM status of 10001011b
		Status	<10001100b>	; Translated IBM status of 10001100b
		Status	<10001101b>	; Translated IBM status of 10001101b
		Status	<10001110b>	; Translated IBM status of 10001110b
		Status	<10001111b>	; Translated IBM status of 10001111b
		Status	<10010000b>	; Translated IBM status of 10010000b
		Status	<10010001b>	; Translated IBM status of 10010001b
		Status	<10010010b>	; Translated IBM status of 10010010b
		Status	<10010011b>	; Translated IBM status of 10010011b
		Status	<10010100b>	; Translated IBM status of 10010100b
		Status	<10010101b>	; Translated IBM status of 10010101b
		Status	<10010110b>	; Translated IBM status of 10010110b
		Status	<10010111b>	; Translated IBM status of 10010111b
		Status	<10011000b>	; Translated IBM status of 10011000b
		Status	<10011001b>	; Translated IBM status of 10011001b
		Status	<10011010b>	; Translated IBM status of 10011010b
		Status	<10011011b>	; Translated IBM status of 10011011b
		Status	<10011100b>	; Translated IBM status of 10011100b
		Status	<10011101b>	; Translated IBM status of 10011101b
		Status	<10011110b>	; Translated IBM status of 10011110b
		Status	<10011111b>	; Translated IBM status of 10011111b
		Status	<10100000b>	; Translated IBM status of 10100000b
		Status	<10100001b>	; Translated IBM status of 10100001b
		Status	<10100010b>	; Translated IBM status of 10100010b
		Status	<10100011b>	; Translated IBM status of 10100011b
		Status	<10100100b>	; Translated IBM status of 10100100b
		Status	<10100101b>	; Translated IBM status of 10100101b
		Status	<10100110b>	; Translated IBM status of 10100110b
		Status	<10100111b>	; Translated IBM status of 10100111b
		Status	<10101000b>	; Translated IBM status of 10101000b
		Status	<10101001b>	; Translated IBM status of 10101001b
		Status	<10101010b>	; Translated IBM status of 10101010b
		Status	<10101011b>	; Translated IBM status of 10101011b
		Status	<10101100b>	; Translated IBM status of 10101100b
		Status	<10101101b>	; Translated IBM status of 10101101b
		Status	<10101110b>	; Translated IBM status of 10101110b
		Status	<10101111b>	; Translated IBM status of 10101111b
		Status	<10110000b>	; Translated IBM status of 10110000b
		Status	<10110001b>	; Translated IBM status of 10110001b
		Status	<10110010b>	; Translated IBM status of 10110010b
		Status	<10110011b>	; Translated IBM status of 10110011b
		Status	<10110100b>	; Translated IBM status of 10110100b
		Status	<10110101b>	; Translated IBM status of 10110101b
		Status	<10110110b>	; Translated IBM status of 10110110b
		Status	<10110111b>	; Translated IBM status of 10110111b
		Status	<10111000b>	; Translated IBM status of 10111000b
		Status	<10111001b>	; Translated IBM status of 10111001b
		Status	<10111010b>	; Translated IBM status of 10111010b
		Status	<10111011b>	; Translated IBM status of 10111011b
		Status	<10111100b>	; Translated IBM status of 10111100b
		Status	<10111101b>	; Translated IBM status of 10111101b
		Status	<10111110b>	; Translated IBM status of 10111110b
		Status	<10111111b>	; Translated IBM status of 10111111b
		Status	<11000000b>	; Translated IBM status of 11000000b
		Status	<11000001b>	; Translated IBM status of 11000001b
		Status	<11000010b>	; Translated IBM status of 11000010b
		Status	<11000011b>	; Translated IBM status of 11000011b
		Status	<11000100b>	; Translated IBM status of 11000100b
		Status	<11000101b>	; Translated IBM status of 11000101b
		Status	<11000110b>	; Translated IBM status of 11000110b
		Status	<11000111b>	; Translated IBM status of 11000111b
		Status	<11001000b>	; Translated IBM status of 11001000b
		Status	<11001001b>	; Translated IBM status of 11001001b
		Status	<11001010b>	; Translated IBM status of 11001010b
		Status	<11001011b>	; Translated IBM status of 11001011b
		Status	<11001100b>	; Translated IBM status of 11001100b
		Status	<11001101b>	; Translated IBM status of 11001101b
		Status	<11001110b>	; Translated IBM status of 11001110b
		Status	<11001111b>	; Translated IBM status of 11001111b
		Status	<11010000b>	; Translated IBM status of 11010000b
		Status	<11010001b>	; Translated IBM status of 11010001b
		Status	<11010010b>	; Translated IBM status of 11010010b
		Status	<11010011b>	; Translated IBM status of 11010011b
		Status	<11010100b>	; Translated IBM status of 11010100b
		Status	<11010101b>	; Translated IBM status of 11010101b
		Status	<11010110b>	; Translated IBM status of 11010110b
		Status	<11010111b>	; Translated IBM status of 11010111b
		Status	<11011000b>	; Translated IBM status of 11011000b
		Status	<11011001b>	; Translated IBM status of 11011001b
		Status	<11011010b>	; Translated IBM status of 11011010b
		Status	<11011011b>	; Translated IBM status of 11011011b
		Status	<11011100b>	; Translated IBM status of 11011100b
		Status	<11011101b>	; Translated IBM status of 11011101b
		Status	<11011110b>	; Translated IBM status of 11011110b
		Status	<11011111b>	; Translated IBM status of 11011111b
		Status	<11100000b>	; Translated IBM status of 11100000b
		Status	<11100001b>	; Translated IBM status of 11100001b
		Status	<11100010b>	; Translated IBM status of 11100010b
		Status	<11100011b>	; Translated IBM status of 11100011b
		Status	<11100100b>	; Translated IBM status of 11100100b
		Status	<11100101b>	; Translated IBM status of 11100101b
		Status	<11100110b>	; Translated IBM status of 11100110b
		Status	<11100111b>	; Translated IBM status of 11100111b
		Status	<11101000b>	; Translated IBM status of 11101000b
		Status	<11101001b>	; Translated IBM status of 11101001b
		Status	<11101010b>	; Translated IBM status of 11101010b
		Status	<11101011b>	; Translated IBM status of 11101011b
		Status	<11101100b>	; Translated IBM status of 11101100b
		Status	<11101101b>	; Translated IBM status of 11101101b
		Status	<11101110b>	; Translated IBM status of 11101110b
		Status	<11101111b>	; Translated IBM status of 11101111b
		Status	<11110000b>	; Translated IBM status of 11110000b
		Status	<11110001b>	; Translated IBM status of 11110001b
		Status	<11110010b>	; Translated IBM status of 11110010b
		Status	<11110011b>	; Translated IBM status of 11110011b
		Status	<11110100b>	; Translated IBM status of 11110100b
		Status	<11110101b>	; Translated IBM status of 11110101b
		Status	<11110110b>	; Translated IBM status of 11110110b
		Status	<11110111b>	; Translated IBM status of 11110111b
		Status	<11111000b>	; Translated IBM status of 11111000b
		Status	<11111001b>	; Translated IBM status of 11111001b
		Status	<11111010b>	; Translated IBM status of 11111010b
		Status	<11111011b>	; Translated IBM status of 11111011b
		Status	<11111100b>	; Translated IBM status of 11111100b
		Status	<11111101b>	; Translated IBM status of 11111101b
		Status	<11111110b>	; Translated IBM status of 11111110b
		Status	<11111111b>	; Translated IBM status of 11111111b
Serial_ID	Equ	This Byte	; Serial port ID string
		Db	"Serial Port",0
Match_Table	Equ	This Byte	; Start of serial parameter match table
		Db	7		; Match table entry size
		Db	2,"PORT  "      ; Serial port selection parameter
		Db	1,"BAUD  "      ; Baud rate selection parameter
		Db	1,"STOP  "      ; Stop bits selection parameter
		Db	1,"LENGTH"      ; Data length selection parameter
		Db	2,"PARITY"      ; Parity selection parameter
		Db	2,"PULLUP"      ; Pullup selection parameter
		Db	0		; End of the serial match table
Parm_Table	Equ	This Word	; Start of parameter jump table
		Dw	Offset Port_Option
		Dw	Offset Baud_Option
		Dw	Offset Stop_Option
		Dw	Offset Length_Option
		Dw	Offset Parity_Option
		Dw	Offset Pullup_Option
Port_Table	Equ	This Byte	; Start of port parameter match table
		Db	5		; Match table entry size
		Db	4,"COM1"        ; Serial port one parameter
		Db	4,"COM2"        ; Serial port two parameter
		Db	1,"1   "        ; Alternate serial port one parameter
		Db	1,"2   "        ; Alternate serial port two parameter
		Db	1,"A   "        ; Alternate serial port one parameter
		Db	1,"B   "        ; Alternate serial port two parameter
		Db	0		; End of the port match table
Baud_Table	Equ	This Byte	; Start of baud parameter match table
		Db	6		; Match table entry size
		Db	3,"EXT  "       ; External baud rate parameter
		Db	2,"50   "       ; Baud rate 50 parameter
		Db	2,"75   "       ; Baud rate 75 parameter
		Db	3,"110  "       ; Baud rate 110 parameter
		Db	3,"134  "       ; Baud rate 134 parameter
		Db	3,"150  "       ; Baud rate 150 parameter
		Db	3,"300  "       ; Baud rate 300 parameter
		Db	3,"600  "       ; Baud rate 600 parameter
		Db	4,"1200 "       ; Baud rate 1200 parameter
		Db	4,"1800 "       ; Baud rate 1800 parameter
		Db	4,"2400 "       ; Baud rate 2400 parameter
		Db	4,"3600 "       ; Baud rate 3600 parameter
		Db	4,"4800 "       ; Baud rate 4800 parameter
		Db	4,"7200 "       ; Baud rate 7200 parameter
		Db	4,"9600 "       ; Baud rate 9600 parameter
		Db	5,"19200"       ; Baud rate 19200 parameter
		Db	0		; End of the baud match table
Stop_Table	Equ	This Byte	; Start of stop parameter match table
		Db	2		; Match table entry size
		Db	1,"1"           ; One stop bit parameter
		Db	1,"2"           ; Two stop bits parameter
		Db	0		; End of the stop bits match table
Length_Table	Equ	This Byte	; Start of length parameter match table
		Db	2		; Match table entry size
		Db	1,"5"           ; Data size of 5 bits parameter
		Db	1,"6"           ; Data size of 6 bits parameter
		Db	1,"7"           ; Data size of 7 bits parameter
		Db	1,"8"           ; Data size of 8 bits parameter
		Db	0		; End of the length match table
Parity_Table	Equ	This Byte	; Start of parity parameter match table
		Db	6		; Match table entry size
		Db	1,"NONE "       ; No parity parameter (None odd)
		Db	1,"ODD  "       ; Odd parity parameter
		Db	1,"NONE "       ; No parity parameter (None even)
		Db	1,"EVEN "       ; Even parity parameter
		Db	1,"NONE "       ; No parity parameter (None mark)
		Db	1,"MARK "       ; Marked (1) parity parameter
		Db	1,"NONE "       ; No parity parameter (None space)
		Db	1,"SPACE"       ; Spacing (0) parity parameter
		Db	0		; End of the parity match table
Pullup_Table	Equ	This Byte	; Start of pullup parameter match table
		Db	4		; Match table entry size
		Db	3,"DCD"         ; Delta carrier detect (DCD) pullup
		Db	3,"DSR"         ; Data set ready (DSR) pullup parameter
		Db	3,"CTS"         ; Clear to send (CTS) pullup parameter
		Db	0		; End of the pullup match table
Line_Table	Equ	This Byte	; Start of line signal mask table
		Db	80h		; DCD line signal mask value
		Db	20h		; DSR line signal mask value
		Db	10h		; CTS line signal mask value
;******************************************************************************
;
;	Define the end of the Emulator Code Segment
;
;******************************************************************************
Emulate Ends
	End				; End of the Serial module
